#' Plot jeek result specified by user input
#'
#' This function can plot and return multiple sparse graphs distinguished by edge colors
#' from the result generated by jeek
#' @author Beilun Wang, Zhaoyang Wang (Author), Zhaoyang Wang (maintainer) \email{zw4dn@virginia.edu}
#' @param x output generated from jeek function (jeek class)
#' @param type type of graph. There are four options:
#' * "task" (graph for each task (including shared part) specified further by subID (task number))
#' * "share" (shared graph for all tasks)
#' * "taskspecific" (graph for each task specific graph (excluding shared part)
#' specified further by subID (task number) )
#' * "neighbour" (zoom into nodes in the graph specified further by neighbouroptoin, subID (task number)
#' and index (node id))
#' @param neighbouroption determines what type of graph to zoom into when parameter **"type"** is **"neighbour"**. There are two options:
#' * "task" (zoom into graph for each task (including shared part))
#' * "taskspecific" (zoom into graph for each task specific (excluding shared part))
#' @param subID selects which task to display. There are four options:
#' * 0 (only allowed when
#' **"type"** is **"task"** or **"type"** is **"neighbour"** and **"neighbouroption"** is **"task"**) (selects share graph)
#' * positive task number (selects that particular task)
#' * a vector of task number (selects multiple tasks)
#' * NULL (selects all tasks (all graphs))
#' @param index determines which node(s) to zoom into when parameter **"type"** is **"neighbour"**.
#' This parameter could either be an integer or vector of integers representing node ids
#' (zoom into one node or multiple nodes)
#' @param hastitle determines whether the graph title is displayed or not (TRUE to display / FALSE to hide)
#' @param haslegend determines whether the graph legend is displayed or not (TRUE to display / FALSE to hide)
#' @param ... extra parameters passed to plot.igraph() and legend() (only the argument "legend" for legend() is available).
#' Please see \code{\link{plot.igraph}} and \code{\link{legend}}
#' @return a plot of graph / subgraph from jeek result specified by user input
#' @details when only the jeek result is provided, the function will plot all graphs with default numeric labels
#' User can specify multiple subID and multiple index to zoom in multiple nodes on multiple graphs
#' Each graph will include a descriptive title and legend to indicate correspondence between edge color and task.
#' @examples
#' library(JointNets)
#' data(exampleData)
#' result = jeek(X = exampleData, 0.3, covType = "cov", parallel = FALSE)
#' plot(result)
#' @method plot jeek
#' @export
#' @export plot.jeek
#' @import igraph
plot.jeek <-
  function(x,
           type = "task",
           neighbouroption = "task",
           subID = NULL,
           index = NULL,
           hastitle = TRUE,
           haslegend = TRUE,
           ...)
  {
    jointplot(x,
              type,
              neighbouroption,
              subID,
              index,
              hastitle,
              haslegend,
              ...)
  }



#' Plot simule result specified by user input
#'
#' This function can plot and return multiple sparse graphs distinguished by edge colors
#' from the result generated by simule
#' @author Beilun Wang, Zhaoyang Wang (Author), Zhaoyang Wang (maintainer) \email{zw4dn@virginia.edu}
#' @param x output generated from simule function (simule class)
#' @param type type of graph. There are four options:
#' * "task" (graph for each task (including shared part) specified further by subID (task number))
#' * "share" (shared graph for all tasks)
#' * "taskspecific" (graph for each task specific graph (excluding shared part)
#' specified further by subID (task number) )
#' * "neighbour" (zoom into nodes in the graph specified further by neighbouroptoin, subID (task number)
#' and index (node id))
#' @param neighbouroption determines what type of graph to zoom into when parameter **"type"** is **"neighbour"**. There are two options:
#' * "task" (zoom into graph for each task (including shared part))
#' * "taskspecific" (zoom into graph for each task specific (excluding shared part))
#' @param subID selects which task to display. There are four options:
#' * 0 (only allowed when
#' **"type"** is **"task"** or **"type"** is **"neighbour"** and **"neighbouroption"** is **"task"**) (selects share graph)
#' * positive task number (selects that particular task)
#' * a vector of task number (selects multiple tasks)
#' * NULL (selects all tasks (all graphs))
#' @param index determines which node(s) to zoom into when parameter **"type"** is **"neighbour"**.
#' This parameter could either be an integer or vector of integers representing node ids
#' (zoom into one node or multiple nodes)
#' @param hastitle determines whether the graph title is displayed or not (TRUE to display / FALSE to hide)
#' @param haslegend determines whether the graph legend is displayed or not (TRUE to display / FALSE to hide)
#' @param ... extra parameters passed to plot.igraph() and legend() (only the argument "legend" for legend() is available).
#' Please see \code{\link{plot.igraph}} and \code{\link{legend}}
#' @return a plot of graph / subgraph from simule result specified by user input
#' @details when only the simule result is provided, the function will plot all graphs with default numeric labels.
#' User can specify multiple subID and multiple index to zoom in multiple nodes on multiple graphs.
#' Each graph will include a descriptive title and legend to indicate correspondence between edge color and task.
#' @examples
#' library(JointNets)
#' data(exampleData)
#' result = simule(X = exampleData , lambda = 0.1, epsilon = 0.45, covType = "cov", FALSE)
#' plot(result)
#' @method plot simule
#' @export
#' @export plot.simule
#' @import igraph
plot.simule <-
  function(x,
           type = "task",
           neighbouroption = "task",
           subID = NULL,
           index = NULL,
           hastitle = TRUE,
           haslegend = TRUE,
           ...)
  {
    jointplot(x,
              type,
              neighbouroption,
              subID,
              index,
              hastitle,
              haslegend,
              ...)
  }

#' Plot simulatedgraph result (generated from function simulation()) (class simulation)
#'
#' This function can plot and return multiple sparse graphs distinguished by edge colors
#' from the result generated by simulation()
#' @author Beilun Wang, Zhaoyang Wang (Author), Zhaoyang Wang (maintainer) \email{zw4dn@virginia.edu}
#' @param x output generated from simule function (simule class)
#' @param type type of graph. There are four options:
#' * "task" (graph for each task (including shared part) specified further by subID (task number))
#' * "share" (shared graph for all tasks)
#' * "taskspecific" (graph for each task specific graph (excluding shared part)
#' specified further by subID (task number) )
#' * "neighbour" (zoom into nodes in the graph specified further by neighbouroptoin, subID (task number)
#' and index (node id))
#' @param neighbouroption determines what type of graph to zoom into when parameter **"type"** is **"neighbour"**. There are two options:
#' * "task" (zoom into graph for each task (including shared part))
#' * "taskspecific" (zoom into graph for each task specific (excluding shared part))
#' @param subID selects which task to display. There are four options:
#' * 0 (only allowed when
#' **"type"** is **"task"** or **"type"** is **"neighbour"** and **"neighbouroption"** is **"task"**) (selects share graph)
#' * positive task number (selects that particular task)
#' * a vector of task number (selects multiple tasks)
#' * NULL (selects all tasks (all graphs))
#' @param index determines which node(s) to zoom into when parameter **"type"** is **"neighbour"**.
#' This parameter could either be an integer or vector of integers representing node ids
#' (zoom into one node or multiple nodes)
#' @param hastitle determines whether the graph title is displayed or not (TRUE to display / FALSE to hide)
#' @param haslegend determines whether the graph legend is displayed or not (TRUE to display / FALSE to hide)
#' @param ... extra parameters passed to plot.igraph() and legend() (only the argument "legend" for legend() is available).
#' Please see \code{\link{plot.igraph}} and \code{\link{legend}}
#' @return a plot of graph / subgraph from simulatedgraph result specified by user input
#' @details when only the simulatedgraph is provided, the function will plot all graphs with default numeric labels.
#' User can specify multiple subID and multiple index to zoom in multiple nodes on multiple graphs.
#' Each graph will include a descriptive title and legend to indicate correspondence between edge color and task.
#' @examples
#' library(JointNets)
#' data(exampleData)
#' result = simulation(n = c(100,100,100))$simulatedgraphs
#' plot(result)
#' @method plot simulation
#' @export
#' @export plot.simulation
#' @import igraph
plot.simulation <-
  function(x,
           type = "task",
           neighbouroption = "task",
           subID = NULL,
           index = NULL,
           hastitle = TRUE,
           haslegend = TRUE,
           ...)
  {
    jointplot(x,
              type,
              neighbouroption,
              subID,
              index,
              hastitle,
              haslegend,
              ...)
  }


#' Plot wsimule result specified by user input
#'
#' This function can plot and return multiple sparse graphs distinguished by edge colors
#' from the result generated by wsimule
#' @author Beilun Wang, Zhaoyang Wang (Author), Zhaoyang Wang (maintainer) \email{zw4dn@virginia.edu}
#' @param x output generated from wsimule function (wsimule class)
#' @param type type of graph. There are four options:
#' * "task" (graph for each task (including shared part) specified further by subID (task number))
#' * "share" (shared graph for all tasks)
#' * "taskspecific" (graph for each task specific graph (excluding shared part)
#' specified further by subID (task number) )
#' * "neighbour" (zoom into nodes in the graph specified further by neighbouroptoin, subID (task number)
#' and index (node id))
#' @param neighbouroption determines what type of graph to zoom into when parameter **"type"** is **"neighbour"**. There are two options:
#' * "task" (zoom into graph for each task (including shared part))
#' * "taskspecific" (zoom into graph for each task specific (excluding shared part))
#' @param subID selects which task to display. There are four options:
#' * 0 (only allowed when
#' **"type"** is **"task"** or **"type"** is **"neighbour"** and **"neighbouroption"** is **"task"**) (selects share graph)
#' * positive task number (selects that particular task)
#' * a vector of task number (selects multiple tasks)
#' * NULL (selects all tasks (all graphs))
#' @param index determines which node(s) to zoom into when parameter **"type"** is **"neighbour"**.
#' This parameter could either be an integer or vector of integers representing node ids
#' (zoom into one node or multiple nodes)
#' @param hastitle determines whether the graph title is displayed or not (TRUE to display / FALSE to hide)
#' @param haslegend determines whether the graph legend is displayed or not (TRUE to display / FALSE to hide)
#' @param ... extra parameters passed to plot.igraph() and legend() (only the argument "legend" for legend() is available).
#' Please see \code{\link{plot.igraph}} and \code{\link{legend}}
#' @return a plot of graph / subgraph from wsimule result specified by user input
#' @details when only the wsimule result is provided, the function will plot all graphs with default numeric labels.
#' User can specify multiple subID and multiple index to zoom in multiple nodes on multiple graphs.
#' Each graph will include a descriptive title and legend to indicate correspondence between edge color and task.
#' @examples
#' library(JointNets)
#' data(exampleData)
#' result = wsimule(X = exampleData , lambda = 0.1, epsilon = 0.45,
#' W = matrix(1,20,20), covType = "cov", FALSE)
#' plot(result)
#' @method plot wsimule
#' @export
#' @export plot.wsimule
#' @import igraph
plot.wsimule <-
  function(x,
           type = "task",
           neighbouroption = "task",
           subID = NULL,
           index = NULL,
           hastitle = TRUE,
           haslegend = TRUE,
           ...)
  {
    jointplot(x,
              type,
              neighbouroption,
              subID,
              index,
              hastitle,
              haslegend,
              ...)
  }

#' Plot fasjem result specified by user input
#'
#' This function can plot and return multiple sparse graphs distinguished by edge colors
#' from the result generated by fasjem
#' @author Beilun Wang, Zhaoyang Wang (Author), Zhaoyang Wang (maintainer) \email{zw4dn@virginia.edu}
#' @param x output generated from fasjem function (fasjem class)
#' @param type type of graph. There are four options:
#' * "task" (graph for each task (including shared part) specified further by subID (task number))
#' * "share" (shared graph for all tasks)
#' * "taskspecific" (graph for each task specific graph (excluding shared part)
#' specified further by subID (task number) )
#' * "neighbour" (zoom into nodes in the graph specified further by neighbouroptoin, subID (task number)
#' and index (node id))
#' @param neighbouroption determines what type of graph to zoom into when parameter **"type"** is **"neighbour"**. There are two options:
#' * "task" (zoom into graph for each task (including shared part))
#' * "taskspecific" (zoom into graph for each task specific (excluding shared part))
#' @param subID selects which task to display. There are four options:
#' * 0 (only allowed when
#' **"type"** is **"task"** or **"type"** is **"neighbour"** and **"neighbouroption"** is **"task"**) (selects share graph)
#' * positive task number (selects that particular task)
#' * a vector of task number (selects multiple tasks)
#' * NULL (selects all tasks (all graphs))
#' @param index determines which node(s) to zoom into when parameter **"type"** is **"neighbour"**.
#' This parameter could either be an integer or vector of integers representing node ids
#' (zoom into one node or multiple nodes)
#' @param hastitle determines whether the graph title is displayed or not (TRUE to display / FALSE to hide)
#' @param haslegend determines whether the graph legend is displayed or not (TRUE to display / FALSE to hide)
#' @param ... extra parameters passed to plot.igraph() and legend() (only the argument "legend" for legend() is available).
#' Please see \code{\link{plot.igraph}} and \code{\link{legend}}
#' @return a plot of graph / subgraph from fasjem result specified by user input
#' @details when only the fasjem result is provided, the function will plot all graphs with default numeric labels.
#' User can specify multiple subID and multiple index to zoom in multiple nodes on multiple graphs.
#' Each graph will include a descriptive title and legend to indicate correspondence between edge color and task.
#' @examples
#' library(JointNets)
#' data(exampleData)
#' result = fasjem(X = exampleData, method = "fasjem-g", 0.5, 0.1, 0.1, 0.05, 10)
#' plot(result)
#' @method plot fasjem
#' @export
#' @export plot.fasjem
#' @import igraph
plot.fasjem <-
  function(x,
           type = "task",
           neighbouroption = "task",
           subID = NULL,
           index = NULL,
           hastitle = TRUE,
           haslegend = TRUE,
           ...)
  {
    jointplot(x,
              type,
              neighbouroption,
              subID,
              index,
              hastitle,
              haslegend,
              ...)
  }


#' plot diffee result specified by user input
#'
#' This function can plot diffee result
#' @author Beilun Wang, Zhaoyang Wang (Author), Zhaoyang Wang (maintainer) \email{zw4dn@virginia.edu}
#' @param x output generated from diffee function (diffee class)
#' @param type type of graph. There are two options:
#' * "task" (difference graph)
#' * "neighbour" (zoom into nodes in the difference graph specified further by parameter
#' **"index"** (node id)
#' @param index determines which node(s) to zoom into when parameter **"type"** is **"neighbour"**.
#' This parameter could either be an integer or vector of integers representing node ids
#' @param hastitle determines whether the graph title is displayed or not (TRUE to display / FALSE to hide)
#' @param ... extra parameters passed to  \code{\link{plot.igraph}}
#' (zoom into one node or multiple nodes)
#' @return a plot of the difference graph from diffee result specified by user input
#' @details when only the diffee result is provided, the function will plot all graphs with default numeric labels.
#' Users can specify multiple subID to zoom in multiple nodes.
#' Each graph will include a descriptive title.
#' @examples
#' library(JointNets)
#' data(exampleData)
#' result = diffee(exampleData[[1]], exampleData[[2]], 0.45)
#' plot.diffee(result)
#' @method plot diffee
#' @export
#' @export plot.diffee
#' @import igraph
plot.diffee <-
  function(x,
           type = "task",
           index = NULL,
           hastitle = TRUE,
           ...)
  {
    jointplot(x, type, "task", NULL, index, hastitle, FALSE, ...)
  }


#' plot diffeek result specified by user input
#'
#' This function can plot diffeek result
#' @author Beilun Wang, Zhaoyang Wang (Author), Zhaoyang Wang (maintainer) \email{zw4dn@virginia.edu}
#' @param x output generated from diffee function (diffee class)
#' @param type type of graph. There are two options:
#' * "task" (difference graph)
#' * "neighbour" (zoom into nodes in the difference graph specified further by parameter
#' **"index"** (node id)
#' @param index determines which node(s) to zoom into when parameter **"type"** is **"neighbour"**.
#' This parameter could either be an integer or vector of integers representing node ids
#' @param hastitle determines whether the graph title is displayed or not (TRUE to display / FALSE to hide)
#' @param ... extra parameters passed to  \code{\link{plot.igraph}}
#' (zoom into one node or multiple nodes)
#' @return a plot of the difference graph from diffeek result specified by user input
#' @details when only the diffeek result is provided, the function will plot all graphs with default numeric labels.
#' Users can specify multiple subID to zoom in multiple nodes.
#' Each graph will include a descriptive title.
#' @examples
#' library(JointNets)
#' data(exampleData)
#' result = diffeek(exampleData[[1]], exampleData[[2]],
#' W = matrix(1,20,20), g = rep(0,20),epsilon = 0.2,
#' lambda = 0.4,covType = "cov")
#' plot(result)
#' @method plot diffeek
#' @export
#' @export plot.diffeek
#' @import igraph
plot.diffeek <-
  function(x,
           type = "task",
           index = NULL,
           hastitle = TRUE,
           ...)
  {
    jointplot(x, type, "task", NULL, index, hastitle, FALSE, ...)
  }

#' core function to plot
#'
#' @param x output generated from JointNets
#' @param type type of graph. There are four options:
#' * "task" (graph for each task (including shared part) specified further by subID (task number))
#' * "share" (shared graph for all tasks)
#' * "taskspecific" (graph for each task specific graph (excluding shared part)
#' specified further by subID (task number) )
#' * "neighbour" (zoom into nodes in the graph specified further by neighbouroptoin, subID (task number)
#' and index (node id))
#' @param neighbouroption determines what type of graph to zoom into when parameter **"type"** is **"neighbour"**. There are two options:
#' * "task" (zoom into graph for each task (including shared part))
#' * "taskspecific" (zoom into graph for each task specific (excluding shared part))
#' @param subID selects which task to display. There are four options:
#' * 0 (only allowed when
#' **"type"** is **"task"** or **"type"** is **"neighbour"** and **"neighbouroption"** is **"task"**) (selects share graph)
#' * positive task number (selects that particular task)
#' * a vector of task number (selects multiple tasks)
#' * NULL (selects all tasks (all graphs))
#' @param index determines which node(s) to zoom into when parameter **"type"** is **"neighbour"**.
#' This parameter could either be an integer or vector of integers representing node ids
#' (zoom into one node or multiple nodes)
#' @param hastitle determines whether the graph title is displayed or not (TRUE to display / FALSE to hide)
#' @param haslegend determines whether the graph legend is displayed or not (TRUE to display / FALSE to hide)
#' @param ... extra parameters passed to plot.igraph() and legend() (only the argument "legend" for legend() is available).
#' Please see \code{\link{plot.igraph}} and \code{\link{legend}}
#' @return a plot of graph
#' @import methods
jointplot <-
  function(x,
           type = "task",
           neighbouroption = "task",
           subID = NULL,
           index = NULL,
           hastitle = TRUE,
           haslegend = TRUE,
           ...)
  {
    subID = unique(subID)
    index = unique(index)
    args = list(...)

    ### use S3 method returngraph
    gadj = returngraph(
      x,
      type = type,
      neighbouroption = neighbouroption,
      subID = subID,
      index = index
    )

    ### title for the graph
    title = NA
    if (hastitle) {
      glabel = V(gadj)$label
      if (methods::hasArg('vertex.label')) {
        glabel = args$vertex.label
      }
      ## make title according to user input
      title = maketitle(
        x,
        type = type,
        subID = subID,
        index = index,
        graphlabel = glabel,
        neighbouroption = neighbouroption
      )
    }

    ### plot the graph
    #layout to specify layout
    #vertex.label to specify label names
    igraph::plot.igraph(
      gadj,
      vertex.label.font = ifelse(methods::hasArg('vertex.label.font'), args$vertex.label.font, 2),
      vertex.shape =  ifelse(methods::hasArg('vertex.shape'), args$vertex.shape, "none"),
      vertex.label.color = ifelse(
        methods::hasArg('vertex.label.color'),
        args$vertex.label.color,
        "gray40"
      ),
      vertex.label.cex = ifelse(methods::hasArg('vertex.label.cex'), args$vertex.label.cex, .7),
      vertex.frame.color = ifelse(
        methods::hasArg('vertex.frame.color'),
        args$vertex.frame.color,
        "white"
      ),
      main = ifelse(methods::hasArg('main'), args$main, title),
      vertex.size = ifelse(methods::hasArg('vertex.size'), args$vertex.size, 10),
      ...
    )

    ### legend for the graph
    #colorspace::rainbow_hcl(5) better color segmentation but need to install colorspace
    #not providing the option to specify different legends since there are overlapping argumens with plot.igraph
    #user can specify further using the function graphics::legend
    #user can specify their own legend
    if (haslegend) {
      glegend = c(paste("task", c(1:length(x$graphs)), "specific"), "share")
      if (methods::hasArg('legend')) {
        glegend = args$legend
      }
      graphics::legend(
        "topright" ,
        inset=c(0,0),
        legend = glegend,
        col = grDevices::rainbow(length(x$graphs)+1),
        lty = 1,
        cex = 0.5,
        box.lty = 0,
        bg = "transparent"
      )
    }
  }


