% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lmm.R
\name{lmm}
\alias{lmm}
\title{Fit Linear Mixed Model}
\usage{
lmm(
  formula,
  repetition,
  structure,
  data,
  weights = NULL,
  scale.Omega = NULL,
  method.fit = NULL,
  df = NULL,
  type.information = NULL,
  trace = NULL,
  control = NULL
)
}
\arguments{
\item{formula}{[formula] Specify the model for the mean.
On the left hand side the outcome and on the right hand side the covariates affecting the mean value.
E.g. Y ~ Gender + Gene.}

\item{repetition}{[formula] Specify the model for the covariance.
On the right hand side the time/repetition variable and the grouping variable, e.g. ~ time|id.
On the left hand side, a possible stratification variable, e.g. group ~ time|id. In that case the mean structure should only be stratified on this variable using interactions.}

\item{structure}{[character] type of covariance structure, either \code{"CS"} (compound symmetry) or \code{"UN"} (unstructured).}

\item{data}{[data.frame] dataset (in the long format) containing the observations.}

\item{weights}{[formula or character] variable in the dataset used to weight the log-likelihood and its derivative. Should be constant within cluster.}

\item{scale.Omega}{[formula or character] variable in the dataset used to rescale the residual variance-covariance matrix. Should be constant within cluster.}

\item{method.fit}{[character] Should Restricted Maximum Likelihoood (\code{"REML"}) or Maximum Likelihoood (\code{"ML"}) be used to estimate the model parameters?}

\item{df}{[logical] Should the degree of freedom be computed using a Satterthwaite approximation?}

\item{type.information}{[character] Should the expected information be computed  (i.e. minus the expected second derivative) or the observed inforamtion (i.e. minus the second derivative).}

\item{trace}{[interger, >0] Show the progress of the execution of the function.}

\item{control}{[list] Control values for the optimization method. The element \code{optimizer} indicates which optimizer to use and additional argument will be pass to the optimizer.}
}
\value{
an object of class \code{lmm} containing the estimated parameter values, the residuals, and relevant derivatives of the likelihood.
}
\description{
Fit a linear mixed model defined by a mean and a covariance structure.
g
}
\details{
\bold{Computation time} the \code{lmm} has not been developped to be a fast function as, by default, it uses REML estimation with the observed information matrix and uses a Satterthwaite approximation to compute degrees of freedom (this require to compute the third derivative of the log-likelihood which is done by numerical differentiation). The computation time can be substantially reduced by using ML estimation with the expected information matrix and no calculation of degrees of freedom: arguments \code{method.fit="ML"}, \code{type.information="expected"}, \code{df=FALSE}. This will, however, lead to less accurate p-values and confidence intervals in small samples.

By default, the estimation of the model parameters will be made using the \code{nlme::gls} function.
See argument optimizer in \code{\link{LMMstar.options}}
}
\examples{
#### 1- simulate data in the long format ####
set.seed(10)
dL <- sampleRem(100, n.times = 3, format = "long")
dL$X1 <- as.factor(dL$X1)
dL$X2 <- as.factor(dL$X2)

#### 2- fit Linear Mixed Model ####
eCS.lmm <- lmm(Y ~ X1 * X2 + X5, repetition = ~visit|id, structure = "CS", data = dL)

logLik(eCS.lmm) ## -670.9439
summary(eCS.lmm)


#### 3- estimates ####
## reference level
levels(eCS.lmm)$reference
## mean parameters
coef(eCS.lmm)
model.tables(eCS.lmm)
confint(eCS.lmm)

if(require(emmeans)){
  dummy.coef(eCS.lmm)
}

## all parameters
coef(eCS.lmm, effects = "all")
model.tables(eCS.lmm, effects = "all")
confint(eCS.lmm, effects = "all")

## variance-covariance structure
getVarCov(eCS.lmm)

#### 4- diagnostic plots ####
quantile(residuals(eCS.lmm))
quantile(residuals(eCS.lmm, type = "normalized"))

\dontrun{
if(require(ggplot2)){
  ## investigate misspecification of the mean structure
  plot(eCS.lmm, type = "scatterplot")
  ## investigate misspecification of the variance structure
  plot(eCS.lmm, type = "scatterplot2")
  ## investigate misspecification of the correlation structure
  plot(eCS.lmm, type = "correlation")
  ## investigate misspecification of the residual distribution
  plot(eCS.lmm, type = "qqplot")
}
}

#### 5- statistical inference ####
anova(eCS.lmm) ## effect of each variable
anova(eCS.lmm, effects = "X11-X21=0") ## test specific coefficient
## test several hypothese with adjustment for multiple comparisons
summary(anova(eCS.lmm, effects = c("X11=0","X21=0")))

#### 6- prediction ####
## conditional on covariates
newd <- dL[1:3,]
predict(eCS.lmm, newdata = newd, keep.newdata = TRUE)
## conditional on covariates and outcome
newd <- dL[1:3,]
newd$Y[3] <- NA
predict(eCS.lmm, newdata = newd, type = "dynamic", keep.newdata = TRUE)

#### EXTRA ####
if(require(mvtnorm)){
## model for the average over m replicates
## (only works with independent replicates)
Sigma1 <- diag(1,1,1); Sigma5 <- diag(1,5,5)
n <- 100
dfW <- rbind(data.frame(id = 1:n, rep = 5, Y = rowMeans(rmvnorm(n, sigma = Sigma5))),
             data.frame(id = (n+1):(2*n), rep = 1, Y = rmvnorm(n, sigma = Sigma1)))

e.lmmW <- lmm(Y~1, data = dfW, scale.Omega=~rep, control = list(optimizer = "FS"))
e.lmm0 <- lmm(Y~1, data = dfW)
model.tables(e.lmmW, effects = "all")
model.tables(e.lmm0, effects = "all")
## TRUE standard error is 1

}

}
\seealso{
\code{\link{summary.lmm}} for a summary of the model fit. \cr
\code{\link{model.tables.lmm}} for a data.frame containing estimates with their uncertainty. \cr
\code{\link{plot.lmm}} for a graphical display of the model fit or diagnostic plots. \cr
\code{\link{levels.lmm}} to display the reference level. \cr
\code{\link{anova.lmm}} for testing linear combinations of coefficients (F-test, multiple Wald tests) \cr
\code{\link{getVarCov.lmm}} for extracting estimated residual variance-covariance matrices. \cr
\code{\link{residuals.lmm}} for extracting residuals or creating residual plots (e.g. qqplots). \cr
\code{\link{predict.lmm}} for evaluating mean and variance of the outcome conditional on covariates or other outcome values.
}
