#' Convert a LorMe object to a microeco object
#'
#' @param taxobj      A \code{\link{LorMe}} object (usually generated by \code{tax_summary} ).
#' @param use_reads   Logical. \code{FALSE} (default) uses **relative abundance** (\code{Base_percent}); \code{TRUE} uses **raw counts**
#'   (\code{Base}).
#'
#' @return A \code{\link[microeco:microtable]{microtable}} object.
#' @export
#'
#' @examples
#' \dontrun{
#' data("Two_group")
#'
#' Two_group_microeco=Trans_to_microeco(Two_group)
#'
#' t1 <- trans_abund$new(dataset = Two_group_microeco, taxrank = "Phylum", ntaxa = 10)
#' t1$plot_bar(others_color = "grey70",
#'             legend_text_italic = FALSE) +
#'             theme(axis.title.y = element_text(size = 18)) #Just to show it can be used for analysis
#' }
Trans_to_microeco <- function(taxobj, use_reads = TRUE) {
  if (!requireNamespace("microeco", quietly = TRUE))
    stop("microeco required: install.packages('microeco')")
  #extraction
  meta     <- methods::slot(taxobj, "groupfile")
  data_lst <- methods::slot(taxobj, "data")
  tree     <- methods::slot(taxobj, "tree")

  ## select relative abundance
  base_key <- if (use_reads==TRUE) "Base" else "Base_percent"
  if (is.null(data_lst[[base_key]]))
    stop(sprintf("'%s' not found in taxobj", base_key))

  otu_df <- data_lst[[base_key]]
  tax_df <- data_lst$Base_taxonomy

  ## build microeco compartments
  otu_mat <- as.matrix(otu_df[, -1])  %>% as.data.frame()
  rownames(otu_mat) <- otu_df$ID

  tax_mat <- as.matrix(tax_df[, -1])     %>% as.data.frame()
  rownames(tax_mat) <- tax_df$ID

  meta_df <- as.data.frame(meta)
  rownames(meta_df) <- colnames(otu_mat)

  ## encapsulation
  microeco::microtable$new(sample_table = meta_df,
                 otu_table = otu_mat,
                 tax_table = tax_mat,
                 phylo_tree = tree)
}
