% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/analyse_SAR.CWOSL.R
\name{analyse_SAR.CWOSL}
\alias{analyse_SAR.CWOSL}
\title{Analyse SAR CW-OSL measurements}
\usage{
analyse_SAR.CWOSL(object, signal.integral.min, signal.integral.max,
  background.integral.min, background.integral.max,
  rejection.criteria = NULL, dose.points = NULL, mtext.outer,
  plot = TRUE, plot.single = FALSE, onlyLxTxTable = FALSE, ...)
}
\arguments{
\item{object}{\linkS4class{RLum.Analysis} (\strong{required}):
input object containing data for analysis, alternatively a \link{list} of
\linkS4class{RLum.Analysis} objects can be provided.}

\item{signal.integral.min}{\link{integer} (\strong{required}):
lower bound of the signal integral. Can be a \link{list} of \link{integer}s, if \code{object} is
of type \link{list}. If the input is vector (e.g., \code{c(1,2)}) the 2nd value will be interpreted
as the minimum signal integral for the Tx curve.}

\item{signal.integral.max}{\link{integer} (\strong{required}):
upper bound of the signal integral. Can be a \link{list} of \link{integer}s, if \code{object} is
of type \link{list}. If the input is vector (e.g., \code{c(1,2)}) the 2nd value will be interpreted
as the maximum signal integral for the Tx curve.}

\item{background.integral.min}{\link{integer} (\strong{required}):
lower bound of the background integral. Can be a \link{list} of \link{integer}s, if \code{object} is
of type \link{list}. If the input is vector (e.g., \code{c(1,2)}) the 2nd value will be interpreted
as the minimum background integral for the Tx curve.}

\item{background.integral.max}{\link{integer} (\strong{required}):
upper bound of the background integral. Can be a \link{list} of \link{integer}s, if \code{object} is
of type \link{list}. If the input is vector (e.g., \code{c(1,2)}) the 2nd value will be interpreted
as the maximum background integral for the Tx curve.}

\item{rejection.criteria}{\link{list} (\emph{with default}):
provide a named list and set rejection criteria in \strong{percentage} for further calculation. Can be a \link{list} in
a \link{list}, if \code{object} is of type \link{list}

Allowed arguments are \code{recycling.ratio}, \code{recuperation.rate},
\code{palaeodose.error}, \code{testdose.error} and \code{exceed.max.regpoint = TRUE/FALSE}.
Example: \code{rejection.criteria = list(recycling.ratio = 10)}.
Per default all numerical values are set to 10, \code{exceed.max.regpoint = TRUE}.
Every criterium can be set to \code{NA}. In this value are calculated, but not considered, i.e.
the RC.Status becomes always \code{'OK'}}

\item{dose.points}{\link{numeric} (\emph{optional}):
a numeric vector containg the dose points values Using this argument
overwrites dose point values in the signal curves. Can be a \link{list} of
\link{numeric} vectors, if \code{object} is of type \link{list}}

\item{mtext.outer}{\link{character} (\emph{optional}):
option to provide an outer margin mtext. Can be a \link{list} of \link{character}s,
if \code{object} is of type \link{list}}

\item{plot}{\link{logical} (\emph{with default}):
enables or disables plot output.}

\item{plot.single}{\link{logical} (\emph{with default}) or \link{numeric} (\emph{optional}):
single plot output (\code{TRUE/FALSE}) to allow for plotting the results in single plot windows.
If a numerice vector is provided the plots can be selected individually, i.e.
\code{plot.single = c(1,2,3,4)} will plot the TL and Lx, Tx curves but not the legend (5) or the
growth curve (6), (7) and (8) belong to rejection criteria plots. Requires
\code{plot = TRUE}.}

\item{onlyLxTxTable}{\link{logical} (with default): If \code{TRUE} the dose response curve fitting and plotting
is skipped. This allows to get hands on the Lx/Tx table for large datasets without the need for
a curve fitting.}

\item{...}{further arguments that will be passed to the function
\link{plot_GrowthCurve} or \link{calc_OSLLxTxRatio}
(supported: \code{background.count.distribution}, \code{sigmab}, \code{sig0}).
\strong{Please note} that if you consider to use the early light subtraction
method you should provide your own \code{sigmab} value!}
}
\value{
A plot (\emph{optional}) and an \linkS4class{RLum.Results} object is
returned containing the following elements:

\item{data}{\link{data.frame} containing De-values, De-error and further parameters}
\item{LnLxTnTx.values}{\link{data.frame} of all calculated Lx/Tx values including signal, background counts and the dose points}
\item{rejection.criteria}{\link{data.frame} with values that might by used as rejection criteria. NA is produced if no R0 dose point exists.}
\item{Formula}{\link{formula} formula that have been used for the growth curve fitting }

The output should be accessed using the function \link{get_RLum}.
}
\description{
The function performs a SAR CW-OSL analysis on an
\linkS4class{RLum.Analysis} object including growth curve fitting.
}
\details{
The function performs an analysis for a standard SAR protocol measurements
introduced by Murray and Wintle (2000) with CW-OSL curves. For the
calculation of the Lx/Tx value the function \link{calc_OSLLxTxRatio} is
used. For \strong{changing the way the Lx/Tx error is calculated} use the argument
\code{background.count.distribution} and \code{sigmab}, which will be passed to the function
\link{calc_OSLLxTxRatio}.

\strong{Argument \code{object} is of type \code{list}}

If the argument \code{object} is of type \link{list} containing \strong{only}
\linkS4class{RLum.Analysis} objects, the function re-calls itself as often as elements
are in the list. This is usefull if an entire measurement wanted to be analysed without
writing separate for-loops. To gain in full control of the parameters (e.g., \code{dose.points}) for
every aliquot (corresponding to one \linkS4class{RLum.Analysis} object in the list), in
this case the arguments can be provided as \link{list}. This \code{list} should
be of similar length as the \code{list} provided with the argument \code{object},
otherwise the function will create an own list of the requested lenght.
Function output will be just one single \linkS4class{RLum.Results} object.

Please be careful when using this option. It may allow a fast an efficient data analysis, but
the function may also break with an unclear error message, due to wrong input data.

\strong{Working with IRSL data}

The function was originally designed to work just for 'OSL' curves,
following the principles of the SAR protocol. An IRSL measurement protocol
may follow this procedure, e.g., post-IR IRSL protocol (Thomsen et al.,
2008). Therefore this functions has been enhanced to work with IRSL data,
however, the function is only capable of analysing curves that follow the
SAR protocol structure, i.e., to analyse a post-IR IRSL protocol, curve data
have to be pre-selected by the user to fit the standards of the SAR
protocol, i.e., Lx,Tx,Lx,Tx and so on.

Example: Imagine the measurement contains pIRIR50 and pIRIR225 IRSL curves.
Only one curve type can be analysed at the same time: The pIRIR50 curves or
the pIRIR225 curves.

\strong{Supported rejection criteria}

\code{[recycling.ratio]}: calculated for every repeated regeneration dose point.

\code{[recuperation.rate]}: recuperation rate calculated by comparing the
Lx/Tx values of the zero regeneration point with the Ln/Tn value (the Lx/Tx
ratio of the natural signal). For methodological background see Aitken and
Smith (1988).

\code{[testdose.error]}: set the allowed error for the testdose, which per
default should not exceed 10\%. The testdose error is calculated as Tx_net.error/Tx_net.

\code{[palaeodose.error]}: set the allowed error for the De value, which per
default should not exceed 10\%.
}
\note{
This function must not be mixed up with the function
\link{Analyse_SAR.OSLdata}, which works with
\linkS4class{Risoe.BINfileData} objects.

\strong{The function currently does support only 'OSL', 'IRSL' and 'POSL' data!}
}
\section{Function version}{
 0.8.8
}

\examples{

##load data
##ExampleData.BINfileData contains two BINfileData objects
##CWOSL.SAR.Data and TL.SAR.Data
data(ExampleData.BINfileData, envir = environment())

##transform the values from the first position in a RLum.Analysis object
object <- Risoe.BINfileData2RLum.Analysis(CWOSL.SAR.Data, pos=1)

##perform SAR analysis and set rejection criteria
results <- analyse_SAR.CWOSL(
object = object,
signal.integral.min = 1,
signal.integral.max = 2,
background.integral.min = 900,
background.integral.max = 1000,
log = "x",
fit.method = "EXP",
rejection.criteria = list(
  recycling.ratio = 10,
  recuperation.rate = 10,
  testdose.error = 10,
  palaeodose.error = 10,
  exceed.max.regpoint = TRUE)
)

##show De results
get_RLum(results)

##show LnTnLxTx table
get_RLum(results, data.object = "LnLxTnTx.table")

} 

\section{How to cite}{
Kreutzer, S., 2019. analyse_SAR.CWOSL(): Analyse SAR CW-OSL measurements. Function version 0.8.8. In: Kreutzer, S., Burow, C., Dietze, M., Fuchs, M.C., Schmidt, C., Fischer, M., Friedrich, J., 2019. Luminescence: Comprehensive Luminescence Dating Data AnalysisR package version 0.9.4. https://CRAN.R-project.org/package=Luminescence
}

\references{
Aitken, M.J. and Smith, B.W., 1988. Optical dating: recuperation
after bleaching. Quaternary Science Reviews 7, 387-393.

Duller, G., 2003. Distinguishing quartz and feldspar in single grain
luminescence measurements. Radiation Measurements, 37 (2), 161-165.

Murray, A.S. and Wintle, A.G., 2000. Luminescence dating of quartz using an
improved single-aliquot regenerative-dose protocol. Radiation Measurements
32, 57-73.

Thomsen, K.J., Murray, A.S., Jain, M., Boetter-Jensen, L., 2008. Laboratory
fading rates of various luminescence signals from feldspar-rich sediment
extracts. Radiation Measurements 43, 1474-1486.
doi:10.1016/j.radmeas.2008.06.002
}
\seealso{
\link{calc_OSLLxTxRatio}, \link{plot_GrowthCurve}, \linkS4class{RLum.Analysis},
\linkS4class{RLum.Results}, \link{get_RLum}
}
\author{
Sebastian Kreutzer, IRAMAT-CRP2A, UMR 5060, CNRS-Université Bordeaux Montaigne (France)
, RLum Developer Team}
\keyword{datagen}
\keyword{plot}
