% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MonteCarlo.R
\name{MonteCarlo}
\alias{MonteCarlo}
\title{Parallized Monte Carlo Simulation}
\usage{
MonteCarlo(func, nrep, param_list, ncpus = 1, max_grid = 1000,
  time_n_test = FALSE, save_res = FALSE, raw = TRUE, export_also = NULL)
}
\arguments{
\item{func}{The function to be evaluated. See details.}

\item{nrep}{An integer that specifies the desired number of Monte Carlo repetitions.}

\item{param_list}{A list whose components are named after the parameters of \code{func} and each component is a vector containing the desired grid values for that parameter}

\item{ncpus}{An integer specifying the number of cpus to be used. Default is \code{ncpus=1}. 
For \code{ncpus>1} the simulation is parallized automatically using \code{ncpus} cpu units.}

\item{max_grid}{Integer that specifies for which grid size to throw an error, if grid becomes to large. Default is \code{max_grid=1000}.}

\item{time_n_test}{Boolean that specifies whether the required simulation time should be estimated (useful for large simulations or slow functions). 
See details. Default is \code{time_n_test=FALSE}.}

\item{save_res}{Boolean that specifies whether the results of \code{time_n_test} should be saved to the current directory. 
Default is \code{save_res=FALSE}.}

\item{raw}{Boolean that specifies whether the output should be averaged over the nrep repetitions. Default is \code{raw=TRUE}.}

\item{export_also}{List specifying additional objects that are supposed to be exported to the cluster. 
This allows to export data or to bypass the automatic export of functions. Default is \code{export_also=NULL}. See details.}
}
\value{
A list of type \code{MonteCarlo}.
}
\description{
\code{MonteCarlo} runs a Monte Carlo simulation study for a correctly specified function and the desired parameter grids. 
See details for instructions on the specification of the function.
}
\details{
The user defined function \code{func} handles the generation of data, the application of the method of interest 
and the evaluation of the result for a single repetition and parameter combination. 
MonteCarlo handles the generation of loops over the desired parameter grids and the 
repetition of the Monte Carlo experiment for each of the parameter constellations.
 
There are two important formal requirements that \code{func} has to fulfill. 

1. The arguments of \code{func} have to be scalar.

2. The value returned by \code{func} has to be list of (unnamed) scalars (The list elements can be named). 

For the estimation of the required simulation time, 
a separate simulation is run on a reduced grid that only contains the extreme points 
for each parameter, e.g. the smallest and the largest sample size. 
This test simulation is carried out with \code{nrep/10} repetitions and the required 
simulation time is estimated by a linear interpolation. Since the computational complexity is
usually a convex function of the sample size and the dimension of the process, this approach 
tends to overestimate the time required. 

\code{export_also} allows to export data to the cluster in case parallized computations on a dataset are desired. 
It also allows to bypass the automatic export of functions and packages. 
To manually export a function or dataset or to load a package, pass a list to \code{export_also} where the list element is named
"functions", "data" and/or "packages". For example: \code{export_also=list("functions"=c("function_name_1", "function_name_2"), 
"packages"="package_name")}.
}
\examples{
test_func<-function(n,loc,scale){
 sample<-rnorm(n, loc, scale)
 stat<-sqrt(n)*mean(sample)/sd(sample)
 decision<-abs(stat)>1.96
 return(list("decision"=decision))
}

# Example without parallization
n_grid<-c(50,100,250,500)
loc_grid<-seq(0,1,0.2)
scale_grid<-c(1,2)

param_list=list("n"=n_grid, "loc"=loc_grid, "scale"=scale_grid)
erg<-MonteCarlo(func=test_func, nrep=250, param_list=param_list, ncpus=1)
summary(erg)

rows<-c("n")
cols<-c("loc","scale")
MakeTable(output=erg, rows=rows, cols=cols, digits=2)

}

