% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DPMGibbsN_parallel.R
\name{DPMGibbsN_parallel}
\alias{DPMGibbsN_parallel}
\title{Slice Sampling of the Dirichlet Process Mixture Model
with a prior on alpha}
\usage{
DPMGibbsN_parallel(
  Ncpus,
  type_connec,
  z,
  hyperG0,
  a = 1e-04,
  b = 1e-04,
  N,
  doPlot = TRUE,
  nbclust_init = 30,
  plotevery = N/10,
  diagVar = TRUE,
  use_variance_hyperprior = TRUE,
  verbose = TRUE,
  monitorfile = "",
  ...
)
}
\arguments{
\item{Ncpus}{the number of processors available}

\item{type_connec}{The type of connection between the processors. Supported
cluster types are \code{"SOCK"}, \code{"FORK"}, \code{"MPI"}, and
\code{"NWS"}. See also \code{\link[parallel:makeCluster]{makeCluster}}.}

\item{z}{data matrix \code{d x n} with \code{d} dimensions in rows
and \code{n} observations in columns.}

\item{hyperG0}{prior mixing distribution.}

\item{a}{shape hyperparameter of the Gamma prior
on the concentration parameter of the Dirichlet Process. Default is \code{0.0001}.}

\item{b}{scale hyperparameter of the Gamma prior
on the concentration parameter of the Dirichlet Process. Default is \code{0.0001}. If \code{0}, 
then the concentration is fixed set to \code{a}.}

\item{N}{number of MCMC iterations.}

\item{doPlot}{logical flag indicating whether to plot MCMC iteration or not.
Default to \code{TRUE}.}

\item{nbclust_init}{number of clusters at initialization.
Default to 30 (or less if there are less than 30 observations).}

\item{plotevery}{an integer indicating the interval between plotted iterations when \code{doPlot}
is \code{TRUE}.}

\item{diagVar}{logical flag indicating whether the variance of each cluster is
estimated as a diagonal matrix, or as a full matrix.
Default is \code{TRUE} (diagonal variance).}

\item{use_variance_hyperprior}{logical flag indicating whether a hyperprior is added 
for the variance parameter. Default is \code{TRUE} which decrease the impact of the variance prior
on the posterior. \code{FALSE} is useful for using an informative prior.}

\item{verbose}{logical flag indicating whether partition info is
written in the console at each MCMC iteration.}

\item{monitorfile}{a writable \link{connections} or a character string naming a file to write into,
to monitor the progress of the analysis.
Default is \code{""} which is no monitoring.  See Details.}

\item{...}{additional arguments to be passed to \code{\link{plot_DPM}}.
Only used if \code{doPlot} is \code{TRUE}.}
}
\value{
a object of class \code{DPMclust} with the following attributes:
 \itemize{
     \item{\code{mcmc_partitions}:}{ a list of length \code{N}. Each
      element \code{mcmc_partitions[n]} is a vector of length
      \code{n} giving the partition of the \code{n} observations.}
     \item{\code{alpha}:}{a vector of length \code{N}. \code{cost[j]} is the cost
associated to partition \code{c[[j]]}}
      \item{\code{listU_mu}:}{a list of length \code{N} containing the matrices of
      mean vectors for all the mixture components at each MCMC iteration}
      \item{\code{listU_Sigma}:}{a list of length \code{N} containing the arrays of
      covariances matrices for all the mixture components at each MCMC iteration}
      \item{\code{U_SS_list}:}{a list of length \code{N} containing the lists of
      sufficient statistics for all the mixture components at each MCMC iteration}
     \item{\code{weights_list}:}{a list of length \code{N} containing the logposterior values
      at each MCMC iterations}
     \item{\code{logposterior_list}:}{a list of length \code{N} containing the logposterior values
      at each MCMC iterations}
     \item{\code{data}:}{the data matrix \code{d x n} with \code{d} dimensions in rows
and \code{n} observations in columns}
     \item{\code{nb_mcmcit}:}{ the number of MCMC iterations}
     \item{\code{clust_distrib}:}{the parametric distribution of the mixture component - \code{"gaussian"}}
     \item{\code{hyperG0}:}{the prior on the cluster location}
 }
}
\description{
Slice Sampling of the Dirichlet Process Mixture Model
with a prior on alpha
}
\examples{

# Scaling up: ----
rm(list=ls())
#Number of data
n <- 2000
set.seed(1234)

# Sample data
d <- 3
nclust <- 5
m <- matrix(nrow=d, ncol=nclust, runif(d*nclust)*8)
# p: cluster probabilities
p <- runif(nclust)
p <- p/sum(p)

# Covariance matrix of the clusters
sdev <- array(dim=c(d, d, nclust))
for (j in 1:nclust){
    sdev[, ,j] <- matrix(NA, nrow=d, ncol=d)
    diag(sdev[, ,j]) <- abs(rnorm(n=d, mean=0.3, sd=0.1))
    sdev[, ,j][lower.tri(sdev[, ,j], diag = FALSE)] <- rnorm(n=d*(d-1)/2,
    mean=0, sd=0.05)
    sdev[, ,j][upper.tri(sdev[, ,j], diag = FALSE)] <- (sdev[, ,j][
                                                        lower.tri(sdev[, ,j], diag = FALSE)])
}
c <- rep(0,n)
z <- matrix(0, nrow=d, ncol=n)
for(k in 1:n){
    c[k] = which(rmultinom(n=1, size=1, prob=p)!=0)
    z[,k] <- m[, c[k]] + sdev[, , c[k]]\%*\%matrix(rnorm(d, mean = 0, sd = 1), nrow=d, ncol=1)
    #cat(k, "/", n, " observations simulated\n", sep="")
}

# hyperprior on the Scale parameter of DPM
a <- 0.001
b <- 0.001

# Number of iterations
N <- 25

# do some plots
doPlot <- TRUE

# Set parameters of G0
hyperG0 <- list()
hyperG0[["mu"]] <- rep(0, d)
hyperG0[["kappa"]] <- 0.01
hyperG0[["nu"]] <- d + 2
hyperG0[["lambda"]] <- diag(d)/10


nbclust_init <- 30

if(interactive()){
 library(doParallel)
 MCMCsample <- DPMGibbsN_parallel(Ncpus=2, type_connec="FORK", z, hyperG0, a, b, 
                                  N=1000, doPlot=FALSE, nbclust_init=30, 
                                  plotevery=100, gg.add=list(ggplot2::theme_bw(),
                                  ggplot2::guides(shape = 
                                    ggplot2::guide_legend(override.aes = list(fill="grey45")))),
                                  diagVar=FALSE)
}


}
\seealso{
\code{\link{DPMGibbsN}}
}
\author{
Boris Hejblum
}
