% Generated by roxygen2 (4.0.1): do not edit by hand
\name{presid}
\alias{presid}
\title{Probability-scale residual}
\usage{
presid(object, ...)
}
\arguments{
\item{object}{The model object for which the probability-scale residual is calculated}

\item{...}{Additional arguements passed to methods}
}
\value{
The probability-scale residual for the model
}
\description{
\code{presid} calculates the probability-scale residual for various model
function objects. Currently supported models include \code{\link{glm}}
(Poisson, binomial, and gaussian families), \code{\link{lm}} in the
\pkg{stats} library; \code{\link{survreg}} (Weibull, exponential, gaussian,
logistic, and lognormal distributions) and \code{\link{coxph}} in the
\pkg{survival} library; \code{\link{polr}} and \code{\link{glm.nb}} in
the \pkg{MASS} library; and \code{\link{ols}}, \code{\link{cph}},
\code{\link{lrm}}, \code{\link{orm}}, \code{\link{psm}}, and \code{\link{Glm}}
in the \pkg{rms} library.
}
\details{
Probability-scale residual is \eqn{P(Y < y) - P(Y > y)} where \eqn{y} is the observed
outcome and \eqn{Y} is a random variable from the fitted distribution.
}
\examples{
library(survival)
library(stats)

set.seed(100)
n <- 1000
x <- rnorm(n)
t <- rweibull(n, shape=1/3, scale=exp(x))
c <- rexp(n, 1/3)
y <- pmin(t, c)
d <- ifelse(t<=c, 1, 0)

mod.survreg <- survreg(Surv(y, d) ~ x, dist="weibull")
summary(presid(mod.survreg))
plot(x, presid(mod.survreg))

##### example for proprotional hazards model
n <- 1000
x <- rnorm(n)
beta0 <- 1
beta1 <- 0.5
t <- rexp(n, rate = exp(beta0 + beta1*x))
c <- rexp(n, rate=1)
y <- ifelse(t<c, t, c)
delta <- as.integer(t<c)

mod.coxph <- coxph(Surv(y, delta) ~ x)
presid <- presid(mod.coxph)
plot(x, presid, cex=0.4, col=delta+2)

#### example for Negative Binomial regression
library(MASS)

n <- 1000
beta0 <- 1
beta1 <- 0.5
x <- runif(n, min=-3, max=3)
y <- rnbinom(n, mu=exp(beta0 + beta1*x), size=3)

mod.glm.nb <- glm.nb(y~x)
presid <- presid(mod.glm.nb)
summary(presid)
plot(x, presid, cex=0.4)

##### example for proportional odds model
library(MASS)

n <- 1000
x  <- rnorm(n)
y  <- numeric(n)
alpha = c(-1, 0, 1, 2)
beta <- 1
py  <-  (1 + exp(- outer(alpha, beta*x, "+"))) ^ (-1)
aa = runif(n)
for(i in 1:n)
  y[i] = sum(aa[i] > py[,i])
y <-  as.factor(y)


mod.polr <- polr(y~x, method="logistic")
summary(mod.polr)
presid <- presid(mod.polr)
summary(presid)
plot(x, presid, cex=0.4)
}
\references{
Shepherd BE, Li C, Liu Q.  Probability-scale residuals for continuous,
discrete, and censored data.  Submitted.

Li C and Shepherd BE, A new residual for ordinal
outcomes. Biometrika 2012; 99:473-480
}

