#' Gets fluorescence data vectors from \code{RDML} object
#' 
#' Gets fluorescence data vectors from \code{RDML} object for specified method
#' of experiment.
#' 
#' @param request Output from AsTable method(\link{RDML.AsTable})
#' @param limits \code{vector} with two values (min and max values) that limits
#' cycles or temperature that should be gotten. If this values are smaller or bigger 
#' than min or max values at dats - NA will be used.
#' @param data.type Type of fluorescence data (i.e. 'adp' for qPCR or 'mdp' for
#'   melting)
#' @param long.table Output table is ready for ggplot (See \link{RDML.AsTable}
#'   for example)
#' @return \code{matrix} which contains selected fluorescence data and 
#'   additional information fromm request if \code{long.table = TRUE}.
#' @author Konstantin A. Blagodatskikh <k.blag@@yandex.ru>, Stefan Roediger 
#'   <stefan.roediger@@b-tu.de>, Michal Burdukiewicz 
#'   <michalburdukiewicz@@gmail.com> 
#' @keywords manip
#' @docType methods
#' @name RDML.GetFData
#' @rdname getfdata-method
#' @include RDML.R
#' @examples
#' \dontrun{
#' ## internal dataset BioRad_qPCR_melt.rdml (in 'data' directory)
#' ## generated by Bio-Rad CFX96. Contains qPCR and melting data.
#' ## Import without splitting by targets/types and with
#' ## custom name pattern.
#' PATH <- path.package("RDML")
#' filename <- paste(PATH, "/extdata/", "BioRad_qPCR_melt.rdml", sep ="")
#' cfx96 <- RDML$new(filename)
#' ## Select melting fluorescence data with sample.type 'unkn'.
#' library(dplyr)
#' tab <- cfx96$AsTable()
#' fdata <- cfx96$GetFData(filter(tab, sample.type == "unkn")
#'                         data.type = "mdp")
#' ## Show names for getted fdata
#' colnames(fdata)
#' }
RDML$set("public", "GetFData",
         function(request,
                  limits = NULL,
                  dp.type = "adp",
                  long.table = FALSE) {
           out <- 
             ddply(request, .(fdata.name), function(el) {
               private$.experiment[[el$exp.id]]$run[[el$run.id]]$react[[as.character(
                 el$react.id)]]$data[[el$target]]$AsDataFrame(dp.type = dp.type) %>% 
                 cbind(fdata.name = el$fdata.name)
             },
             .id = "fdata.name"
           )
           ## Asserts for limits
           if (!is.null(limits)) {
             out <- out[out[, 1] >= limits[1] & out[, 1] <= limits[2], ]
           }
           if (long.table == FALSE) {
             out <- out %>%
               spread(fdata.name,
                      fluor)
           } else {
             rownames(request) <- request$fdata.name
             for(param in colnames(request)[-1]) {
               for(fn in request$fdata.name) {
                 out[out$fdata.name == fn,
                     param] <- request[fn, param]
               }
             }
           }
           out
         }, 
         overwrite = TRUE)