% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/redcap-metadata-coltypes.R
\name{redcap_metadata_coltypes}
\alias{redcap_metadata_coltypes}
\title{Suggests a col_type for each field in a REDCap project}
\usage{
redcap_metadata_coltypes(
  redcap_uri,
  token,
  print_col_types_to_console = TRUE,
  http_response_encoding = "UTF-8",
  locale = readr::default_locale(),
  verbose = FALSE,
  config_options = NULL,
  handle_httr = NULL
)
}
\arguments{
\item{redcap_uri}{The
\href{https://en.wikipedia.org/wiki/Uniform_Resource_Identifier}{uri}/url
of the REDCap server
typically formatted as "https://server.org/apps/redcap/api/".
Required.}

\item{token}{The user-specific string that serves as the password for a
project.  Required.}

\item{print_col_types_to_console}{Should the \code{\link[readr:cols]{readr::cols()}} object
be printed to the console?}

\item{http_response_encoding}{The encoding value passed to
\code{\link[httr:content]{httr::content()}}.  Defaults to 'UTF-8'.}

\item{locale}{a \code{\link[readr:locale]{readr::locale()}} object to specify preferences like
number, date, and time formats.  This object is passed to
\code{\link[readr:read_delim]{readr::read_csv()}}.  Defaults to \code{\link[readr:locale]{readr::default_locale()}}.}

\item{verbose}{A boolean value indicating if \code{message}s should be printed
to the R console during the operation.  The verbose output might contain
sensitive information (\emph{e.g.} PHI), so turn this off if the output might
be visible somewhere public. Optional.}

\item{config_options}{A list of options passed to \code{\link[httr:POST]{httr::POST()}}.
See details at \code{\link[httr:httr_options]{httr::httr_options()}}. Optional.}

\item{handle_httr}{The value passed to the \code{handle} parameter of
\code{\link[httr:POST]{httr::POST()}}.
This is useful for only unconventional authentication approaches.  It
should be \code{NULL} for most institutions.  Optional.}
}
\value{
A \code{\link[readr:cols]{readr::cols()}} object is returned, which can be
passed to \code{\link[=redcap_read]{redcap_read()}} or \code{\link[=redcap_read_oneshot]{redcap_read_oneshot()}}.

Additionally objected is printed to the console, see the Details below.
}
\description{
This function inspects a REDCap project to
determine a \code{\link[readr:cols]{readr::cols()}} object that is compatible with the
the project's current definition.  It can be copied and pasted into the
R code so future calls to the server will produce a \code{\link[tibble:tibble]{tibble::tibble()}}
with an equivalent set of data types.
}
\details{
\code{redcap_metadata_coltypes()} returns a \code{\link[readr:cols]{readr::cols()}} object in two ways.
First, a literal object is returned that can be passed to
\code{\link[=redcap_read]{redcap_read()}} or \code{\link[=redcap_read_oneshot]{redcap_read_oneshot()}}.

Second, the function acts as a code generator.
It prints text to the console so that it can be copied
and pasted into an R file.  This is useful to (a) document what
fields and data types are expected, and (b) adjust those fields and
data types if the defaults can be customized for your needs.
For instance, you may choose to exclude some variables or tweak a
data type (\emph{e.g.}, changing a patient's height from an integer to
a double).

When printing to the console, each data type decision is accompanied
by an explanation on the far right.  See the output from the
examples below.  Please file an
\href{https://github.com/OuhscBbmc/REDCapR/issues}{issue} if you think
something is too restrictive or can be improved.

The overall heuristic is assign a data type down a waterfall of decisions:
\enumerate{
\item Is the field built into REDCap? This includes
an autonumber \code{record_id},
\code{redcap_event_name}, \code{redcap_repeat_instrument}, \code{redcap_repeat_instance},
and an instrument's completion status.
\item What is the field's type?  For example, sliders should be an
\href{https://stat.ethz.ch/R-manual/R-devel/library/base/html/integer.html}{integer},
while check marks should be
\link{logical}(https://stat.ethz.ch/R-manual/R-devel/library/base/html/logical.html.
\item If the field type is "text", what is the validation type?
For instance, a postal code should be a
\href{https://stat.ethz.ch/R-manual/R-devel/library/base/html/character.html}{character}
(even though it looks like a number),
a "mdy" should be cast to a
\href{https://stat.ethz.ch/R-manual/R-devel/library/base/html/date.html}{date},
and a "number_2dp" should be cast to a
\href{https://stat.ethz.ch/R-manual/R-devel/library/base/html/double.html}{floating point}
\item If the field type or validation type is not recognized,
the field will be cast to
\href{https://stat.ethz.ch/R-manual/R-devel/library/base/html/character.html}{character}.
This will happen when REDCap develops & releases a new type.
If you see something like, "# validation doesn't have an associated col_type.
Tell us in a new REDCapR issue", please make sure REDCapR is running the newest
\href{https://ouhscbbmc.github.io/REDCapR/index.html#installation-and-documentation}{GitHub release}
and file a new \href{https://github.com/OuhscBbmc/REDCapR/issues}{issue} if it's still not
recognized.
}

For details of the current implementation,
the decision logic starts about half-way down in the
\href{https://github.com/OuhscBbmc/REDCapR/blob/HEAD/R/redcap-metadata-coltypes.R}{function's source code}

*\emph{Validation does NOT Guarantee Conformity}

If you're coming to REDCap from a database world, this will be unexpected.
A validation type does NOT guarantee that all retrieved values will conform to
complementary the data type.
The validation setting affects only the values entered
\emph{after} the validation was set.

For example, if values like "abcd" where entered in a field for a few months, then
the project manager selected the "integer" validation option, all those
"abcd" values remain untouched.

This is one reason \code{redcap_metadata_coltypes()} prints it suggestions to the console.
It allows the developer to adjust the specifications to match the values
returned by the API.  The the "abcd" scenario, consider (a) changing the type
from \code{col_integer} to \code{col_character}, (b) excluding the trash values,
then (c) in a \code{\link[dplyr:mutate]{dplyr::mutate()}} statement,
use \code{\link[readr:parse_atomic]{readr::parse_integer()}} to cast it to the desired type.
}
\examples{
\dontrun{
uri      <- "https://redcap-dev-2.ouhsc.edu/redcap/api/"

# A simple project
token      <- "9A068C425B1341D69E83064A2D273A70" # simple
col_types  <- redcap_metadata_coltypes(uri, token)
redcap_read_oneshot(uri, token, col_types = col_types)$data

# A longitudinal project
token      <- "DA6F2BB23146BD5A7EA3408C1A44A556" # longitudinal
col_types  <- redcap_metadata_coltypes(uri, token)
redcap_read_oneshot(uri, token, col_types = col_types)$data

# A repeating instruments project
token      <- "64720C527CA236880FBA785C9934F02A" # repeating-instruments-sparse
col_types  <- redcap_metadata_coltypes(uri, token)
redcap_read_oneshot(uri, token, col_types = col_types)$data

# A project with every field type and validation type.
#   Notice it throws a warning that some fields use a comma for a decimal,
#   while other fields use a period/dot as a decimal
token      <- "EB1FD5DDE583364AE605629AB7619397" # validation-types-1
col_types  <- redcap_metadata_coltypes(uri, token)
redcap_read_oneshot(uri, token, col_types = col_types)$data
}
}
\references{
The official documentation can be found on the 'API Help Page'
and 'API Examples' pages on the REDCap wiki (\emph{i.e.},
https://community.projectredcap.org/articles/456/api-documentation.html and
https://community.projectredcap.org/articles/462/api-examples.html).
If you do not have an account for the wiki, please ask your campus REDCap
administrator to send you the static material.
}
\author{
Will Beasley, Philip Chase
}
