#' Mosaic a set of MODIS images
#'
#' \code{modMosaic} merges the MODIS imagery that covers a 
#' region of interest on the same dates.
#'
#' The function mosaics the imagery in the \code{src} folder. The folder can
#' hold GTiff images from several tiles, dates and bands. When only a subset
#' of bands or dates has to be mosaicked, the band names or dates should be
#' provided through the argument \code{bFilter} or \code{dayFilter}. Band
#' names are defined by the letter “b” and the two-digit band number (e.g., 
#' ‘b01’). The dates must be provided as a \code{Date} class object. Once
#' mosaicked, the images can be cropped to fit the \code{extent} (optional).
#' The \code{extent} can be defined in any coordinate reference system, since
#' \code{modMosaic} automatically reproject the extent to match the projection
#' of the image. The outputs will be placed in the \code{AppRoot} directory,
#' under the folder named as \code{out.name}. If no name is provided, the 
#' folder is named “outfile”. To use \code{gutils = TRUE}, a proper installation
#' of `GDAL' is required. This method is faster than the native `R' functions.
#'
#'
#' @param src the path of the folder with the MODIS images in GTiff format.
#' @param AppRoot the directory where the mosaicked images are saved.
#' @param out.name  the name of the folder that stores the outputs. By default,
#' “outfile” is assigned.
#' @param extent an \code{extent}, \code{Raster*}, or \code{Spatial*} object
#' representing the region of interest.
#' @param overwrite logical argument. If \code{TRUE}, overwrites the existing 
#' images with the same name.
#' @param gutils logical argument. If \code{TRUE}, the function uses `GDAL' 
#' utilities for mosaicking.
#' @param verbose logical argument. If \code{TRUE}, the function prints the 
#' running steps and warnings.
#' @param ... arguments for nested functions:
#'  \itemize{
#'   \item \code{pathrow} a \code{list} of vectors with the path and row numbers
#'   of the tiles concerning the region of interest.
#'   \item \code{bFilter} a vector with the bands to be mosaicked. If not
#'   supplied, all bands are mosaicked.
#'   \item \code{dayFilter} a vector with the capturing dates being considered
#'   for mosaicking. If not supplied, all dates are mosaicked.
#' }
#' @examples
#' \dontrun{
#' # load a spatial polygon object of Navarre
#' data(ex.navarre)
#' # main output directory
#' src <- paste0(tempdir(),"/Path_for_downloading_folder")
#' print(src)
#' # download MODIS images
#' modDownload(product = "MOD09GA",
#'             startDate = as.Date("01-01-2018", "%d-%m-%Y"),
#'             endDate = as.Date("03-01-2018", "%d-%m-%Y"),
#'             username = "username",
#'             password = "password",
#'             AppRoot = src,
#'             extract.tif = TRUE,
#'             collection = 6,
#'             extent = ex.navarre)
#' # folder with the MODIS images extracted 
#' src.mod <- file.path(src, "Modis", "MOD09GA")
#' tif.src <- file.path(src.mod, "tif")
#' # mosaic the MODIS images
#' modMosaic(tif.src,
#'           AppRoot = src.mod,
#'           out.name = "Navarre",
#'           gutils = TRUE,
#'           overwrite = TRUE,
#'           extent = ex.navarre)
#' }
modMosaic<-function(src,
                    AppRoot,
                    extent = NULL,
                    out.name = "outfile",
                    verbose = FALSE,
                    gutils = FALSE,
                    overwrite = FALSE,
                    ...){
  arg<-list(...)
  src<-pathWinLx(src)
  AppRoot<-pathWinLx(AppRoot)
  #read all folder names to get all the days
  imgFolders<-list.files(src,full.names = TRUE)
  #remove folders
  #imgFolders<-imgFolders[nchar(basename(imgFolders))==41]
  if(length(imgFolders)==0)stop(paste0("There is no images in ",src," path."))
  dates<-unique(modGetDates(imgFolders))
  bpath<-file.path(AppRoot,out.name)

  #filter dates
  if("dayFilter"%in%names(arg)){
    dates<-dates[dates%in%arg$dayFilter]
  }

  for(d in 1:length(dates)){
    #filter the images to one day
    dayImg<-imgFolders[modGetDates(imgFolders)%in%dates[d]]
    if(length(dayImg)<1){
      if(verbose)
        warning(paste0("No tiles for date ",dates[d]))
      next #breaks one iteration only
    }
    #filter the images by pathrow
    if("pathrow"%in%names(arg)){
      prstr<-c()
      for(pr in arg$pathrow){
        prstr<-c(prstr,paste0("h",sprintf("%02d",pr[1]),"v",sprintf("%02d",pr[2])))
      }
      dayImg<-dayImg[modGetPathRow(dayImg)%in%prstr]
      stopifnot(length(dayImg)>0)
    }

    flist<-list.files(dayImg,recursive=TRUE,full.names=TRUE,pattern="\\.tif$")
    #filter the images by data type
    if("bFilter"%in%names(arg)){
      flist<-flist[Reduce("|", lapply(arg$bFilter,grepl,flist))]
    }
    dtype<-flist
    for(x in basename(dayImg)){
      dtype<-gsub(x,"",basename(dtype))
    }
    dtype<-unique(dtype)


    if(gutils){
      message(paste0("Merging and constraining the extent of the image at ",dates[d]," using gdalUtils library"))
    }else{
      message(paste0("Merging and cutting for day ",dates[d]," using raster library"))
    }
    AppRoot<-file.path(bpath,format(dates[d],"%Y%j"))
    dir.create(AppRoot,recursive = TRUE,showWarnings = verbose)
    for(dt in 1:length(dtype)){
      out.file.path<-file.path(AppRoot,paste0(out.name,"_",format(dates[d],"%Y%j"),"_",dtype[dt]))
      if((!file.exists(out.file.path))|overwrite){
        typechunks<-flist[grepl(dtype[dt],flist)]
        if(length(typechunks)==0)next
        if(!gutils){
          #mosaic with native R libraries
          typechunks<-lapply(typechunks,raster)
          tryCatch(
            {
              img<- genMosaicList(typechunks,verbose)
            },
            error=function(cond) {
              if(any(grepl("different CRS",cond))){
                message(paste0("Different CRS when mosaicing tiles!\nProjecting to the same CRS..."))
                typechunks<-lapply(typechunks,projectRaster,to=typechunks[[1]])
                img<- genMosaicList(typechunks,verbose)
              }else if(any(grepl("subscript out of bounds",cond))){
                warning("Tile for ",dt," not found! Check ",d," date image forders for extrating errors.")
              }else{
                stop(cond)
              }
            })
          if(!is.null(extent)){
            if(class(extent)!="Extent")
              extent<-spTransform(extent,crs(img))
            img<-crop(img,extent)
          }
          writeRaster(img,out.file.path,overwrite=overwrite)
        }else{
          #mosaic with gdalutils no support cutline
          if(is.null(extent)){
            mosaic_rasters(typechunks,out.file.path,overwrite=overwrite)
          }else{
            ext<-extent(extent)
            temp<-file.path(AppRoot,paste0(out.name,"_",format(dates[d],"%Y%j"),dtype[dt],"_temp.tif"))
            mosaic_rasters(typechunks,
                           dst_dataset=temp,
                           overwrite=TRUE)
            gdalwarp(srcfile=temp,
                     dstfile=out.file.path,
                     te=c(ext@xmin,ext@ymin,ext@xmax,ext@ymax),
                     te_srs=proj4string(extent),
                     overwrite=overwrite)
            file.remove(temp)
          }
        }
      }else{
        if(verbose){
          warning("File exists! not mergin...")
        }
      }
    }
      
  }
  message(paste0("Region saved in HDD.\nFiles in: ",bpath))
}
