% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rocit.R
\name{rocit}
\alias{rocit}
\title{ROC Analysis of Binary Classifier}
\usage{
rocit(score, class, negref = NULL, method = "empirical", step = FALSE)
}
\arguments{
\item{score}{An numeric array of diagnostic score.}

\item{class}{An array of equal length of score,
containing the class of the observations.}

\item{negref}{The reference value, same as the
\code{reference} in \code{\link{convertclass}}.
Depending on the class of \code{x},
it can be numeric or character type. If specified, this value
is converted to 0 and other is converted to 1. If NULL, reference is
set alphabetically.}

\item{method}{The method of estimating ROC curve. Currently supports
\code{"empirical"}, \code{"binormal"} and \code{"nonparametric"}.
Pattern matching allowed thorough \code{\link[base:grep]{grep}}.}

\item{step}{Logical, default in \code{FALSE}. Only applicable for
\code{empirical} method and ignored for others. Indicates
whether only horizontal and vertical steps should be used
to produce the ROC curve. See "Details".}
}
\value{
A list of class \code{"rocit"}, having following elements:
\item{method}{The method applied to estimate ROC curve.}
\item{pos_count}{Number of positive responses.}
\item{neg_count}{Number of negative responses.}
\item{pos_D}{Array of diagnostic scores in positive responses.}
\item{neg_D}{Array of diagnostic scores in negative responses.}
\item{AUC}{Area under curve. See "Details".}
\item{Cutoff}{Array of cutoff values at which the
true positive rates and  false positive rates
 are evaluated. Applicable for \code{"empirical"} and
 \code{"nonparametric"}.}
\item{param}{Maximum likelihood estimates of \eqn{\mu} and \eqn{\sigma} of
the diagnostic score in two groups. Applicable for \code{"binormal"}.}
\item{TPR}{Array of true positive rates (or sensitivities or recalls),
evaluated at the cutoff values.}
\item{FPR}{Array of false positive rates (or 1-specificity),
evaluated at the cutoff values.}
}
\description{
\code{rocit} is the main function of \pkg{ROCit} package.
With the diagnostic score and the class of each observation,
it calculates true positive rate (sensitivity) and
false positive rate (1-Specificity) at convenient cutoff values
to construct ROC curve. The function returns \code{"rocit"} object,
which can be passed as arguments for other S3 methods.
}
\details{
ROC curve is defined as the set of ordered pairs,
\eqn{(FPR(c), TPR(c))}, where, \eqn{-\infty < c < \infty},
where, \eqn{FPR(c) = P(D \ge c | Y = 0)} and \eqn{FPR(c) = P(D \ge c | Y = 1)}
at cutoff \eqn{c}.
Alternately, it can be defined as:
\deqn{y(x) = 1 - G[F^{-1}(1-x)],  0 \le x \le 1}
where \eqn{F} and \eqn{G} are the cumulative density functions of the
diagnostic score in negative and positive responses respectively.
\code{rocit} evaluates TPR and FPR values at convenient cutoffs.

    As the name implies, empirical TPR and FPR values are evaluated
 for \code{method = "empirical"}. For \code{"binormal"}, the distribution
 of diagnostic are assumed to be normal and maximum likelihood parameters
 are estimated. If \code{method = "nonparametric"}, then kernel density
 estimates (using \code{\link[stats:density]{density}}) are applied with
 following bandwidth:
 \itemize{
 \item \eqn{h_Y = 0.9 * min(\sigma_Y, IQR(D_Y)/1.34)/((n_Y)^{(1/5)})}
 \item \eqn{h_{\bar{Y}} = 0.9 * min(\sigma_{\bar{Y}},
 IQR(D_{\bar{Y}})/1.34)/((n_{\bar{Y}})^{(1/5)})}
 }
 as described in Zou et al. From the kernel estimates of PDFs, CDFs are
 estimated using trapezoidal rule.

For \code{"empirical"} ROC, the algorithm firt rank orders the
data and calculates TPR and FPR by treating all predicted
up to certain level as positive. If \code{step} is \code{TRUE},
then the ROC curve is generated based on all the calculated
\{FPR, TPR\} pairs regardless of tie in the data. If \code{step} is
\code{FALSE}, then the ROC curve follows a diagonal path for the ties.


For \code{"empirical"} ROC, trapezoidal rule is
applied to estimate area under curve (AUC). For \code{"binormal"}, AUC is estimated by
\eqn{\Phi(A/\sqrt(1 + B^2)}, where \eqn{A} and \eqn{B} are functions
of mean and variance of the diagnostic in two groups.
For \code{"nonparametric"}, AUC is estimated as
by
\deqn{\frac{1}{n_Yn_{\bar{Y}}}
\sum_{i=1}^{n_{\bar{Y}}}
\sum_{j=1}^{n_{Y}}
\Phi(
  \frac{D_{Y_j}-D_{{\bar{Y}}_i}}{\sqrt{h_Y^2+h_{\bar{Y}}^2}}
 )}
}
\note{
The algorithm is designed for complete cases. If NA(s) found in
either \code{score} or \code{class}, then removed.
}
\examples{
# ---------------------
data("Diabetes")
roc_empirical <- rocit(score = Diabetes$chol, class = Diabetes$dtest,
                       negref = "-") # default method empirical
roc_binormal <- rocit(score = Diabetes$chol, class = Diabetes$dtest,
                     negref = "-", method = "bin")

# ---------------------
summary(roc_empirical)
summary(roc_binormal)

# ---------------------
plot(roc_empirical)
plot(roc_binormal, col = c("#00BA37", "#F8766D"),
       legend = FALSE, YIndex = FALSE)


}
\references{
Pepe, Margaret Sullivan. \emph{The statistical evaluation
of medical tests for classification and prediction.} Medicine, 2003.

Zou, Kelly H., W. J. Hall, and David E. Shapiro.
"Smooth non-parametric receiver operating characteristic (ROC)
 curves for continuous diagnostic tests." \emph{Statistics in medicine}
 16, no. 19 (1997): 2143-2156.
}
\seealso{
\code{\link{ciROC}}, \code{\link{ciAUC}}, \code{\link{plot.rocit}},
\code{\link{gainstable}}, \code{\link{ksplot}}
}
