% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Rase.R
\name{Rase}
\alias{Rase}
\title{Construct the random subspace ensemble classifier.}
\usage{
Rase(
  xtrain,
  ytrain,
  xval = NULL,
  yval = NULL,
  B1 = 200,
  B2 = 500,
  D = NULL,
  dist = NULL,
  base = c("lda", "qda", "knn", "logistic", "tree", "svm", "randomforest", "gamma"),
  criterion = NULL,
  ranking = TRUE,
  k = c(3, 5, 7, 9, 11),
  cores = 1,
  seed = NULL,
  iteration = 0,
  cutoff = TRUE,
  cv = 10,
  scale = FALSE,
  C0 = 0.1,
  kl.k = NULL,
  ...
)
}
\arguments{
\item{xtrain}{n * p observation matrix. n observations, p features.}

\item{ytrain}{n 0/1 observatons.}

\item{xval}{observation matrix for validation. Default = \code{NULL}. Useful only when \code{criterion} = 'validation'.}

\item{yval}{0/1 observation for validation. Default = \code{NULL}. Useful only when \code{criterion} = 'validation'.}

\item{B1}{the number of weak learners. Default = 200.}

\item{B2}{the number of subspace candidates generated for each weak learner. Default = 500.}

\item{D}{the maximal subspace size when generating random subspaces from the uniform distribution. Default = \code{NULL}, which is \eqn{min(\sqrt n0, \sqrt n1, p)} when \code{base} = 'lda' and is \eqn{min(\sqrt n, p)} otherwise.}

\item{dist}{the distribution for features when generating random subspaces. Default = \code{NULL}, which represents the uniform distribution. First generate an integer \eqn{d} from \eqn{1,...,D} uniformly, then uniformly generate a subset with cardinality \eqn{d}.}

\item{base}{the type of base classifier. Default = 'lda'.
\itemize{
\item lda: linear discriminant analysis. \code{\link[MASS]{lda}} in \code{MASS} package.
\item qda: quadratic discriminant analysis. \code{\link[MASS]{qda}} in \code{MASS} package.
\item knn: k-nearest neighbor. \code{\link[class]{knn}}, \code{\link[class]{knn.cv}} in \code{class} package and \code{\link[caret]{knn3}} in \code{caret} package.
\item logistic: logistic regression. \code{\link[glmnet]{glmnet}} in \code{glmnet} package.
\item tree: decision tree. \code{\link[rpart]{rpart}} in \code{rpart} package.
\item svm: support vector machine. \code{\link[e1071]{svm}} in \code{e1071} package.
\item randomforest: random forest. \code{\link[randomForest]{randomForest}} in \code{randomForest} package.
\item gamma: Bayesian classifier for multivariate gamma distribution with independent marginals.
}}

\item{criterion}{the criterion to choose the best subspace for each weak learner. Default = 'ric' when \code{base} = 'lda', 'qda', 'gamma'; default = 'ebic' and set \code{gam} = 0 when \code{base} = 'logistic'; default = 'loo' when \code{base} = 'knn'; default = 'training' when \code{base} = 'tree', 'svm', 'randomforest'.
\itemize{
\item ric: minimizing ratio information criterion with parametric estimation (Tian, Y. and Feng, Y., 2020). Available when \code{base} = 'lda', 'qda', 'gamma' or 'logistic'.
\item nric: minimizing ratio information criterion with non-parametric estimation (Tian, Y. and Feng, Y., 2020; Wang, Q., Kulkarni, S.R. and Verdú, S., 2009). Available when \code{base} = 'lda', 'qda', 'gamma' or 'logistic'.
\item training: minimizing training error. Not available when \code{base} = 'knn'.
\item loo: minimizing leave-one-out error. Only available when  \code{base} = 'knn'.
\item validation: minimizing validation error based on the validation data. Available for all base classifiers.
\item cv: minimizing k-fold cross-validation error. k equals to the value of \code{cv}. Default = 10. Not available when \code{base} = 'gamma'.
\item ebic: minimizing extended Bayesian information criterion (Chen, J. and Chen, Z., 2008; 2012). Need to assign value for \code{gam}. When \code{gam} = 0, it denotes the classical BIC. Available when \code{base} = 'lda', 'qda' or 'logistic'.

EBIC = -2 * log-likelihood + |S| * log(n) + 2 * |S| * gam * log(p).
}}

\item{ranking}{whether the function outputs the selected percentage of each feature in B1 subspaces. Logistic, default = TRUE.}

\item{k}{the number of nearest neightbors considered when \code{base} = 'knn'. Only useful when \code{base} = 'knn'.}

\item{cores}{the number of cores used for parallel computing. Default = 1.}

\item{seed}{the random seed assigned at the start of the algorithm, which can be a real number or \code{NULL}. Default = \code{NULL}, in which case no random seed will be set.}

\item{iteration}{the number of iterations. Default = 0.}

\item{cutoff}{whether to use the empirically optimal threshold. Logistic, default = TRUE. If it is FALSE, the threshold will be set as 0.5.}

\item{cv}{the number of cross-validations used. Default = 10. Only useful when \code{criterion} = 'cv'.}

\item{scale}{whether to normalize the data. Logistic, default = FALSE.}

\item{C0}{the threshold used to adjust the sampling probabilities of features when \code{iteration} > 0. Default = 0.1.}

\item{kl.k}{the number of nearest neighbors used to estimate KL divergences when \code{criterion} = 'nric'. 2-dimensional vector. Default = \code{NULL}, in which case it will be set as \eqn{\sqrt n0, \sqrt n1}.}

\item{...}{additional arguments.}
}
\value{
An object with S3 class \code{'RaSE'}.
\item{marginal}{the marginal probability for each class.}
\item{base}{the type of base classifier.}
\item{criterion}{the criterion to choose the best subspace for each weak learner.}
\item{B1}{the number of weak learners.}
\item{B2}{the number of subspace candidates generated for each weak learner.}
\item{iteration}{the number of iterations.}
\item{fit.list}{sequence of B1 fitted base classifiers.}
\item{cutoff}{the empirically optimal threshold.}
\item{subspace}{sequence of subspaces correponding to B1 weak learners.}
\item{ranking}{the selected percentage of each feature in B1 subspaces.}
\item{scale}{a list of scaling parameters, including the scaling center and the scale parameter for each feature. Equals to \code{NULL} when the data is not scaled in \code{RaSE} model fitting.}
\item{C0}{the threshold used to adjust the sampling probabilities of features when \code{iteration} > 0.}
}
\description{
\code{Rase} is a novel model-free ensemble classification framework to solve the sparse classification problem. In RaSE algorithm, for each of the B1 weak learners, B2 random subspaces are generated and the optimal one is chosen to train the model on the basis of some criterion.
}
\examples{
set.seed(0, kind = "L'Ecuyer-CMRG")
train.data <- RaModel(1, n = 100, p = 50)
test.data <- RaModel(1, n = 100, p = 50)
xtrain <- train.data$x
ytrain <- train.data$y
xtest <- test.data$x
ytest <- test.data$y

# test RaSE classifier with LDA base classifier
fit <- Rase(xtrain, ytrain, B1 = 100, B2 = 50, iteration = 0, base = 'lda',
cores = 2, criterion = 'ric')
mean(predict(fit, xtest) != ytest)

\dontrun{
# test RaSE classifier with LDA base classifier and 1 iteration round
fit <- Rase(xtrain, ytrain, B1 = 100, B2 = 50, iteration = 1, base = 'lda',
cores = 2, criterion = 'ric')
mean(predict(fit, xtest) != ytest)

# test RaSE classifier with QDA base classifier and 1 iteration round
fit <- Rase(xtrain, ytrain, B1 = 100, B2 = 50, iteration = 1, base = 'qda',
cores = 2, criterion = 'ric')
mean(predict(fit, xtest) != ytest)

# test RaSE classifier with knn base classifier
fit <- Rase(xtrain, ytrain, B1 = 100, B2 = 50, iteration = 0, base = 'knn',
cores = 2, criterion = 'loo')
mean(predict(fit, xtest) != ytest)

# test RaSE classifier with logistic regression base classifier
fit <- Rase(xtrain, ytrain, B1 = 100, B2 = 50, iteration = 0, base = 'logistic',
cores = 2, criterion = 'ebic', gam = 0)
mean(predict(fit, xtest) != ytest)

# test RaSE classifier with svm base classifier
fit <- Rase(xtrain, ytrain, B1 = 100, B2 = 50, iteration = 0, base = 'svm',
cores = 2, criterion = 'training')
mean(predict(fit, xtest) != ytest)

# test RaSE classifier with random forest base classifier
fit <- Rase(xtrain, ytrain, B1 = 20, B2 = 10, iteration = 0, base = 'randomforest',
cores = 2, criterion = 'cv', cv = 3)
mean(predict(fit, xtest) != ytest)
}
}
\references{
Tian, Y. and Feng, Y., 2020. RaSE: Random subspace ensemble classification. arXiv preprint arXiv:2006.08855.

Chen, J. and Chen, Z., 2008. Extended Bayesian information criteria for model selection with large model spaces. Biometrika, 95(3), pp.759-771.

Chen, J. and Chen, Z., 2012. Extended BIC for small-n-large-P sparse GLM. Statistica Sinica, pp.555-574.

Wang, Q., Kulkarni, S.R. and Verdú, S., 2009. Divergence estimation for multidimensional densities via $ k $-nearest-neighbor distances. IEEE Transactions on Information Theory, 55(5), pp.2392-2405.#' @examples
}
\seealso{
\code{\link{predict.RaSE}}, \code{\link{RaModel}}, \code{\link{print.RaSE}}, \code{\link{RaPlot}}.
}
