% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/perturb_vr.R
\name{perturb_vr}
\alias{perturb_vr}
\title{Perturbation analysis of vital rates in a matrix population model}
\usage{
perturb_vr(
  matU,
  matF,
  matC = NULL,
  pert = 1e-06,
  type = "sensitivity",
  demog_stat = "lambda",
  ...
)
}
\arguments{
\item{matU}{The survival component of a matrix population model (i.e., a
square projection matrix reflecting survival-related transitions; e.g., 
progression, stasis, and retrogression).}

\item{matF}{The sexual component of a matrix population model (i.e., a square
projection matrix reflecting transitions due to sexual reproduction).}

\item{matC}{The clonal component of a matrix population model (i.e., a square
projection matrix reflecting transitions due to clonal reproduction).
Defaults to \code{NULL}, indicating no clonal reproduction (i.e., 
\code{matC} is a matrix of zeros).}

\item{pert}{Magnitude of the perturbation. Defaults to \code{1e-6}.}

\item{type}{Whether to return \code{sensitivity} or \code{elasticity} values. Defaults
to \code{sensitivity}.}

\item{demog_stat}{The demographic statistic to be used, as in "the
sensitivity/elasticity of \code{demog_stat} to vital rate perturbations."
Defaults to the per-capita population growth rate at equilibrium
(\eqn{\lambda}). Also accepts a user-supplied function that performs a
calculation on a projection matrix and returns a single numeric value.}

\item{...}{Additional arguments passed to the function \code{demog_stat}.}
}
\value{
A list with 5 elements:
\item{survival}{sensitivity or elasticity of \code{demog_stat} to survival}
\item{growth}{sensitivity or elasticity of \code{demog_stat} to growth}
\item{shrinkage}{sensitivity or elasticity of \code{demog_stat} to shrinkage}
\item{fecundity}{sensitivity or elasticity of \code{demog_stat} to sexual
fecundity}
\item{clonality}{sensitivity or elasticity of \code{demog_stat} to clonality}
}
\description{
Perturbs lower-level vital rates within a matrix population model and
measures the response (sensitivity or elasticity) of the per-capita 
population growth rate at equilibrium (\eqn{\lambda}), or, with a 
user-supplied function, any other demographic statistic.

These decompositions assume that all transition rates are products of a
stage-specific survival term (column sums of \code{matU}) and a lower level
vital rate that is conditional on survival (growth, shrinkage, stasis,
dormancy, or reproduction). Reproductive vital rates that are not conditional
on survival (i.e., within a stage class from which there is no survival) are
also allowed.
}
\examples{
matU <- rbind(c(0.1,   0,   0,   0),
              c(0.5, 0.2, 0.1,   0),
              c(  0, 0.3, 0.3, 0.1),
              c(  0,   0, 0.5, 0.6))

matF <- rbind(c(  0,   0, 1.1, 1.6),
              c(  0,   0, 0.8, 0.4),
              c(  0,   0,   0,   0),
              c(  0,   0,   0,   0))

perturb_vr(matU, matF)

# use elasticities rather than sensitivities
perturb_vr(matU, matF, type = "elasticity")

# use a larger perturbation than the default
perturb_vr(matU, matF, pert = 0.01)

# calculate the sensitivity/elasticity of the damping ratio to vital rate
#  perturbations
damping <- function(matA) {  # define function for damping ratio
  eig <- eigen(matA)$values
  dm <- rle(Mod(eig))$values
  return(dm[1] / dm[2])
}

perturb_vr(matU, matF, demog_stat = "damping")

}
\seealso{
Other perturbation analysis: 
\code{\link{perturb_matrix}()},
\code{\link{perturb_stochastic}()},
\code{\link{perturb_trans}()},
\code{\link{pop_vectors}()}
}
\author{
Rob Salguero-Gomez <rob.salguero@zoo.ox.ac.uk>

Patrick Barks <patrick.barks@gmail.com>
}
\concept{perturbation analysis}
