/*
 * This file is part of ReporteRs
 * Copyright (c) 2014, David Gohel All rights reserved.
 * This program is licensed under the GNU GENERAL PUBLIC LICENSE V3.
 * You may obtain a copy of the License at :
 * http://www.gnu.org/licenses/gpl.html
 */

package org.lysis.reporters.pptx4r.elements.template;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;

import org.docx4j.dml.CTTransform2D;
import org.lysis.reporters.tools.Debug;
import org.pptx4j.pml.Shape;

public class LayoutDescription {
	
	public static int TITLE = 0;
	public static int FOOTER = 1;
	public static int SLIDENUMBER = 2;
	public static int DATE = 3;
	public static int SUBTITLE = 4;
	public static int CRTTITLE = 5;
	public static int BODY = 6;
	
	private HashMap<Integer, ShapeDescription> listBodyShapeDescription;
	private HashMap<Integer, ShapeDescription> listHeaderShapeDescription;
	
	private String label;
	private String xmlFilename;
	
	private int max_content;
	
	private boolean has_title = false;
	private boolean has_footer = false;
	private boolean has_slidenumber = false;
	private boolean has_date = false;
	private boolean has_subtitle = false;
	private boolean has_crttitle = false;
	
	private long idx_title ;
	private long idx_footer ;
	private long idx_slidenumber;
	private long idx_date;
	private long idx_subtitle;
	private long idx_crttitle;

	public LayoutDescription(List<Object> slidesSet){
		max_content = 0;
		listBodyShapeDescription = new HashMap<Integer, ShapeDescription>();
		listHeaderShapeDescription = new HashMap<Integer, ShapeDescription>();
		
		Iterator<Object> itr = slidesSet.iterator();
		while(itr.hasNext()) {
			Object element = itr.next();
			if( element.getClass().equals(org.pptx4j.pml.Shape.class) ) {
				Shape s = (Shape)element;
				ShapeDescription sd = new ShapeDescription( s );
				
				if( sd.getShapeType().equals("title") ) {
					has_title = true;
					idx_title = sd.getIdx();
					listHeaderShapeDescription.put(MasterLayoutProperties.TITLE, sd);
				} else if( sd.getShapeType().equals("dt") ) {
					has_date = true;
					idx_date = sd.getIdx();
					listHeaderShapeDescription.put(MasterLayoutProperties.DATE, sd);
				} else if( sd.getShapeType().equals("ftr") ) {
					has_footer = true;
					idx_footer = sd.getIdx();
					listHeaderShapeDescription.put(MasterLayoutProperties.FOOTER, sd);
				} else if( sd.getShapeType().equals("sldNum") ) {
					has_slidenumber = true;
					idx_slidenumber = sd.getIdx();
					listHeaderShapeDescription.put(MasterLayoutProperties.SLIDENUMBER, sd);
				} else if( sd.getShapeType().equals("subTitle") ) {
					has_subtitle = true;
					idx_subtitle = sd.getIdx();
					listHeaderShapeDescription.put(MasterLayoutProperties.SUBTITLE, sd);
				} else if( sd.getShapeType().equals("ctrTitle") ) {
					has_crttitle = true;
					idx_crttitle = sd.getIdx();
					listHeaderShapeDescription.put(MasterLayoutProperties.CRTTITLE, sd);
				} else if( sd.getShapeType().equals("body") 
						|| sd.getShapeType().equals("tbl")
						|| sd.getShapeType().equals("obj") 
						|| sd.getShapeType().equals("pic") ) {
					listBodyShapeDescription.put( max_content, sd );
					max_content++;
					if( Debug.doc_explorer ) System.err.println("\t* shape (has xfrm : " + sd.hasXfrm() + ") of type: " + sd.getShapeType() );
				}
			}
	    }
	}
	

	public void fixMissingXfrm( MasterLayoutProperties masterLayout ){
		
		if( hasTitle() && !listHeaderShapeDescription.get(TITLE).hasXfrm() ){
			if( Debug.doc_explorer ) System.err.println("fixMissingXfrm: hasTitle and no xfrm" );
			listHeaderShapeDescription.get(TITLE).setXfrm(masterLayout.getShapeDescription(TITLE).getXfrm());
		}
		if( hasSubtitle() && !listHeaderShapeDescription.get(SUBTITLE).hasXfrm() ){
			if( Debug.doc_explorer ) System.err.println("fixMissingXfrm: hasSubTitle and no xfrm" );
			listHeaderShapeDescription.get(SUBTITLE).setXfrm(masterLayout.getShapeDescription(SUBTITLE).getXfrm());
		}
		if( hasFooter() && !listHeaderShapeDescription.get(FOOTER).hasXfrm() ){
			if( Debug.doc_explorer ) System.err.println("fixMissingXfrm: hasFooter and no xfrm" );
			listHeaderShapeDescription.get(FOOTER).setXfrm(masterLayout.getShapeDescription(FOOTER).getXfrm());
		}
		if( hasSlideNumber() && !listHeaderShapeDescription.get(SLIDENUMBER).hasXfrm() ){
			if( Debug.doc_explorer ) System.err.println("fixMissingXfrm: hasSlideNumber and no xfrm" );
			listHeaderShapeDescription.get(SLIDENUMBER).setXfrm(masterLayout.getShapeDescription(SLIDENUMBER).getXfrm());
		}
		if( hasDate() && !listHeaderShapeDescription.get(DATE).hasXfrm() ){
			if( Debug.doc_explorer ) System.err.println("fixMissingXfrm: hasDate and no xfrm" );
			listHeaderShapeDescription.get(DATE).setXfrm(masterLayout.getShapeDescription(DATE).getXfrm());
		}
		if( hasCrttitle() && !listHeaderShapeDescription.get(CRTTITLE).hasXfrm() ){
			if( Debug.doc_explorer ) System.err.println("fixMissingXfrm: hasCrtTitle and no xfrm" );
			listHeaderShapeDescription.get(CRTTITLE).setXfrm(masterLayout.getShapeDescription(CRTTITLE).getXfrm());
		}
		if( max_content == 1 && !listBodyShapeDescription.get(0).hasXfrm() ){
			if( Debug.doc_explorer ) System.err.println("fixMissingXfrm: 1 body and no xfrm" );
			listBodyShapeDescription.get(0).setXfrm(masterLayout.getShapeDescription(BODY).getXfrm());
		}
	}
	
	
	public void setLabel(String _label ){
		label = _label;
	}
	
	public void setXMLFilename(String _key ){
		xmlFilename = _key;
	}
	
	public ShapeDescription getContent( int id ){
		return listBodyShapeDescription.get(id);
	}
	


	public int[] getHeaderDimensions() {
		int num_headers = getNumberOfHeaders();
		int[] data = new int[ 5 * num_headers ];
		int index = 0;
		
		Iterator<Entry<Integer, ShapeDescription>> partIterator = listHeaderShapeDescription.entrySet().iterator();
		
		while (partIterator.hasNext()) {
			Entry<Integer, ShapeDescription> en =partIterator.next(); 
			int[] temp = getHeaderDimensions(en.getKey());
			data[index] = en.getKey();index++;
			data[index] = temp[0];index++;
			data[index] = temp[1];index++;
			data[index] = temp[2];index++;
			data[index] = temp[3];index++;
		}

		return data;
	}
	
	public int[] getHeaderDimensions( int i) {
		int[] data = new int[ 4 ];
		CTTransform2D xfrm ;
		
		if( i == TITLE && !hasTitle() ) return data;
		else if( i == FOOTER && !hasFooter() ) return data;
		else if( i == SLIDENUMBER && !hasSlideNumber() ) return data;
		else if( i == DATE && !hasDate() ) return data;
		else if( i == SUBTITLE && !hasSubtitle() ) return data;
		else if( i == CRTTITLE && !hasCrttitle() ) return data;
		
		xfrm = listHeaderShapeDescription.get(i).getXfrm();

		Long offx = new Long(xfrm.getOff().getX());
		Long offy = new Long(xfrm.getOff().getY());
		Long extx = new Long(xfrm.getExt().getCx());
		Long exty = new Long(xfrm.getExt().getCy());
		
		data[0] = offx.intValue();
		data[1] = offy.intValue();
		data[2] = extx.intValue();
		data[3] = exty.intValue();

		return data;
	}

	public int[] getContentDimensions() {
		int[] data = new int[ 4 * max_content ];
		CTTransform2D xfrm ;
		int index = 0;
		for(int id = 0 ; id < max_content ; id ++ ){
			xfrm = listBodyShapeDescription.get(id).getXfrm();

			Long offx = new Long(xfrm.getOff().getX());
			Long offy = new Long(xfrm.getOff().getY());
			Long extx = new Long(xfrm.getExt().getCx());
			Long exty = new Long(xfrm.getExt().getCy());
			data[index] = offx.intValue();index++;
			data[index] = offy.intValue();index++;
			data[index] = extx.intValue();index++;
			data[index] = exty.intValue();index++;
		}

		return data;
	}
	public int[] getContentDimensions(int id) {
		int[] data = new int[4];
		CTTransform2D xfrm ;

		xfrm = listBodyShapeDescription.get(id).getXfrm();

		Long offx = new Long(xfrm.getOff().getX());
		Long offy = new Long(xfrm.getOff().getY());
		Long extx = new Long(xfrm.getExt().getCx());
		Long exty = new Long(xfrm.getExt().getCy());
		data[0] = offx.intValue();
		data[1] = offy.intValue();
		data[2] = extx.intValue();
		data[3] = exty.intValue();
 
		return data;
	}
	
	public String getLabel( ){
		return label;
	}
	
	public String getXMLFilename( ){
		return xmlFilename;
	}
	
	public HashMap<Integer, Boolean> getIsFilledHeaderModel(){
		HashMap<Integer, Boolean> out = new HashMap<Integer, Boolean>();
		
		if( hasTitle() ){
			out.put(MasterLayoutProperties.TITLE, false);
		}
		if( hasDate() ){
			out.put(MasterLayoutProperties.DATE, false);
		}
		if( hasFooter() ){
			out.put(MasterLayoutProperties.FOOTER, false);
		}
		if( hasSlideNumber() ){
			out.put(MasterLayoutProperties.SLIDENUMBER, false);
		}
		if( hasCrttitle() ){
			out.put(MasterLayoutProperties.CRTTITLE, false);
		}
		if( hasSubtitle() ){
			out.put(MasterLayoutProperties.SUBTITLE, false);
		}
	
		return out;
	}

	
	public int getContentSize(){
		return max_content;
	}
	
	public boolean hasTitle() {
		return has_title;
	}

	public boolean hasFooter() {
		return has_footer;
	}

	public boolean hasSlideNumber() {
		return has_slidenumber;
	}

	public boolean hasDate() {
		return has_date;
	}

	public boolean hasSubtitle() {
		return has_subtitle;
	}

	public boolean hasCrttitle() {
		return has_crttitle;
	}

	
	private int getNumberOfHeaders(){
		int num_headers = 0;
		if( hasTitle() ){
			num_headers++;
		}
		if( hasDate() ){
			num_headers++;
		}
		if( hasFooter() ){
			num_headers++;
		}
		if( hasSlideNumber() ){
			num_headers++;
		}
		if( hasCrttitle() ){
			num_headers++;
		}
		if( hasSubtitle() ){
			num_headers++;
		}
		return num_headers;
	}
	

	
	public long getIdx_title() {
		return idx_title;
	}

	public long getIdx_footer() {
		return idx_footer;
	}

	public long getIdx_slidenumber() {
		return idx_slidenumber;
	}

	public long getIdx_date() {
		return idx_date;
	}

	public long getIdx_subtitle() {
		return idx_subtitle;
	}

	public long getIdx_crttitle() {
		return idx_crttitle;
	}
}
