#' MaxEnt Thresholds
#'
#' Returns the value of the thresholds generated by the MaxEnt software.
#'
#' @param model \linkS4class{SDMmodel} object trained using the "Maxent" method.
#'
#' @return data.frame with the thresholds.
#' @export
#'
#' @author Sergio Vignali
#'
#' @seealso \code{\link{maxentVarImp}}
#'
#' @examples
#' \donttest{
#' # Acquire environmental variables
#' files <- list.files(path = file.path(system.file(package = "dismo"), "ex"),
#'                     pattern = "grd", full.names = TRUE)
#' predictors <- raster::stack(files)
#'
#' # Prepare presence and background locations
#' p_coords <- virtualSp$presence
#' bg_coords <- virtualSp$background
#'
#' # Create SWD object
#' data <- prepareSWD(species = "Virtual species", p = p_coords, a = bg_coords,
#'                    env = predictors, categorical = "biome")
#'
#' # Train a Maxent model
#' model <- train(method = "Maxent", data = data, fc = "l")
#' maxentTh(model)
#' }
maxentTh <- function(model) {

  if (class(model@model) != "Maxent")
    stop("model must be an SDMmodel object trained using the 'Maxent' method!")

  thresholds <- grep(".threshold", rownames(model@model@results), value = TRUE)
  thresholds <- grep("cumulative.threshold", thresholds, value = TRUE,
                     invert = TRUE)
  labels <- gsub(".threshold", "", thresholds)
  df <- data.frame(threshold = model@model@results[thresholds, ])
  rownames(df) <- labels
  return(df)
}
