% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sa.aco.R
\name{sa.aco}
\alias{sa.aco}
\title{Sensitivity Analysis for Structural Equation Modeling Using ACO}
\usage{
sa.aco(
  data = NULL,
  sample.cov,
  sample.nobs,
  model,
  sens.model,
  n.of.sens.pars = NULL,
  opt.fun,
  d = NULL,
  paths = NULL,
  verbose = TRUE,
  max.value = Inf,
  max.iter = 1000,
  e = 1e-10,
  n.of.ants = 10,
  k = 50,
  q = 1e-04,
  sig.level = 0.05,
  xi = 0.5,
  seed = NULL
)
}
\arguments{
\item{data}{The data set used for analysis.}

\item{sample.cov}{covariance matrix}

\item{sample.nobs}{Number of observations for covariance matrix}

\item{model}{The analytic model of interest set up as a lavaan format.}

\item{sens.model}{Sensitivity analysis model template for
structural equation modeling
with a phantom variable. This is the model of interest
with phantom variable and sensitivity parameters added.
See examples provided.}

\item{n.of.sens.pars}{number of sensitivity parameters
added in the sens.model.}

\item{opt.fun}{Customized or preset optimization function.
    The argument can be customized as a function, e.g., opt.fun =
    quote(new.par$pvalue[paths]-old.par$pvalue[paths]), where new.par and old.par
    are the parameter estimates from the sensitivity analysis and analytic models,
    respectively.
    When opt.fun is
    1, the optimization function is the average departure of new estimate
    from the old estimate divided by the old estimate
    y <-  mean(abs(new.par$est[paths] -
    old.par$est[paths]))/mean(abs(old.par$est[paths])); When opt.fun is
    2, the optimization function is the standard deviation of deviance
    divided by the old estimate
    y <-  stats::sd(new.par$est[paths] - old.par$est[paths])/
    mean(abs(old.par$est[paths]));
    When opt.fun is 3, the optimization function is the average
    p value changed or
    y <-  mean(abs(new.par$pvalue[paths] - old.par$pvalue[paths]))

    When opt.fun is 4, the optimization function is the average distance
    from significance level or y <-  mean(abs(new.par$pvalue[paths] -
    rep(sig.level,length(paths))))#'}

\item{d}{Domains for initial sampling, default is c(-1 ,1) for all. It can
be specified as a list of ranges (e.g., d = list(-1, 1, -1, 1) for two
sampling domains).}

\item{paths}{Paths in the model to be evaluated in a sensitivity analysis.}

\item{verbose}{Print out evaluation process if true, default is TRUE.}

\item{max.value}{Maximal value of optimization when used as
the stopping criterion}

\item{max.iter}{Maximal number of function evaluations when used as
the stopping criterion}

\item{e}{Maximum error value used when solution quality used as
stopping criterion, default is 1e-10.}

\item{n.of.ants}{Number of ants used in each iteration after the initialization
of k length, default value is 10.}

\item{k}{Size of the solution archive, default is 50.}

\item{q}{Locality of the search (0,1), default is 0.0001}

\item{sig.level}{Significance level, default value is 0.05.}

\item{xi}{Convergence pressure (0,Inf), suggested: (0,1), default is 0.5}

\item{seed}{Random seed if specified, default is NULL.}
}
\value{
Sensitivity analysis results
}
\description{
This function can perform sensitivity analysis for
    structural equation modeling using ant colony optimization (ACO).
}
\examples{
library(lavaan)
# generate data
sim.model <- ' x =~ x1 + 0.8*x2 + 1.2*x3
               y =~ y1 + 0.5*y2 + 1.5*y3
               m ~ 0.5*x
               y ~ 0.5*x + 0.8*m'
set.seed(10)
data <- simulateData(sim.model, sample.nobs = 1000L)
# standardize dataset
data = data.frame(apply(data,2,scale))

# Step 1: Set up the analytic model of interest
model <- 'x =~ x1 + x2 + x3
          y =~ y1 + y2 + y3
          m ~ x
          y ~ x + m'

# Step 2: Set up sensitivity analysis model
#         the sensitivity parameters are phantom1, phantom2 and phantom3
sens.model = 'x =~ x1 + x2 + x3
              y =~ y1 + y2 + y3
              m ~ x
              y ~ x + m
              x ~ phantom1*phantom
              m ~ phantom2*phantom
              y ~ phantom3*phantom
              phantom =~ 0 # added for mean of zero
              phantom ~~ 1*phantom' # added for unit variance

# Step 3: check the analytic model results and decide parameter of interests
#         for sensitivity analysis
old.model = lavaan::lavaanify(model = model, auto = TRUE, model.type="sem")
old.out = lavaan::lavaan(model = old.model, data = data)
old.par = lavaan::standardizedSolution(old.out, type="std.all")
old.par # we are interested in lines 7, 8 and 9 for the indirect and direct effects

# Step 4: perform sensitivity analysis
my.sa <- sa.aco(data, model = model, sens.model = sens.model,
                n.of.sens.pars = 3, k = 5,
                opt.fun = quote(1/abs(new.par$pvalue[9]-0.05)), #p-value
                paths = 9,
                max.iter = 10)
   # Note: We run with k = 5 and max.iter = 10 for illustration purpose in 5 seconds,
  #  please specify them as larger numbers (e.g., default value of k = 50 and mat.iter = 1000)


# Step 5: summarize sensitivity analysis results
tables <- sens.tables(my.sa)
tables

}
\references{
Socha, K., & Dorigo, M. (2008). Ant colony optimization for
  continuous domains. \emph{European Journal of Operational Research,
  185}(3), 1155-1173.

  Harring, J. R., McNeish, D. M., & Hancock, G. R. (2017).
  Using phantom variables in structural equation modeling to
  assess model sensitivity to external misspecification.
  \emph{Psychological Methods, 22}(4), 616.

  We thank Dr. Krzysztof Socha for providing us the
  original code (http://iridia.ulb.ac.be/supp/IridiaSupp2008-001/)
  that the current function is based on.
}
