% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulation.R
\docType{methods}
\name{simInit}
\alias{simInit}
\alias{simInit,ANY,ANY,ANY,ANY,ANY,ANY,ANY,ANY-method}
\alias{simInit,ANY,ANY,ANY,character,ANY,ANY,ANY,ANY-method}
\alias{simInit,ANY,ANY,character,ANY,ANY,ANY,ANY,ANY-method}
\alias{simInit,list,list,list,list,list,data.frame,data.frame,character-method}
\title{Initialize a new simulation}
\usage{
simInit(times, params, modules, objects, paths, inputs, outputs, loadOrder)


  \S4method{simInit}{list,list,list,list,list,data.frame,data.frame,character}(times,
  params, modules, objects, paths, inputs, outputs, loadOrder)

\S4method{simInit}{ANY,ANY,ANY,character,ANY,ANY,ANY,ANY}(times, params,
  modules, objects, paths, inputs, outputs, loadOrder)

\S4method{simInit}{ANY,ANY,character,ANY,ANY,ANY,ANY,ANY}(times, params,
  modules, objects, paths, inputs, outputs, loadOrder)

\S4method{simInit}{ANY,ANY,ANY,ANY,ANY,ANY,ANY,ANY}(times, params, modules,
  objects, paths, inputs, outputs, loadOrder)
}
\arguments{
\item{times}{A named list of numeric simulation start and end times
(e.g., \code{times = list(start = 0.0, end = 10.0)}).}

\item{params}{A list of lists of the form list(moduleName=list(param1=value, param2=value)).
See details.}

\item{modules}{A named list of character strings specfying the names
of modules to be loaded for the simulation. Note: the module name
should correspond to the R source file from which the module is loaded.
Example: a module named "caribou" will be sourced form the file
\file{caribou.R}, located at the specified \code{modulePath(simList)} (see below).}

\item{objects}{(optional) A vector of object names (naming objects
that are in the calling environment of
the \code{simInit}, which is often the
\code{.GlobalEnv} unless used programmatically
-- NOTE: this mechanism will
fail if object name is in a package dependency), or
a named list of data objects to be
passed into the simList (more reliable).
These objects will be accessible
from the simList as a normal list, e.g,. \code{mySim$obj}.}

\item{paths}{An optional named list with up to 4 named elements,
\code{modulePath}, \code{inputPath}, \code{outputPath}, and \code{cachePath}.
See details.}

\item{inputs}{A \code{data.frame}. Can specify from 1 to 6
columns with following column names: \code{objectName} (character, required),
\code{file} (character), \code{fun} (character), \code{package} (character),
\code{interval} (numeric), \code{loadTime} (numeric).
See \code{\link{inputs}} and vignette("ii-modules") section about inputs.}

\item{outputs}{A \code{data.frame}. Can specify from 1 to 5
columns with following column names: \code{objectName} (character, required),
\code{file} (character), \code{fun} (character), \code{package} (character),
\code{saveTime} (numeric). See \code{\link{outputs}} and
\code{vignette("ii-modules")} section about outputs.}

\item{loadOrder}{An optional list of module names specfiying the order in
which to load the modules. If not specified, the module
load order will be determined automatically.}
}
\value{
A \code{simList} simulation object, pre-initialized from values
specified in the arguments supplied.
}
\description{
Create a new simulation object, the "sim" object. This object is implemented
using an \code{environment} where all objects and functions are placed.
Since environments in \code{R} are
pass by reference, "putting" objects in the sim object does no actual copy. This
is also the location of all parameters, and other important simulation information, such
as times, paths, modules, and module load order. See more details below.
}
\details{
Calling this simInit function does several things including the following:
- sources all module files, placing all function definitions in the sim object
- optionally copies objects from the global environment to the sim object
- optionally loads objects from disk
- schedules all "init" events from all modules
- assesses module dependencies via the inputs and outputs identified in their metadata
- determines time units of modules and how they fit together

\code{params} can only contain updates to any parameters that are defined in
the metadata of modules. Take the example of a module named, \code{Fire}, which
has a parameter named \code{.plotInitialTime}. In the metadata of that moduel,
it says TRUE. Here we can override that default with:
\code{list(Fire=list(.plotInitialTime=NA))}, effectively turning off plotting. Since
this is a list of lists, one can override the module defaults for multiple parameters
from multiple modules all at once, with say:
\code{list(Fire = list(.plotInitialTime = NA, .plotInterval = 2),
           caribouModule = list(N = 1000))}.

We implement a discrete event simulation in a more modular fashion so it is
easier to add modules to the simulation. We use S4 classes and methods,
and use \code{data.table} instead of \code{data.frame} to implement the event
queue (because it is much faster).

\code{paths} specifies the location of the module source files,
the data input files, and the saving output files. If no paths are specified
the defaults are as follows:

\itemize{
  \item \code{cachePath}: \code{getOption("spades.cachePath")};

\item \code{inputPath}: \code{getOption("spades.modulePath")};

\item \code{modulePath}: \code{getOption("spades.inputPath")};

\item \code{inputPath}: \code{getOption("spades.outputPath")}.
}
}
\note{
The user can opt to run a simpler simInit call without inputs, outputs, and times.
These can be added later with the accessor methods (See example). These are not required for initializing the
simulation via simInit. \code{modules}, \code{paths}, \code{params}, and \code{objects}
are all needed for initialization.
}
\examples{
\dontrun{
 mySim <- simInit(
   times = list(start = 0.0, end = 2.0, timeunit = "year"),
   params = list(
     .globals = list(stackName = "landscape", burnStats = "nPixelsBurned")
   ),
   modules = list("randomLandscapes", "fireSpread", "caribouMovement"),
   paths = list(modulePath = system.file("sampleModules", package = "SpaDES"))
 )
 spades(mySim, .plotInitialTime = NA)

 # Change more parameters, removing plotting
 wantPlotting <- FALSE
 mySim <- simInit(
   times = list(start = 0.0, end = 2.0, timeunit = "year"),
   params = list(
     .globals = list(stackName = "landscape", burnStats = "nPixelsBurned"),
     fireSpread = list(.plotInitialTime=wantPlotting),
     #caribouMovement = list(.plotInitialTime=wantPlotting),
     #randomLandscapes = list(.plotInitialTime=wantPlotting)
   ),
   modules = list("randomLandscapes", "fireSpread", "caribouMovement"),
   paths = list(modulePath = system.file("sampleModules", package = "SpaDES"))
 )
 outSim <- spades(mySim)

# A little more complicated with inputs and outputs
if (require(rgdal)) {
   mapPath <- system.file("maps", package = "SpaDES")
   mySim <- simInit(
     times = list(start = 0.0, end = 2.0, timeunit = "year"),
     params = list(
       .globals = list(stackName = "landscape", burnStats = "nPixelsBurned")
     ),
     modules = list("randomLandscapes", "fireSpread", "caribouMovement"),
     paths = list(modulePath = system.file("sampleModules", package = "SpaDES"),
                  outputPath = tempdir()),
     inputs = data.frame(
       files = dir(file.path(mapPath), full.names = TRUE, pattern = "tif")[1:2],
       functions = "raster",
       package = "raster",
       loadTime = 0,
       stringsAsFactors = FALSE),
     outputs = data.frame(
       expand.grid(objectName = c("caribou","landscape"),
       saveTime = 1:2,
       stringsAsFactors = FALSE))
   )

   # Use accessors for inputs, outputs, times
   mySim2 <- simInit(modules = list("randomLandscapes", "fireSpread",
                                    "caribouMovement"),
                     params = list(.globals = list(stackName = "landscape",
                                                   burnStats = "nPixelsBurned")),
                     paths = list(modulePath = system.file("sampleModules",
                                                           package = "SpaDES"),
                                  outputPath = tempdir()))
   # add by accessor: note need current in times() accessor
   times(mySim2) <- list(current=0, start = 0.0, end = 2.0, timeunit = "year")
   inputs(mySim2) <- data.frame(
       files = dir(file.path(mapPath), full.names = TRUE, pattern = "tif")[1:2],
       functions = "raster",
       package = "raster",
       loadTime = 3,
       stringsAsFactors = FALSE)
   outputs(mySim2) <- data.frame(
       expand.grid(objectName = c("caribou","landscape"),
       saveTime = 1:2,
       stringsAsFactors = FALSE))
   all.equal(mySim, mySim2) # TRUE
  }
}

}
\author{
Alex Chubaty and Eliot McIntire
}
\references{
Matloff, N. (2011). The Art of R Programming (ch. 7.8.3). San Fransisco, CA: No Starch Press, Inc.. Retrieved from \url{https://www.nostarch.com/artofr.htm}
}
\seealso{
\code{\link{spades}},
\code{\link{times}}, \code{\link{params}}, \code{\link{objs}}, \code{\link{paths}},
\code{\link{modules}}, \code{\link{inputs}}, \code{\link{outputs}}
}

