% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hypothesis_testing.R
\name{hyptest}
\alias{hyptest}
\title{Test for Time-Varying Covariance via Local PCA and Bootstrap}
\usage{
hyptest(returns, m, B = 200, kernel_func = epanechnikov_kernel)
}
\arguments{
\item{returns}{A numeric matrix of asset returns with dimensions \eqn{T × p} (time periods by assets).}

\item{m}{Integer. The number of factors to extract in the local PCA. See \code{\link{determine_factors}}.}

\item{B}{Integer. The number of bootstrap replications to perform. Default is 200}

\item{kernel_func}{Function. A kernel function for weighting observations in the local PCA. Default is \code{epanechnikov_kernel}.}
}
\value{
A list containing:
\item{J_NT}{The test statistic \eqn{J_{pT}} computed on the original data.}
\item{p_value}{The bootstrap p-value, indicating the significance of time variation in covariance.}
\item{J_pT_bootstrap}{A numeric vector of bootstrap test statistics from each replication.}
}
\description{
This function performs a hypothesis test for time-varying covariance in asset returns based on Su and Wang (2017).
It first standardizes the input returns and then computes a time-varying covariance estimator
using a local principal component analysis (Local PCA) approach. The test statistic \eqn{J_{pT}}
is computed and its significance is assessed using a bootstrap procedure. The procedure is available either as a stand-alone
function or as a method in the `TVMVP` R6 class.
}
\details{
Two usage styles:

\preformatted{
# Function interface
hyptest(returns, m=2)

# R6 method interface
tv <- TVMVP$new()
tv$set_data(returns)
tv$determine_factors(max_m=5)
tv$hyptest()
tv
tv$get_bootstrap()         # prints bootstrap test statistics
}
   
When using the method form, if `m` are omitted,
they default to values stored in the object. Results are cached and
retrievable via class methods.

The function follows the steps below:

\enumerate{
  \item Standardizes the returns.
  \item Computes the optimal bandwidth using the Silverman rule.
  \item Performs a local PCA on the standardized returns to extract local factors and loadings.
  \item Computes a global factor model via singular value decomposition (SVD) to obtain global factors.
  \item Calculates residuals by comparing the local PCA reconstruction to the standardized returns.
  \item Computes a test statistic \eqn{J_{pT}} based on a function of the residuals and covariance estimates as:

  \deqn{\hat{J}_{pT} = \frac{T p^{1/2} h^{1/2} \hat{M} - \hat{\mathbb{B}}_{pT}}{\sqrt{\hat{\mathbb{V}}_{pT}}},}

  where:

  \deqn{\hat{M} = \frac{1}{pT} \sum_{i=1}^p \sum_{t=1}^T \left(\hat{\lambda}_{it}' \hat{F}_t - \tilde{\lambda}_{i0}' \tilde{F}_t\right),}

  \deqn{\hat{\mathbb{B}}_{pT} = \frac{h^{1/2}}{T^2 p^{1/2}} \sum_{i=1}^p \sum_{t=1}^T \sum_{s=1}^T \left(k_{h,st} \hat{F}_s' \hat{F}_t - \tilde{F}_s' \tilde{F}_t\right)^2 \hat{e}_{is}^2,}

  and

  \deqn{\hat{\mathbb{V}}_{pT} = \frac{2}{p h T^2} \sum_{1\leq s \neq r \leq T} \bar{k}_{sr}^2 \left(\hat{F}_s' \hat{\Sigma}_F \hat{F}_r \right)^2 \left(\hat{e}_r' \hat{e}_s \right)^2.}

  \item A bootstrap procedure is then used to compute the distribution of \eqn{J_{pT}} and derive a p-value.
}

The function prints a message indicating the strength of evidence for time-varying covariance based on the p-value.
}
\section{References}{
 
Su, L., & Wang, X. (2017). On time-varying factor models: Estimation and testing. Journal of Econometrics, 198(1), 84–101
}

\examples{
\donttest{
# Simulate some random returns (e.g., 100 periods, 30 assets)
set.seed(123)
returns <- matrix(rnorm(100*30, mean = 0, sd = 0.02), nrow = 100, ncol = 30)

# Test for time-varying covariance using 3 factors and 10 bootstrap replications
test_result <- hyptest(returns, m = 3, B = 10, kernel_func = epanechnikov_kernel)

# Print test statistic and p-value
print(test_result$J_NT)
print(test_result$p_value)

# Or use R6 method interface
tv <- TVMVP$new()
tv$set_data(returns)
tv$determine_factors(max_m=5)
tv$hyptest(iB = 10, kernel_func = epanechnikov_kernel)
tv
tv$get_bootstrap()         # prints bootstrap test statistics
}

}
