% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MaximizeParsimony.R
\encoding{UTF-8}
\name{MaximizeParsimony}
\alias{MaximizeParsimony}
\alias{Resample}
\alias{EasyTrees}
\alias{EasyTreesy}
\title{Find most parsimonious trees}
\usage{
MaximizeParsimony(
  dataset,
  tree,
  ratchIter = 6L,
  tbrIter = 2L,
  startIter = 2L,
  finalIter = 1L,
  maxHits = NTip(dataset) * 1.8,
  maxTime = 60,
  quickHits = 1/3,
  concavity = Inf,
  tolerance = sqrt(.Machine$double.eps),
  constraint,
  verbosity = 3L
)

Resample(
  dataset,
  tree,
  method = "jack",
  proportion = 2/3,
  ratchIter = 1L,
  tbrIter = 8L,
  finalIter = 3L,
  maxHits = 12L,
  concavity = Inf,
  tolerance = sqrt(.Machine$double.eps),
  constraint,
  verbosity = 2L,
  ...
)

EasyTrees()

EasyTreesy()
}
\arguments{
\item{dataset}{A phylogenetic data matrix of class \code{\link[phangorn]{phyDat}},
whose names correspond to the labels of any accompanying tree.}

\item{tree}{(optional) A bifurcating tree of class \code{\link{phylo}},
containing only the tips listed in \code{dataset}, from which the search
should begin.
If unspecified, an \link[=AdditionTree]{addition tree} will be generated from
\code{dataset}, respecting any supplied \code{constraint}.
Edge lengths are not supported and will be deleted.}

\item{ratchIter}{Numeric specifying number of iterations of the
parsimony ratchet \insertCite{Nixon1999}{TreeSearch} to conduct.}

\item{tbrIter}{Numeric specifying the maximum number of \acronym{TBR}
break points to evaluate before concluding each search.
The counter is reset to zero each time tree score improves.
The counter is reset to zero each time tree score improves.
One 'iteration' comprises breaking a single branch and evaluating all
possible reconnections.}

\item{startIter}{Numeric: an initial round of tree search with
\code{startIter} × \code{tbrIter} \acronym{TBR} break points is conducted in
order to locate a local optimum before beginning ratchet searches.}

\item{finalIter}{Numeric: a final round of tree search will evaluate
\code{finalIter} × \code{tbrIter} \acronym{TBR} break points, in order to
sample the final optimal neighbourhood more intensely.}

\item{maxHits}{Numeric specifying the maximum times that an optimal
parsimony score may be hit before concluding a ratchet iteration or final
search concluded.}

\item{maxTime}{Numeric: after \code{maxTime} minutes, stop tree search at the
next opportunity.}

\item{quickHits}{Numeric: iterations on subsampled datasets
will retain \code{quickHits} × \code{maxHits} trees with the best score.}

\item{concavity}{Numeric specifying concavity constant for implied step
weighting; set as \code{Inf} for equal step weights (which is a bad idea; see
\insertCite{Smith2019;textual}{TreeSearch}).}

\item{tolerance}{Numeric specifying degree of suboptimality to tolerate
before rejecting a tree.  The default, \code{sqrt(.Machine$double.eps)}, retains
trees that may be equally parsimonious but for rounding errors.
Setting to larger values will include trees suboptimal by up to \code{tolerance}
in search results, which may improve the accuracy of the consensus tree
(at the expense of resolution) \insertCite{Smith2019}{TreeSearch}.}

\item{constraint}{An object of class \code{phyDat}; returned trees will be
perfectly compatible with each character in \code{constraint}.
See \code{\link[=ImposeConstraint]{ImposeConstraint()}} and
\href{https://ms609.github.io/TreeSearch/articles/inapplicable.html}{vignette}
for further examples.}

\item{verbosity}{Integer specifying level of messaging; higher values give
more detailed commentary on search progress. Set to \code{0} to run silently.}

\item{method}{Unambiguous abbreviation of \code{jackknife} or \code{bootstrap}
specifying how to resample characters.  Note that jackknife is considered
to give more meaningful results.}

\item{proportion}{Numeric between 0 and 1 specifying what proportion of
characters to retain under jackknife resampling.}

\item{\dots}{Additional parameters to \code{MaximizeParsimony()}.}
}
\value{
\code{MaximizeParsimony()} returns a list of trees with class
\code{multiPhylo}. This lists all trees found during each search step that
are within \code{tolerance} of the optimal score, listed in the sequence that
they were first visited; it may contain more than \code{maxHits} elements.
Note that the default search parameters may need to be increased in order for
these trees to be the globally optimal trees; examine the messages printed
during tree search to evaluate whether the optimal score has stabilized.

The return value has the attribute \code{newTrees}, a named integer vector listing
the number of optimal trees visited for the first time in each stage of
the tree search.

\code{Resample()} returns a \code{multiPhylo} object containing a list of
trees obtained by tree search using a resampled version of \code{dataset}.
}
\description{
Search for most parsimonious trees using the parsimony ratchet and
\acronym{TBR} rearrangements, treating inapplicable data as such using the
algorithm of \insertCite{Brazeau2019;textual}{TreeSearch}.

Tree search will be conducted from a specified or automatically-generated
starting tree in order to find a tree with an optimal parsimony score,
under implied or equal weights, treating inapplicable characters as such
in order to avoid the artefacts of the standard Fitch algorithm
\insertCite{@see @Maddison1993; @Brazeau2019}{TreeSearch}.
The tree scoring implementation uses the MorphyLib C library
\insertCite{Brazeau2017}{TreeSearch}.
}
\details{
Tree search commences with \code{ratchIter} iterations of the parsimony ratchet
\insertCite{Nixon1999}{TreeSearch}, which bootstraps the input dataset
in order to escape local optima.
A final round of tree bisection and reconnection (\acronym{TBR})
is conducted to broaden the sampling of trees.

This function can be called using the R command line / terminal, or through
the 'shiny' graphical user interface app (type \code{EasyTrees()} to launch).

For detailed documentation of the 'TreeSearch' package, including full
instructions for loading phylogenetic data into R and initiating and
configuring tree search, see the
\href{https://ms609.github.io/TreeSearch/}{package documentation}.
}
\section{Resampling}{

Note that bootstrap support is a measure of the amount of data supporting
a split, rather than the amount of confidence that should be afforded the
grouping.
"Bootstrap support of 100\% is not enough, the tree must also be correct"
\insertCite{Phillips2004}{TreeSearch}.
See discussion in \insertCite{Egan2006;textual}{TreeSearch};
\insertCite{Wagele2009;textual}{TreeSearch};
\insertCite{Simmons2011}{TreeSearch};
\insertCite{Kumar2012;textual}{TreeSearch}.

For a discussion of suitable search parameters in resampling estimates, see
\insertCite{Muller2005;textual}{TreeSearch}.
The user should decide whether to start each resampling
from the optimal tree (which may be quicker, but result in overestimated
support values as searches get stuck in local optima close to the
optimal tree) or a random tree (which may take longer as more rearrangements
are necessary to find an optimal tree on each iteration).

For other ways to estimate clade concordance, see \code{\link[=SiteConcordance]{SiteConcordance()}}.
}

\examples{
## Only run examples in interactive R sessions
if (interactive()) {
  # launch 'shiny' point-and-click interface
  EasyTrees()
  
  # Here too, use the "continue search" function to ensure that tree score
  # has stabilized and a global optimum has been found
}


# Load data for analysis in R
library('TreeTools')
data('congreveLamsdellMatrices', package = 'TreeSearch')
dataset <- congreveLamsdellMatrices[[42]]

# A very quick run for demonstration purposes
trees <- MaximizeParsimony(dataset, ratchIter = 0, startIter = 0,
                           tbrIter = 1, maxHits = 4, maxTime = 1/100,
                           concavity = 10, verbosity = 4)

# In actual use, be sure to check that the score has converged on a global
# optimum, conducting additional iterations and runs as necessary.
 
if (interactive()) {
# Jackknife resampling
nReplicates <- 10
jackTrees <- replicate(nReplicates,
  #c() ensures that each replicate returns a list of trees
  c(Resample(dataset, trees, ratchIter = 0, tbrIter = 2, startIter = 1,
             maxHits = 5, maxTime = 1 / 10,
             concavity = 10, verbosity = 0))
 )

# In a serious analysis, more replicates would be conducted, and each
# search would undergo more iterations.

# Now we must decide what to do with the multiple optimal trees from
# each replicate.

# Treat each tree equally
JackLabels(ape::consensus(trees), unlist(jackTrees, recursive = FALSE))

# Take the strict consensus of all trees for each replicate
JackLabels(ape::consensus(trees), lapply(jackTrees, ape::consensus))

# Take a single tree from each replicate (the first; order's irrelevant)
JackLabels(ape::consensus(trees), lapply(jackTrees, `[[`, 1))
}

# Tree search with a constraint
constraint <- MatrixToPhyDat(c(a = 1, b = 1, c = 0, d = 0, e = 0, f = 0))
characters <- MatrixToPhyDat(matrix(
  c(0, 1, 1, 1, 0, 0,
    1, 1, 1, 0, 0, 0), ncol = 2,
  dimnames = list(letters[1:6], NULL)))
MaximizeParsimony(characters, constraint = constraint, verbosity = 0)

}
\references{
\insertAllCited{}
}
\seealso{
Tree search \emph{via} graphical user interface: \code{\link[=EasyTrees]{EasyTrees()}}

Other split support functions: 
\code{\link{JackLabels}()},
\code{\link{Jackknife}()},
\code{\link{SiteConcordance}}
}
\author{
\href{https://smithlabdurham.github.io/}{Martin R. Smith}
(\href{mailto:martin.smith@durham.ac.uk}{martin.smith@durham.ac.uk})
}
\concept{split support functions}
