#'@title Retrieve user contributions
#'
#'@description Retrieves metadata associated with the most recent contributions by a
#'specified user.
#'
#'@param con A connector object, generated by \code{\link{wiki_con}},
#'that corresponds to the project you're trying to query.
#'
#'@param user The username of the user whose contributions you want to retrieve.
#'Due to limitations at the API end, you can only retrieve edits for one user at a time.
#'
#'@param properties The metadata you want associated with each edit. Potential metadata includes "ids"
#'(the revision ID of the revision, which can be passed into \code{\link{wiki_revision}}),
#'"title" (the name of the page that was edited), "timestamp" (the timestamp of the edit,
#'which can be parsed with \code{\link{wiki_timestamp}}), "comment" (the edit summary associated
#'with the revision), "parsedcomment" (the same, but parsed, generating HTML from any wikitext
#'in that comment), "size" (the size, in uncompressed bytes, of the edit), "sizediff" (the size
#'delta between this edit, and the last edit to the page), "flags" (whether the revision was 
#''minor' or not), and "tags" (any tags associated with the revision).
#'
#'@param mainspace A boolean flag; FALSE retrieves all of the most recent contributions, while
#'TRUE limits the retrieved contributions to those in the 'mainspace' - in other words, edits to
#'actual articles. Set to FALSE by default
#'
#'@param limit The number of edits to be retrieved. 50 is the maximum for logged-out API users,
#'and putting in more than 50 will generate a warning.
#'
#'@export

wiki_usercontribs <- function(con,
                              user,
                              properties = c("ids", "title", "timestamp",
                                             "comment", "parsedcomment", "size", 
                                             "sizediff", "flags", "tags"),
                              mainspace = FALSE,
                              limit = 50){
  
  #Check and collapse properties
  properties <- match.arg(properties, several.ok = TRUE)
  properties <- paste(properties, collapse = "|")
  
  #Check that >1 users wasn't submitted
  user <- LimitHandler(user, 1)
  
  #Construct URL
  contribs_url <- paste(con$URL, "&action=query&list=usercontribs&uclimit=", limit,
                        "&ucuser=", user, "&ucprop=", properties, sep = "")

  #If only article contributions are desired, note that.
  if(mainspace){
    
    contribs_url <- paste(contribs_url, "&ucnamespace=0", sep = "")
    
  }
  
  #Get
  contribs_content <- wiki_call(contribs_url, con$CurlOpts)
    
  #Check
  MissingUsersHandler(contribs_content)
  
  #Return
  return(contribs_content)
}