% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/activeC.R
\name{C_GP}
\alias{C_GP}
\title{C matrix closed form expression for a GP.}
\usage{
C_GP(model)
}
\arguments{
\item{model}{\code{homGP} or \code{hetGP} GP model, see \code{\link[hetGP]{hetGP-package}} containing, e.g.,
 a vector of \code{theta}s, type of covariance \code{ct}, an inverse covariance matrix \code{Ki},
a design matrix \code{X0}, and response vector \code{Z0}.}
}
\value{
a \code{const_C} object with elements
\itemize{
\item \code{model}: GP model provided;
\item \code{mat}: C matrix estimated;
\item \code{Wij}: list of W matrices, of size number of variables;
\item \code{ct}: covariance type (1 for "Gaussian", 2 for "Matern3_2", 3 for "Matern5_2").
}
}
\description{
Computes the integral (uniform measure) over the [0,1] box domain of the outer product of the gradients of a Gaussian process. 
The corresponding matrix is the C matrix central in the active subspace methodology.
}
\examples{
################################################################################
### Active subspace of a Gaussian process
################################################################################
library(hetGP); library(lhs)
set.seed(42)

nvar <- 2
n <- 100

# theta gives the subspace direction
f <- function(x, theta, nugget = 1e-3){
  if(is.null(dim(x))) x <- matrix(x, 1)
  xact <- cos(theta) * x[,1] - sin(theta) * x[,2]
  return(hetGP::f1d(xact) + rnorm(n = nrow(x), sd = rep(nugget, nrow(x))))
}

theta_dir <- pi/6
act_dir <- c(cos(theta_dir), -sin(theta_dir))

# Create design of experiments and initial GP model
design <- X <- matrix(signif(maximinLHS(n, nvar), 2), ncol = nvar)
response <- Y <- apply(design, 1, f, theta = theta_dir)
model <- mleHomGP(design, response, known = list(beta0 = 0))

C_hat <- C_GP(model)

# Subspace distance to true subspace:
print(subspace_dist(C_hat, matrix(act_dir, nrow = nvar), r = 1))
plot(design \%*\% eigen(C_hat$mat)$vectors[,1], response, 
  main = "Projection along estimated active direction")
plot(design \%*\% eigen(C_hat$mat)$vectors[,2], response, 
  main = "Projection along estimated inactive direction")
  
# For other plots:
# par(mfrow = c(1, 3)) # uncomment to have all plots together
plot(C_hat)
# par(mfrow = c(1, 1)) # restore graphical window

}
\references{
N. Wycoff, M. Binois, S. Wild (2019+), Sequential Learning of Active Subspaces, preprint.\cr

P. Constantine (2015), Active Subspaces, Philadelphia, PA: SIAM.
}
\seealso{
\code{\link[activegp]{print.const_C}}, \code{\link[activegp]{plot.const_C}}
}
