% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/derive_advs_params.R
\name{derive_param_bsa}
\alias{derive_param_bsa}
\title{Adds a Parameter for BSA (Body Surface Area) Using the Specified Method}
\usage{
derive_param_bsa(
  dataset,
  by_vars,
  method,
  set_values_to = exprs(PARAMCD = "BSA"),
  height_code = "HEIGHT",
  weight_code = "WEIGHT",
  get_unit_expr,
  filter = NULL,
  constant_by_vars = NULL
)
}
\arguments{
\item{dataset}{Input dataset

The variables specified by the \code{by_vars} argument are expected to be in the dataset.
\code{PARAMCD}, and \code{AVAL} are expected as well.

The variable specified by \code{by_vars} and \code{PARAMCD} must be a unique key of
the input dataset after restricting it by the filter condition (\code{filter}
parameter) and to the parameters specified by \code{HEIGHT} and \code{WEIGHT}.}

\item{by_vars}{Grouping variables

For each group defined by \code{by_vars} an observation is added to the output
dataset. Only variables specified in \code{by_vars} will be populated
in the newly created records.

\emph{Permitted Values}: list of variables created by \code{exprs()}
e.g. \code{exprs(USUBJID, VISIT)}}

\item{method}{Derivation method to use. Note that \code{HEIGHT} is expected
in cm and \code{WEIGHT} is expected in kg:

Mosteller: \code{sqrt(height * weight / 3600)}

DuBois-DuBois: \code{0.20247 * (height/100) ^ 0.725 * weight ^ 0.425}

Haycock: \code{0.024265 * height ^ 0.3964 * weight ^ 0.5378}

Gehan-George: \code{0.0235 * height ^ 0.42246 * weight ^ 0.51456}

Boyd: \code{0.0003207 * (height ^ 0.3) * (1000 * weight) ^ (0.7285 - (0.0188 * log10(1000 * weight)))}

Fujimoto: \code{0.008883 * height ^ 0.663 * weight ^ 0.444}

Takahira: \code{0.007241 * height ^ 0.725 * weight ^ 0.425}

\emph{Permitted Values:} character value}

\item{set_values_to}{Variables to be set

The specified variables are set to the specified values for the new
observations. For example \code{exprs(PARAMCD = "MAP")} defines the parameter code
for the new parameter.

\emph{Permitted Values}: List of variable-value pairs}

\item{height_code}{HEIGHT parameter code

The observations where \code{PARAMCD} equals the specified value are considered
as the HEIGHT assessments. It is expected that HEIGHT is measured in cm.

\emph{Permitted Values:} character value}

\item{weight_code}{WEIGHT parameter code

The observations where \code{PARAMCD} equals the specified value are considered
as the WEIGHT assessments. It is expected that WEIGHT is measured in kg.

\emph{Permitted Values:} character value}

\item{get_unit_expr}{An expression providing the unit of the parameter

The result is used to check the units of the input parameters.

Permitted Values: A variable of the input dataset or a function call}

\item{filter}{Filter condition

The specified condition is applied to the input dataset before deriving the
new parameter, i.e., only observations fulfilling the condition are taken
into account.

\emph{Permitted Values:} a condition}

\item{constant_by_vars}{By variables for when HEIGHT is constant

When HEIGHT is constant, the HEIGHT parameters (measured only once) are merged
to the other parameters using the specified variables.

If height is constant (e.g. only measured once at screening or baseline) then
use \code{constant_by_vars} to select the subject-level variable to merge on (e.g. \code{USUBJID}).
This will produce BSA at all visits where weight is measured.  Otherwise
it will only be calculated at visits with both height and weight collected.

\emph{Permitted Values}: list of variables created by \code{exprs()}
e.g. \code{exprs(USUBJID, VISIT)}}
}
\value{
The input dataset with the new parameter added. Note, a variable will only
be populated in the new parameter rows if it is specified in \code{by_vars}.
}
\description{
Adds a record for BSA (Body Surface Area) using the specified derivation
method for each by group (e.g., subject and visit) where the source parameters are
available.

\strong{Note:} This is a wrapper function for the more generic \code{derive_param_computed()}.
}
\examples{
library(tibble)

# Example 1: Derive BSA where height is measured only once using constant_by_vars
advs <- tibble::tribble(
  ~USUBJID, ~PARAMCD, ~PARAM, ~AVAL, ~VISIT,
  "01-701-1015", "HEIGHT", "Height (cm)", 170, "BASELINE",
  "01-701-1015", "WEIGHT", "Weight (kg)", 75, "BASELINE",
  "01-701-1015", "WEIGHT", "Weight (kg)", 78, "MONTH 1",
  "01-701-1015", "WEIGHT", "Weight (kg)", 80, "MONTH 2",
  "01-701-1028", "HEIGHT", "Height (cm)", 185, "BASELINE",
  "01-701-1028", "WEIGHT", "Weight (kg)", 90, "BASELINE",
  "01-701-1028", "WEIGHT", "Weight (kg)", 88, "MONTH 1",
  "01-701-1028", "WEIGHT", "Weight (kg)", 85, "MONTH 2",
)

derive_param_bsa(
  advs,
  by_vars = exprs(USUBJID, VISIT),
  method = "Mosteller",
  set_values_to = exprs(
    PARAMCD = "BSA",
    PARAM = "Body Surface Area (m^2)"
  ),
  get_unit_expr = extract_unit(PARAM),
  constant_by_vars = exprs(USUBJID)
)

derive_param_bsa(
  advs,
  by_vars = exprs(USUBJID, VISIT),
  method = "Fujimoto",
  set_values_to = exprs(
    PARAMCD = "BSA",
    PARAM = "Body Surface Area (m^2)"
  ),
  get_unit_expr = extract_unit(PARAM),
  constant_by_vars = exprs(USUBJID)
)

# Example 2: Derive BSA where height is measured only once and keep only one record
# where both height and weight are measured.

derive_param_bsa(
  advs,
  by_vars = exprs(USUBJID, VISIT),
  method = "Mosteller",
  set_values_to = exprs(
    PARAMCD = "BSA",
    PARAM = "Body Surface Area (m^2)"
  ),
  get_unit_expr = extract_unit(PARAM)
)

# Example 3: Pediatric study where height and weight are measured multiple times
advs <- tibble::tribble(
  ~USUBJID, ~PARAMCD, ~PARAM, ~AVAL, ~VISIT,
  "01-101-1001", "HEIGHT", "Height (cm)", 47.1, "BASELINE",
  "01-101-1001", "HEIGHT", "Height (cm)", 59.1, "WEEK 12",
  "01-101-1001", "HEIGHT", "Height (cm)", 64.7, "WEEK 24",
  "01-101-1001", "HEIGHT", "Height (cm)", 68.2, "WEEK 48",
  "01-101-1001", "WEIGHT", "Weight (kg)", 2.6, "BASELINE",
  "01-101-1001", "WEIGHT", "Weight (kg)", 5.3, "WEEK 12",
  "01-101-1001", "WEIGHT", "Weight (kg)", 6.7, "WEEK 24",
  "01-101-1001", "WEIGHT", "Weight (kg)", 7.4, "WEEK 48",
)
derive_param_bsa(
  advs,
  by_vars = exprs(USUBJID, VISIT),
  method = "Mosteller",
  set_values_to = exprs(
    PARAMCD = "BSA",
    PARAM = "Body Surface Area (m^2)"
  ),
  get_unit_expr = extract_unit(PARAM)
)
}
\seealso{
\code{\link[=compute_bsa]{compute_bsa()}}

BDS-Findings Functions for adding Parameters/Records: 
\code{\link{default_qtc_paramcd}()},
\code{\link{derive_expected_records}()},
\code{\link{derive_extreme_event}()},
\code{\link{derive_extreme_records}()},
\code{\link{derive_locf_records}()},
\code{\link{derive_param_bmi}()},
\code{\link{derive_param_computed}()},
\code{\link{derive_param_doseint}()},
\code{\link{derive_param_exist_flag}()},
\code{\link{derive_param_exposure}()},
\code{\link{derive_param_framingham}()},
\code{\link{derive_param_map}()},
\code{\link{derive_param_qtc}()},
\code{\link{derive_param_rr}()},
\code{\link{derive_param_wbc_abs}()},
\code{\link{derive_summary_records}()}
}
\concept{der_prm_bds_findings}
\keyword{der_prm_bds_findings}
