\name{ausplots_trajectory}
\alias{ausplots_trajectory}
\title{Visualisation of species composition and community index revisit trajectories
}
\description{Generates 2D plots visualising the trajectory of species composition (PCoA) and diversity (scatterplot or time series) for revisit series. Trajectories are calculated from the TERN AusPlots point intercept module (\code{$veg.PI}) as generated by \code{\link{get_ausplots}} and other pre-processing functions.
}
\usage{
ausplots_trajectory(my.ausplots.object, choices = c("PCoA", "diversity"),
min.revisits = 3, plot_select = NULL, outfile = NULL)
}
\arguments{
  \item{my.ausplots.object}{Input raw data object (list) as generated in a call from the \code{\link{get_ausplots}} function. See Details.
}
  \item{choices}{Character. User selection of PCoA and/or diversity plots. Both are generated by default.
}
  \item{min.revisits}{Integer. Minimum number of revisits required to include a plot.
}
  \item{plot_select}{Optional character vector of AusPlots plot IDs (\code{site_location_name}) to request data for a specific set of plots. Note PCoA will not run on one or very few plots/surveys.
}
\item{outfile}{Optional file path and name to which the output pdf file is written.
}
}
\details{This function primarily applies the functionality of R package \code{ecotraj} (De Cáceres et al. 2019; Sturbois et al. 2021) to the TERN AusPlots vegetation point intercept data by including the required pre-processing of the raw data and modifications to the output plot format. The function is intended as a rapid, visual means to compare revisit trajectories in species composition and diversity (or other community indices) among sites. For in-depth change/trajectory analysis, users are directed to work with the raw data and the extensive functionality in \code{ecotraj} or other approaches.

The default graphical outputs require \code{$site.info}, to order visits by date, and \code{$veg.PI}, for plant species abundance, in the AusPlots data extract. User-generated community indices (any single value per survey/\code{$site_unique}) can be plotted in place of diversity if \code{$community.indices} appears in the \code{my.ausplots.object}. This table must have a \code{$site_unique} column matching surveys in \code{$site.info} and any number of additional columns representing any community indices intended for plotting out over time. See Examples.

Two output options are available. 'PCoA' displays the first two axes (and their respective deviance explained as axis labels) of a metric ordination based on species abundances and so shows the trajectory of plots in species composition space. Similar trajectories on the plot implies parallel trends in species presence and abundance.

Next, 'diversity' by default displays a simple scatterplot with trajectories between revisits. The function presently computes species richness and Shannon diversity metrics using the \code{vegan} package (Oksanen et al. 2022) and displays these as trajectories on a 2D scatterplot, as well as individually against year. By adding a data table \code{$community.indices} to those available via \code{\link{get_ausplots}}, users can also plot out the trajectory of other bespoke community level indices. For > 2 indices, only individual plots against year of survey are generated. 

Note that revisits (and revisit series > 2) are available only for a subset of the full plot network and that \code{ausplots_trajectory} may display trajectories better for targeted sets of plots.
}
\value{
Plots are written to a pdf file to ensure consistent formatting, saved by default in the working directory.
}
\references{
De Cáceres, M., Coll, L., Legendre, P., Allen, R.B., Wiser, S.K., Fortin, M.J., Condit, R. & Hubbell, S. (2019). Trajectory analysis in community ecology. Ecological Monographs 89: e01350

Oksanen J, Simpson G, Blanchet F, Kindt R, Legendre P, Minchin P, O'Hara R, Solymos P, Stevens M, Szoecs E, Wagner H, Barbour M, Bedward M, Bolker B, Borcard D, Carvalho G, Chirico M, De Caceres M, Durand S, Evangelista H, FitzJohn R, Friendly M, Furneaux B, Hannigan G, Hill M, Lahti L, McGlinn D, Ouellette M, Ribeiro Cunha E, Smith T, Stier A, Ter Braak C, Weedon J (2022). _vegan: Community Ecology Package_. R package version 2.6-4, <https://CRAN.R-project.org/package=vegan>.

Sturbois, A., De Cáceres, M., Sánchez-Pinillos, M., Schaal, G., Gauthier, O., Le Mao, P., Ponsero, A., & Desroy, N. (2021). Extending community trajectory analysis : New metrics and representation. Ecological Modelling 440: 109400
}
\author{Greg Guerin
}
\seealso{\code{\link{get_ausplots}}
  \cr
  \cr
	\code{\link{ausplots_visual}} 
}
\examples{
\dontrun{
#Extract site data first:
my.data <- get_ausplots()

#Extract vegetation point intercept data for plots with revisits
#Here, subsetting my.Plots_IDs to revisits in one bioregion
my.data <- get_ausplots(veg.PI = TRUE, 
my.Plot_IDs = unique(subset(my.data$site.info, visit_number > 1 
& bioregion_name == "MAC", select=site_location_name))[,1])

#Display species composition trajectory for all plots with at least 3 revisits
ausplots_trajectory(my.data, choices = "PCoA")

#Display fractional cover trajectories (community index example)
my.data$community.indices <- fractional_cover(my.data$veg.PI)[,c("site_unique",
 "bare", "brown", "green")]

ausplots_trajectory(my.data, choices = "diversity")
}
}
