% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/weighted_posteriors.R
\name{weighted_posteriors}
\alias{weighted_posteriors}
\alias{weighted_posteriors.stanreg}
\alias{weighted_posteriors.brmsfit}
\alias{weighted_posteriors.BFBayesFactor}
\title{Generate posterior distributions weighted across models}
\usage{
weighted_posteriors(..., prior_odds = NULL, missing = 0, verbose = TRUE)

\method{weighted_posteriors}{stanreg}(
  ...,
  prior_odds = NULL,
  missing = 0,
  verbose = TRUE,
  effects = c("fixed", "random", "all"),
  component = c("conditional", "zi", "zero_inflated", "all"),
  parameters = NULL
)

\method{weighted_posteriors}{brmsfit}(
  ...,
  prior_odds = NULL,
  missing = 0,
  verbose = TRUE,
  effects = c("fixed", "random", "all"),
  component = c("conditional", "zi", "zero_inflated", "all"),
  parameters = NULL
)

\method{weighted_posteriors}{BFBayesFactor}(
  ...,
  prior_odds = NULL,
  missing = 0,
  verbose = TRUE,
  iterations = 4000
)
}
\arguments{
\item{...}{Fitted models (see details), all fit on the same data, or a single \code{BFBayesFactor} object (see 'Details').}

\item{prior_odds}{Optional vector of prior odds for the models compared to the first model (or the denominator, for \code{BFBayesFactor} objects).}

\item{missing}{An optional numeric value to use if a model does not contain a parameter that appears in other models. Defaults to 0.}

\item{verbose}{Toggle off warnings.}

\item{effects}{Should results for fixed effects, random effects or both be returned?
Only applies to mixed models. May be abbreviated.}

\item{component}{Should results for all parameters, parameters for the conditional model
or the zero-inflated part of the model be returned? May be abbreviated. Only
applies to \pkg{brms}-models.}

\item{parameters}{Regular expression pattern that describes the parameters that
should be returned. Meta-parameters (like \code{lp__} or \code{prior_}) are
filtered by default, so only parameters that typically appear in the
\code{summary()} are returned. Use \code{parameters} to select specific parameters
for the output.}

\item{iterations}{For \code{BayesFactor} models, how many posterior samples to draw.}
}
\value{
A data frame with posterior distributions (weighted across models) .
}
\description{
Extract posterior samples of parameters, weighted across models.
Weighting is done by comparing posterior model probabilities, via \code{\link{bayesfactor_models}}.
}
\details{
Note that across models some parameters might play different roles. For example,
the parameter \code{A} plays a different role in the model \code{Y ~ A + B} (where it is a main effect)
than it does in the model \code{Y ~ A + B + A:B} (where it is a simple effect). In many cases centering
of predictors (mean subtracting for continuous variables, and effects coding via \code{contr.sum} or
orthonormal coding via {\code{\link{contr.bayes}}} for factors) can reduce this issue. In any case
you should be mindful of this issue.
\cr\cr
See \code{\link{bayesfactor_models}} details for more info on passed models.
\cr\cr
Note that for \code{BayesFactor} models, posterior samples cannot be generated from intercept only models.
\cr\cr
This function is similar in function to \code{brms::\link[brms]{posterior_average}}.
}
\examples{
\donttest{
if (require("rstanarm") && require("see")) {
  stan_m0 <- stan_glm(extra ~ 1, data = sleep,
                      family = gaussian(),
                      refresh=0,
                      diagnostic_file = file.path(tempdir(), "df0.csv"))

  stan_m1 <- stan_glm(extra ~ group, data = sleep,
                      family = gaussian(),
                      refresh=0,
                      diagnostic_file = file.path(tempdir(), "df1.csv"))


  res <- weighted_posteriors(stan_m0, stan_m1)

  plot(eti(res))
}

# With BayesFactor and brms
if (require("BayesFactor") && require("brms")) {
  BFmods <- anovaBF(extra ~ group + ID, sleep, whichRandom = "ID")

  res <- weighted_posteriors(BFmods)[1:3]
  plot(eti(res))

  # Compare to brms::posterior_average
  fit1 <- brm(rating ~ treat + period + carry,
              data = inhaler,
              save_all_pars = TRUE)
  fit2 <- brm(rating ~ period + carry,
              data = inhaler,
              save_all_pars = TRUE)

  res_BT <- weighted_posteriors(fit1, fit2)
  res_brms <- brms::posterior_average(fit1, fit2,
                                      weights = "marglik", missing = 0)[, 1:4]

  plot(eti(res_BT))
  plot(eti(res_brms))
}
}
}
\references{
\itemize{
  \item Clyde, M., Desimone, H., & Parmigiani, G. (1996). Prediction via orthogonalized model mixing. Journal of the American Statistical Association, 91(435), 1197-1208.
  \item Hinne, M., Gronau, Q. F., van den Bergh, D., and Wagenmakers, E. (2019, March 25). A conceptual introduction to Bayesian Model Averaging. \doi{10.31234/osf.io/wgb64}
  \item Rouder, J. N., Haaf, J. M., & Vandekerckhove, J. (2018). Bayesian inference for psychology, part IV: Parameter estimation and Bayes factors. Psychonomic bulletin & review, 25(1), 102-113.
  \item van den Bergh, D., Haaf, J. M., Ly, A., Rouder, J. N., & Wagenmakers, E. J. (2019). A cautionary note on estimating effect size.
}
}
\seealso{
\code{\link{bayesfactor_inclusion}} for Bayesian model averaging.
}
