#include <R.h>
#include <Rmath.h>
#include <Rdefines.h>
#include <stdio.h>
#include <math.h>
#include "R_ext/Applic.h"

typedef double (*Fun3a)(double,double,double);
typedef double (*FunKernel2)(double,double);
typedef double (*FunKernel)(double);
typedef double (*Fun7a)(double,double*,double*,double,int,int,int);
typedef double (*Fun7p)(double,double,double,double*,double*,double*,int);


//typedef double (*Fun2d)(double,double);
//typedef double (*Fun3d)(double,double,double);
//typedef double (*Fun4p)(double,double*,double,int);
//typedef double (*Fun5p)(double,double,double,double*,int);
//typedef double (*Fun6p)(double,double,double,double*,double*,int);
//typedef double (*Fun1p)(double);


//////////////////////////////////////////////////////////////////////////    
// double Gauss_Legendre_Integration_2pts( double a, double b, double (*f)(double) ) 
// void   Gauss_Legendre_Zeros_2pts( double nodes[] )                  
//    void   Gauss_Legendre_Coefs_2pts( double wght[] )                   
//////////////////////////////////////////////////////////////////////////

//  100pts

static const double B100[] = {
    1.56289844215430828714e-02,    4.68716824215916316162e-02,
    7.80685828134366366918e-02,    1.09189203580061115002e-01,
    1.40203137236113973212e-01,    1.71080080538603274883e-01,
    2.01789864095735997236e-01,    2.32302481844973969643e-01,
    2.62588120371503479163e-01,    2.92617188038471964730e-01,
    3.22360343900529151720e-01,    3.51788526372421720979e-01,
    3.80872981624629956772e-01,    4.09585291678301542532e-01,
    4.37897402172031513100e-01,    4.65781649773358042251e-01,
    4.93210789208190933576e-01,    5.20158019881763056670e-01,
    5.46597012065094167460e-01,    5.72501932621381191292e-01,
    5.97847470247178721259e-01,    6.22608860203707771585e-01,
    6.46761908514129279840e-01,    6.70283015603141015784e-01,
    6.93149199355801965946e-01,    7.15338117573056446485e-01,
    7.36828089802020705530e-01,    7.57598118519707176062e-01,
    7.77627909649495475605e-01,    7.96897892390314476375e-01,
    8.15389238339176254384e-01,    8.33083879888400823522e-01,
    8.49964527879591284320e-01,    8.66014688497164623416e-01,
    8.81218679385018415547e-01,    8.95561644970726986709e-01,
    9.09029570982529690453e-01,    9.21609298145333952679e-01,
    9.33288535043079545942e-01,    9.44055870136255977955e-01,
    9.53900782925491742847e-01,    9.62813654255815527284e-01,
    9.70785775763706331929e-01,    9.77809358486918288561e-01,
    9.83877540706057015509e-01,    9.88984395242991747997e-01,
    9.93124937037443459632e-01,    9.96295134733125149166e-01,
    9.98491950639595818382e-01,    9.99713726773441233703e-01
};

static const double A100[] = {
    3.12554234538633569472e-02,    3.12248842548493577326e-02,
    3.11638356962099067834e-02,    3.10723374275665165874e-02,
    3.09504788504909882337e-02,    3.07983790311525904274e-02,
    3.06161865839804484966e-02,    3.04040795264548200160e-02,
    3.01622651051691449196e-02,    2.98909795933328309169e-02,
    2.95904880599126425122e-02,    2.92610841106382766198e-02,
    2.89030896011252031353e-02,    2.85168543223950979908e-02,
    2.81027556591011733175e-02,    2.76611982207923882944e-02,
    2.71926134465768801373e-02,    2.66974591835709626611e-02,
    2.61762192395456763420e-02,    2.56294029102081160751e-02,
    2.50575444815795897034e-02,    2.44612027079570527207e-02,
    2.38409602659682059633e-02,    2.31974231852541216230e-02,
    2.25312202563362727021e-02,    2.18430024162473863146e-02,
    2.11334421125276415432e-02,    2.04032326462094327666e-02,
    1.96530874944353058650e-02,    1.88837396133749045537e-02,
    1.80959407221281166640e-02,    1.72904605683235824399e-02,
    1.64680861761452126430e-02,    1.56296210775460027242e-02,
    1.47758845274413017686e-02,    1.39077107037187726882e-02,
    1.30259478929715422855e-02,    1.21314576629794974079e-02,
    1.12251140231859771176e-02,    1.03078025748689695861e-02,
    9.38041965369445795116e-03,    8.44387146966897140266e-03,
    7.49907325546471157895e-03,    6.54694845084532276405e-03,
    5.58842800386551515727e-03,    4.62445006342211935096e-03,
    3.65596120132637518238e-03,    2.68392537155348241939e-03,
    1.70939265351810523958e-03,    7.34634490505671730396e-04
};

#define NOPZ100  sizeof(B100) / sizeof(double)
#define NOZ100   NOPZ100+NOPZ100

//  96 pts
static const double B96[] = {
    1.62767448496029695789e-02,    4.88129851360497311115e-02,
    8.12974954644255589937e-02,    1.13695850110665920914e-01,
    1.45973714654896941992e-01,    1.78096882367618602759e-01,
    2.10031310460567203601e-01,    2.41743156163840012331e-01,
    2.73198812591049141485e-01,    3.04364944354496353015e-01,
    3.35208522892625422607e-01,    3.65696861472313635024e-01,
    3.95797649828908603298e-01,    4.25478988407300545368e-01,
    4.54709422167743008628e-01,    4.83457973920596359777e-01,
    5.11694177154667673569e-01,    5.39388108324357436218e-01,
    5.66510418561397168413e-01,    5.93032364777572080659e-01,
    6.18925840125468570377e-01,    6.44163403784967106801e-01,
    6.68718310043916153943e-01,    6.92564536642171561332e-01,
    7.15676812348967626199e-01,    7.38030643744400132876e-01,
    7.59602341176647498681e-01,    7.80369043867433217620e-01,
    8.00308744139140817216e-01,    8.19400310737931675557e-01,
    8.37623511228187121497e-01,    8.54959033434601455438e-01,
    8.71388505909296502900e-01,    8.86894517402420416068e-01,
    9.01460635315852341334e-01,    9.15071423120898074215e-01,
    9.27712456722308690977e-01,    9.39370339752755216934e-01,
    9.50032717784437635746e-01,    9.59688291448742539290e-01,
    9.68326828463264212168e-01,    9.75939174585136466459e-01,
    9.82517263563014677430e-01,    9.88054126329623799458e-01,
    9.92543900323762624555e-01,    9.95981842987209290633e-01,
    9.98364375863181677739e-01,    9.99689503883230766825e-01
};

static const double A96[] = {
    3.25506144923631662418e-02,    3.25161187138688359885e-02,
    3.24471637140642693631e-02,    3.23438225685759284293e-02,
    3.22062047940302506674e-02,    3.20344562319926632176e-02,
    3.18287588944110065352e-02,    3.15893307707271685576e-02,
    3.13164255968613558124e-02,    3.10103325863138374230e-02,
    3.06713761236691490147e-02,    3.02999154208275937943e-02,
    2.98963441363283859846e-02,    2.94610899581679059697e-02,
    2.89946141505552365432e-02,    2.84974110650853856455e-02,
    2.79700076168483344400e-02,    2.74129627260292428232e-02,
    2.68268667255917621977e-02,    2.62123407356724139131e-02,
    2.55700360053493614996e-02,    2.49006332224836102884e-02,
    2.42048417923646912830e-02,    2.34833990859262198430e-02,
    2.27370696583293740018e-02,    2.19666444387443491945e-02,
    2.11729398921912989884e-02,    2.03567971543333245953e-02,
    1.95190811401450224097e-02,    1.86606796274114673859e-02,
    1.77825023160452608374e-02,    1.68854798642451724498e-02,
    1.59705629025622913804e-02,    1.50387210269949380059e-02,
    1.40909417723148609158e-02,    1.31282295669615726374e-02,
    1.21516046710883196352e-02,    1.11621020998384985916e-02,
    1.01607705350084157574e-02,    9.14867123078338663265e-03,
    8.12687692569875921698e-03,    7.09647079115386526927e-03,
    6.05854550423596168331e-03,    5.01420274292751769241e-03,
    3.96455433844468667376e-03,    2.91073181793494640833e-03,
    1.85396078894692173237e-03,    7.96792065552012429429e-04
};
#define NOPZ96  sizeof(B96) / sizeof(double)
#define NOZ96   NOPZ96+NOPZ96
//  82 pts

static const double B82[] = {
    1.90384554820068137613e-02,    5.70877625059269319014e-02,
    9.50542977326448739519e-02,    1.32883013316257525763e-01,
    1.70519061236190159767e-01,    2.07907872821592350966e-01,
    2.44995237870704537844e-01,    2.81727383250480102378e-01,
    3.18051050862501505004e-01,    3.53913574862147936411e-01,
    3.89262958019054815918e-01,    4.24047947108150722153e-01,
    4.58218107221963197832e-01,    4.91723894896449322455e-01,
    5.24516729944327767963e-01,    5.56549065891763812346e-01,
    5.87774458916284867180e-01,    6.18147635185978682035e-01,
    6.47624556502346598549e-01,    6.76162484151647024571e-01,
    7.03720040872166678473e-01,    7.30257270847596245034e-01,
    7.55735697639560417317e-01,    7.80118379975358245207e-01,
    8.03369965310108299287e-01,    8.25456741085767656024e-01,
    8.46346683612913701800e-01,    8.66009504504765448567e-01,
    8.84416694596724708250e-01,    9.01541565288841258185e-01,
    9.17359287253280107145e-01,    9.31846926454622887881e-01,
    9.44983477438960489203e-01,    9.56749893861610903139e-01,
    9.67129116252590696019e-01,    9.76106097094498828710e-01,
    9.83667823522007603353e-01,    9.89803338839908949953e-01,
    9.94503768378075628926e-01,    9.97762385826385862536e-01,
    9.99575191501652648887e-01
};

static const double A82[] = {
    3.80723096401418712092e-02,    3.80171084314352699063e-02,
    3.79067860505057847784e-02,    3.77415024542758696721e-02,
    3.75214972881850208729e-02,    3.72470895387276641869e-02,
    3.69186770709544569983e-02,    3.65367360516076528430e-02,
    3.61018202587270230770e-02,    3.56145602787274726798e-02,
    3.50756625921126903860e-02,    3.44859085491507055064e-02,
    3.38461532369968587430e-02,    3.31573242399072113277e-02,
    3.24204202943406050765e-02,    3.16365098409002455390e-02,
    3.08067294752156298136e-02,    2.99322823000127246344e-02,
    2.90144361807644039617e-02,    2.80545219074542304725e-02,
    2.70539312651247715195e-02,    2.60141150160170237537e-02,
    2.49365807962407551563e-02,    2.38228909300478263420e-02,
    2.26746601649141031023e-02,    2.14935533307748440428e-02,
    2.02812829269121589024e-02,    1.90396066401789250728e-02,
    1.77703247984984071478e-02,    1.64752777639837088909e-02,
    1.51563432707625617883e-02,    1.38154337141264593880e-02,
    1.24544934011421046793e-02,    1.10754957817598963206e-02,
    9.68044070437107373722e-03,    8.27135181838368560408e-03,
    6.85027453418352618420e-03,    5.41927623244676509051e-03,
    3.98045793785607461882e-03,    2.53605469685610610985e-03,
    1.09011859527583086610e-03
};

#define NOPZ82  sizeof(B82) / sizeof(double)
#define NOZ82   NOPZ82+NOPZ82
//  64 pts

static const double B64[] = {
    2.43502926634244325092e-02,    7.29931217877990394521e-02,
    1.21462819296120554468e-01,    1.69644420423992818036e-01,
    2.17423643740007084148e-01,    2.64687162208767416384e-01,
    3.11322871990210956161e-01,    3.57220158337668115941e-01,
    4.02270157963991603693e-01,    4.46366017253464087978e-01,
    4.89403145707052957474e-01,    5.31279464019894545634e-01,
    5.71895646202634034291e-01,    6.11155355172393250241e-01,
    6.48965471254657339884e-01,    6.85236313054233242559e-01,
    7.19881850171610826840e-01,    7.52819907260531896590e-01,
    7.83972358943341407619e-01,    8.13265315122797559746e-01,
    8.40629296252580362743e-01,    8.65999398154092819759e-01,
    8.89315445995114105856e-01,    9.10522137078502805780e-01,
    9.29569172131939575846e-01,    9.46411374858402816069e-01,
    9.61008799652053718944e-01,    9.73326827789910963733e-01,
    9.83336253884625956939e-01,    9.91013371476744320732e-01,
    9.96340116771955279355e-01,    9.99305041735772139465e-01
};

static const double A64[] = {
    4.86909570091397203845e-02,    4.85754674415034269347e-02,
    4.83447622348029571711e-02,    4.79993885964583077283e-02,
    4.75401657148303086630e-02,    4.69681828162100173254e-02,
    4.62847965813144172952e-02,    4.54916279274181444806e-02,
    4.45905581637565630589e-02,    4.35837245293234533762e-02,
    4.24735151236535890083e-02,    4.12625632426235286114e-02,
    3.99537411327203413872e-02,    3.85501531786156291275e-02,
    3.70551285402400460401e-02,    3.54722132568823838103e-02,
    3.38051618371416093907e-02,    3.20579283548515535845e-02,
    3.02346570724024788683e-02,    2.83396726142594832269e-02,
    2.63774697150546586725e-02,    2.43527025687108733385e-02,
    2.22701738083832541592e-02,    2.01348231535302093718e-02,
    1.79517157756973430852e-02,    1.57260304760247193221e-02,
    1.34630478967186425983e-02,    1.11681394601311288184e-02,
    8.84675982636394772300e-03,    6.50445796897836285625e-03,
    4.14703326056246763525e-03,    1.78328072169643294728e-03
};

#define NOPZ64  sizeof(B64) / sizeof(double)
#define NOZ64   NOPZ64+NOPZ64
//  56 pts
static const double B56[] = {
    2.77970352872754370942e-02,    8.33051868224353744437e-02,
    1.38555846810376242012e-01,    1.93378238635275258246e-01,
    2.47602909434337203968e-01,    3.01062253867220669048e-01,
    3.53591032174954520968e-01,    4.05026880927091278110e-01,
    4.55210814878459578940e-01,    5.03987718384381714188e-01,
    5.51206824855534618750e-01,    5.96722182770663320077e-01,
    6.40393106807006894294e-01,    6.82084612694470455497e-01,
    7.21667834450188083544e-01,    7.59020422705128902207e-01,
    7.94026922893866498037e-01,    8.26579132142881651667e-01,
    8.56576433762748635397e-01,    8.83926108327827540810e-01,
    9.08543620420655490867e-01,    9.30352880247496300556e-01,
    9.49286479561962635648e-01,    9.65285901905490183650e-01,
    9.78301709140256383354e-01,    9.88293715540161511084e-01,
    9.95231226081069747231e-01,    9.99094343801465584350e-01
};

static const double A56[] = {
    5.55797463065143958469e-02,    5.54079525032451232193e-02,
    5.50648959017624257965e-02,    5.45516368708894210621e-02,
    5.38697618657144857091e-02,    5.30213785240107639691e-02,
    5.20091091517413998428e-02,    5.08360826177984805586e-02,
    4.95059246830475789197e-02,    4.80227467936002581221e-02,
    4.63911333730018967605e-02,    4.46161276526922832122e-02,
    4.27032160846670865107e-02,    4.06583113847445178803e-02,
    3.84877342592476624857e-02,    3.61981938723151860357e-02,
    3.37967671156117612965e-02,    3.12908767473104478680e-02,
    2.86882684738227417295e-02,    2.59969870583919521920e-02,
    2.32253515625653169375e-02,    2.03819298824025726345e-02,
    1.74755129114009465053e-02,    1.45150892780214718081e-02,
    1.15098243403833821739e-02,    8.46906316330788766164e-03,
    5.40252224601533776151e-03,    2.32385537577321550118e-03
};

#define NOPZ56  sizeof(B56) / sizeof(double)
#define NOZ56   NOPZ56+NOPZ56
//  48 pts
static const double B48[] = {
    3.23801709628693620343e-02,    9.70046992094626989322e-02,
    1.61222356068891718055e-01,    2.24763790394689061224e-01,
    2.87362487355455576728e-01,    3.48755886292160738148e-01,
    4.08686481990716729925e-01,    4.66902904750958404535e-01,
    5.23160974722233033658e-01,    5.77224726083972703838e-01,
    6.28867396776513624013e-01,    6.77872379632663905208e-01,
    7.24034130923814654658e-01,    7.67159032515740339276e-01,
    8.07066204029442627087e-01,    8.43588261624393530704e-01,
    8.76572020274247885885e-01,    9.05879136715569672805e-01,
    9.31386690706554333107e-01,    9.52987703160430860724e-01,
    9.70591592546247250472e-01,    9.84124583722826857765e-01,
    9.93530172266350757526e-01,    9.98771007252426118580e-01
};

static const double A48[] = {
    6.47376968126839225006e-02,    6.44661644359500822082e-02,
    6.39242385846481866207e-02,    6.31141922862540256548e-02,
    6.20394231598926639029e-02,    6.07044391658938800517e-02,
    5.91148396983956357477e-02,    5.72772921004032157044e-02,
    5.51995036999841628676e-02,    5.28901894851936670964e-02,
    5.03590355538544749590e-02,    4.76166584924904748267e-02,
    4.46745608566942804201e-02,    4.15450829434647492133e-02,
    3.82413510658307063158e-02,    3.47772225647704388909e-02,
    3.11672278327980889025e-02,    2.74265097083569482001e-02,
    2.35707608393243791410e-02,    1.96161604573555278142e-02,
    1.55793157229438487279e-02,    1.14772345792345394895e-02,
    7.32755390127626210220e-03,    3.15334605230583863260e-03
};

#define NOPZ48  sizeof(B48) / sizeof(double)
#define NOZ48   NOPZ48+NOPZ48
//  40 pts
static const double B40[] = {
    3.87724175060508219329e-02,    1.16084070675255208481e-01,
    1.92697580701371099719e-01,    2.68152185007253681152e-01,
    3.41994090825758473008e-01,    4.13779204371605001525e-01,
    4.83075801686178712903e-01,    5.49467125095128202056e-01,
    6.12553889667980237972e-01,    6.71956684614179548364e-01,
    7.27318255189927103277e-01,    7.78305651426519387712e-01,
    8.24612230833311663197e-01,    8.65959503212259503824e-01,
    9.02098806968874296732e-01,    9.32812808278676533383e-01,
    9.57916819213791655824e-01,    9.77259949983774262679e-01,
    9.90726238699457006464e-01,    9.98237709710559200369e-01
};

static const double A40[] = {
    7.75059479784248112668e-02,    7.70398181642479655914e-02,
    7.61103619006262423723e-02,    7.47231690579682641980e-02,
    7.28865823958040590609e-02,    7.06116473912867796979e-02,
    6.79120458152339038265e-02,    6.48040134566010380719e-02,
    6.13062424929289391679e-02,    5.74397690993915513665e-02,
    5.32278469839368243566e-02,    4.86958076350722320604e-02,
    4.38709081856732719923e-02,    3.87821679744720176413e-02,
    3.34601952825478473933e-02,    2.79370069800234010984e-02,
    2.22458491941669572615e-02,    1.64210583819078887131e-02,
    1.04982845311528136146e-02,    4.52127709853319125846e-03
};

#define NOPZ40  sizeof(B40) / sizeof(double)
#define NOZ40   NOPZ40+NOPZ40
//  32 pts
static const double B32[] = {
    4.83076656877383162364e-02,    1.44471961582796493484e-01,
    2.39287362252137074544e-01,    3.31868602282127649782e-01,
    4.21351276130635345353e-01,    5.06899908932229390044e-01,
    5.87715757240762329066e-01,    6.63044266930215200960e-01,
    7.32182118740289680412e-01,    7.94483795967942406965e-01,
    8.49367613732569970160e-01,    8.96321155766052123971e-01,
    9.34906075937739689159e-01,    9.64762255587506430761e-01,
    9.85611511545268335400e-01,    9.97263861849481563534e-01
};

static const double A32[] = {
    9.65400885147278005666e-02,    9.56387200792748594185e-02,
    9.38443990808045656367e-02,    9.11738786957638847129e-02,
    8.76520930044038111450e-02,    8.33119242269467552223e-02,
    7.81938957870703064685e-02,    7.23457941088485062287e-02,
    6.58222227763618468406e-02,    5.86840934785355471448e-02,
    5.09980592623761761959e-02,    4.28358980222266806557e-02,
    3.42738629130214331033e-02,    2.53920653092620594561e-02,
    1.62743947309056706058e-02,    7.01861000947009660028e-03
};

#define NOPZ32  sizeof(B32) / sizeof(double)
#define NOZ32   NOPZ32+NOPZ32
//  28 pts
static const double B28[] = {
    5.50792898840342704250e-02,    1.64569282133380771286e-01,
    2.72061627635178077665e-01,    3.76251516089078710218e-01,
    4.75874224955118261030e-01,    5.69720471811401719313e-01,
    6.56651094038864961222e-01,    7.35610878013631772022e-01,
    8.05641370917179171424e-01,    8.65892522574395048925e-01,
    9.15633026392132073868e-01,    9.54259280628938197247e-01,
    9.81303165370872753704e-01,    9.96442497573954449940e-01
};

static const double A28[] = {
    1.10047013016475196280e-01,    1.08711192258294135255e-01,
    1.06055765922846417912e-01,    1.02112967578060769818e-01,
    9.69306579979299158501e-02,    9.05717443930328409426e-02,
    8.31134172289012183890e-02,    7.46462142345687790248e-02,
    6.52729239669995957946e-02,    5.51073456757167454319e-02,
    4.42729347590042278397e-02,    3.29014277823043799770e-02,
    2.11321125927712597509e-02,    9.12428259309451773855e-03
};

#define NOPZ28  sizeof(B28) / sizeof(double)
#define NOZ28   NOPZ28+NOPZ28
//  24 pts
static const double B24[] = {
    6.40568928626056260827e-02,    1.91118867473616309153e-01,
    3.15042679696163374398e-01,    4.33793507626045138478e-01,
    5.45421471388839535649e-01,    6.48093651936975569268e-01,
    7.40124191578554364260e-01,    8.20001985973902921981e-01,
    8.86415527004401034190e-01,    9.38274552002732758539e-01,
    9.74728555971309498199e-01,    9.95187219997021360195e-01
};

static const double A24[] = {
    1.27938195346752156976e-01,    1.25837456346828296117e-01,
    1.21670472927803391202e-01,    1.15505668053725601353e-01,
    1.07444270115965634785e-01,    9.76186521041138882720e-02,
    8.61901615319532759152e-02,    7.33464814110803057346e-02,
    5.92985849154367807461e-02,    4.42774388174198061695e-02,
    2.85313886289336631809e-02,    1.23412297999871995469e-02
};

#define NOPZ24  sizeof(B24) / sizeof(double)
#define NOZ24   NOPZ24+NOPZ24
//  20 pts
static const double B20[] = {
    7.65265211334973337513e-02,    2.27785851141645078076e-01,
    3.73706088715419560662e-01,    5.10867001950827097985e-01,
    6.36053680726515025467e-01,    7.46331906460150792634e-01,
    8.39116971822218823420e-01,    9.12234428251325905857e-01,
    9.63971927277913791287e-01,    9.93128599185094924776e-01
};

static const double A20[] = {
    1.52753387130725850699e-01,    1.49172986472603746785e-01,
    1.42096109318382051326e-01,    1.31688638449176626902e-01,
    1.18194531961518417310e-01,    1.01930119817240435039e-01,
    8.32767415767047487264e-02,    6.26720483341090635663e-02,
    4.06014298003869413320e-02,    1.76140071391521183115e-02
};

#define NOPZ20  sizeof(B20) / sizeof(double)
#define NOZ20   NOPZ20+NOPZ20
//  16 pts
static const double B16[] = {
    9.50125098376374401877e-02,    2.81603550779258913231e-01,
    4.58016777657227386350e-01,    6.17876244402643748452e-01,
    7.55404408355003033891e-01,    8.65631202387831743866e-01,
    9.44575023073232576090e-01,    9.89400934991649932601e-01
};

static const double A16[] = {
    1.89450610455068496287e-01,    1.82603415044923588872e-01,
    1.69156519395002538183e-01,    1.49595988816576732080e-01,
    1.24628971255533872056e-01,    9.51585116824927848073e-02,
    6.22535239386478928628e-02,    2.71524594117540948514e-02
};

#define NOPZ16  sizeof(B16) / sizeof(double)
#define NOZ16   NOPZ16+NOPZ16
//  14 pts
static const double B14[] = {
    1.08054948707343662065e-01,    3.19112368927889760448e-01,
    5.15248636358154091979e-01,    6.87292904811685470155e-01,
    8.27201315069764993163e-01,    9.28434883663573517338e-01,
    9.86283808696812338852e-01
};

static const double A14[] = {
    2.15263853463157790196e-01,    2.05198463721295603959e-01,
    1.85538397477937813746e-01,    1.57203167158193534569e-01,
    1.21518570687903184687e-01,    8.01580871597602098063e-02,
    3.51194603317518630327e-02
};

#define NOPZ14  sizeof(B14) / sizeof(double)
#define NOZ14   NOPZ14+NOPZ14
//  12 pts
static const double B12[] = {
    1.25233408511468915478e-01,    3.67831498998180193757e-01,
    5.87317954286617447312e-01,    7.69902674194304687059e-01,
    9.04117256370474856682e-01,    9.81560634246719250712e-01
};

static const double A12[] = {
    2.49147045813402785006e-01,    2.33492536538354808758e-01,
    2.03167426723065921743e-01,    1.60078328543346226338e-01,
    1.06939325995318430960e-01,    4.71753363865118271952e-02
};

#define NOPZ12  sizeof(B12) / sizeof(double)
#define NOZ12   NOPZ12+NOPZ12
//  10 pts
static const double B10[] = {
    1.48874338981631210881e-01,    4.33395394129247190794e-01,
    6.79409568299024406207e-01,    8.65063366688984510759e-01,
    9.73906528517171720066e-01
};

static const double A10[] = {
    2.95524224714752870187e-01,    2.69266719309996355105e-01,
    2.19086362515982044000e-01,    1.49451349150580593150e-01,
    6.66713443086881375920e-02
};

#define NOPZ10  sizeof(B10) / sizeof(double)
#define NOZ10   NOPZ10+NOPZ10
//  9 pts
static const double B9[] = {
    0.00000000000000000000e+00,    8.36031107326635794313e-01,
    9.68160239507626089810e-01,    3.24253423403808929042e-01,
    6.13371432700590397285e-01
};

static const double A9[] = {
    3.30239355001259763154e-01,    1.80648160694857404059e-01,
    8.12743883615744119737e-02,    3.12347077040002840057e-01,
    2.60610696402935462313e-01
};

#define NOPZ9  sizeof(B9) / sizeof(double)
#define NOZ9   NOPZ9+NOPZ9
//  8 pts
static const double B8[] = {
    1.83434642495649804936e-01,    5.25532409916328985830e-01,
    7.96666477413626739567e-01,    9.60289856497536231661e-01
};

static const double A8[] = {
    3.62683783378361982976e-01,    3.13706645877887287338e-01,
    2.22381034453374470546e-01,    1.01228536290376259154e-01
};
#define NOPZ8  sizeof(B8) / sizeof(double)
#define NOZ8   NOPZ8+NOPZ8
//  7 pts
static const double B7[] = {
    0.00000000000000000000e+00,    4.05845151377397166917e-01,
    7.41531185599394439864e-01,    9.49107912342758524541e-01
};

static const double A7[] = {
    4.17959183673469387749e-01,    3.81830050505118944961e-01,
    2.79705391489276667890e-01,    1.29484966168869693274e-01
};

#define NOPZ7  sizeof(B7) / sizeof(double)
#define NOZ7   NOPZ7+NOPZ7
//  6 pts
static const double B6[] = {
    6.61209386466264513688e-01,    2.38619186083196908630e-01,
    9.32469514203152027832e-01
};

static const double A6[] = {
    3.60761573048138607569e-01,    4.67913934572691047389e-01,
    1.71324492379170345043e-01
};

#define NOPZ6  sizeof(B6) / sizeof(double)
#define NOZ6   NOPZ6+NOPZ6
//  5 pts
static const double B5[] = {
    0.00000000000000000000e+00,    5.38469310105683091018e-01,
    9.06179845938663992811e-01
};

static const double A5[] = {
    5.68888888888888888883e-01,    4.78628670499366468030e-01,
    2.36926885056189087515e-01
};

#define NOPZ5  sizeof(B5) / sizeof(double)
#define NOZ5   NOPZ5+NOPZ5
//  4 pts
static const double B4[] = {
    3.39981043584856264792e-01,    8.61136311594052575248e-01
};

static const double A4[] = {
    6.52145154862546142644e-01,    3.47854845137453857383e-01
};

#define NOPZ4  sizeof(B4) / sizeof(double)
#define NOZ4   NOPZ4+NOPZ4
//  3 pts
static const double B3[] = {
    0.00000000000000000000e+00,    7.74596669241483377010e-01
};

static const double A3[] = {
    8.88888888888888888877e-01,    5.55555555555555555562e-01
};

#define NOPZ3  sizeof(B3) / sizeof(double)
#define NOZ3   NOPZ3+NOPZ3

//  2 pts
static const double B2 = 5.77350269189625764507e-01;

static const double A2 =  1.0;

#define NOPZ2  sizeof(B2) / sizeof(double)
#define NOZ2   NOPZ2+NOPZ2

double ftube(double x, double kappa,double alpha){
  return 2.-2.*pnorm(x,0.0,1.0,1,0)+kappa/M_PI*exp(-.5*x*x)-alpha;
}

double dftube(double x, double kappa){
  return -2.*dnorm(x,0.0,1.0,0)-x*kappa/M_PI*exp(-.5*x*x);
}

double NRsolve(double x0, 
	       double (*g)(double,double,double), 
	       double (*pg)(double,double),
	       double kappa, double alpha)
{
  int i=0,imax=1000;
  double dx,tol=0.00000001;
  dx=1.0;
  while(((dx>tol)|(dx/fabs(x0)>tol))&(i<imax)){
    dx =  (*g)(x0,kappa,alpha)/(*pg)(x0,kappa);
    x0 -= dx; dx = fabs(dx);
    i++;
  };
  if(i>=imax) x0=0.0;
  return x0;
}

double GLISupp(double a, double b, double (*f)(double,double,double),
	       double x,double sig)
{
   double integral = 0.0; 
   double c = 0.5 * (b - a);
   double d = 0.5 * (b + a);
   double dum;
   const double *pB = &B10[NOPZ10 - 1];
   const double *pA = &A10[NOPZ10 - 1];

   for (; pB >= B10; pA--, pB--) {
      dum = c * *pB;
      integral += *pA * ( (*f)(d - dum,x,sig) + (*f)(d + dum,x,sig) );
   }

   return c * integral;
}

double kgauss(double x){
  return dnorm(x,0.0,1.0,0);
}

double dkgauss(double x){
  return -x * dnorm(x,0.0,1.0,0);
}

double ddkgauss(double x){
  return (x*x-1.) * dnorm(x,0.0,1.0,0);
}


double funSupport(double t,double x,double sigh){
  //sigh = sig/h for the parsimony of parameters 
  double t2=t*t;
  return cos(t*x)*pow(1.0-t2,3.0)*exp(0.5*sigh*sigh*t2);
}

double fundSupport(double t,double x,double sigh){
  //sigh = sig/h for the parsimony of parameters 
  double t2=t*t;
  return -t*sin(t*x)*pow(1.0-t2,3.0)*exp(0.5*sigh*sigh*t2);
}

double funddSupport(double t,double x,double sigh){
  //sigh = sig/h for the parsimony of parameters 
  double t2=t*t;
  return -t2*cos(t*x)*pow(1.0-t2,3.0)*exp(0.5*sigh*sigh*t2);
}

double ksupport(double x,double sigh){
  //sig = sig/h for the parsimony of parameters 
  Fun3a f[1];
  f[0] = funSupport;
  return GLISupp(0.0,1.0,f[0],x,sigh)/PI;
}

double dksupport(double x,double sigh){
  //sig = sig/h for the parsimony of parameters 
  Fun3a f[1];
  f[0] = fundSupport;
  return GLISupp(0.0,1.0,f[0],x,sigh)/PI;
}

double ddksupport(double x,double sigh){
  //sig = sig/h for the parsimony of parameters 
  Fun3a f[1];
  f[0] = funddSupport;
  return GLISupp(0.0,1.0,f[0],x,sigh)/PI;
}

double klaplace(double x,double sigh){
  double out;
  out = dnorm(x,0.0,1.0,0)*(1.0+pow(sigh,2.0)*(1.0-x*x));
  if(out<0) out=0.0;
  return out;
}

double dklaplace(double x,double sigh){
  double out,sigh2,x2;
  sigh2 = sigh*sigh;
  x2 = x * x;
  
  if(x2 < 1.-1./sigh2){
    out = 0.0000001;
  }else{
    out = sigh2 * (1.0 + x2) - 1.0;
    out = x * out * dnorm(x,0.0,1.0,0);
  }
  return out;
}

double ddklaplace(double x,double sigh){
  double out,sigh2,x2;
  sigh2 = sigh*sigh;
  x2 = x * x;
  
  if(x2 < 1.-1./sigh2){
    out = 0.0000001;
  }else{
    out = (1.0 + 2.0 * sigh2) * x2 - sigh2 * x2*x2 + sigh2 -1.0;
    out = out * dnorm(x,0.0,1.0,0);
  }
  return out;
}

double funKappa(double t,double *x,double *sigh, double h, 
		int n, int kernel, int stype){

  FunKernel2 fK[2],dK[2];
  fK[0] = ksupport;
  fK[1] = klaplace;
  dK[0] = dksupport;
  dK[1] = dklaplace;

  int i;
  double Sn1,Sn2,deSn1,deSn2,dx,z,Sbi,Sdbi;
  double bi[n],dbi[n];
  double t0;
  double SLl,Sll,SLL;
  Sn1 = 0.0; Sn2 = 0.0; deSn1 = 0.0; deSn2 = 0.0;

  if(stype==0){
    for(i=0;i<n;i++){
      dx = t-x[i];
      z = dx/h;
      t0 = sigh[i];
      bi[i] = fK[kernel](z,t0); //store K to compute bi later on
      dbi[i] = dK[kernel](z,t0)/h;//store K' to compute bi later on
      t0 = bi[i] * dx;
      Sn1 += t0;
      Sn2 += t0 * dx;
      deSn1 += dbi[i] * dx + bi[i]; 
      deSn2 += dbi[i] * dx * dx + 2.0 * t0;
    }
    
    Sbi = 0.0; Sdbi = 0.0;
    for(i=0;i<n;i++){
      dx = t-x[i];
      dbi[i] = dbi[i]*(Sn2-Sn1*dx)+ bi[i]*(deSn2-Sn1-deSn1*dx);
      bi[i] *= Sn2-Sn1*dx;
      Sbi += bi[i];
      Sdbi += dbi[i];
    }
    SLl=0.0; Sll=0.0;SLL=0.0;
    for(i=0;i<n;i++){
      dx = t-x[i];
      dbi[i] = dbi[i]/Sbi-bi[i]*Sdbi/Sbi;//reuse dbi for dli
      bi[i] = bi[i] / Sbi;//reuse bi for li
      SLL += bi[i] * bi[i];
      SLl += bi[i] * dbi[i];
      Sll += dbi[i] * dbi[i];
    }
    z = Sll/SLL - SLl*SLl/SLL/SLL; 
  }else{
    // bi is the kernel K(), and dbi is K'()
    Sbi = 0.0; Sdbi = 0.0;
    for(i=0;i<n;i++){
      dx = t-x[i];
      z = dx/h;
      t0 = sigh[i];
      bi[i] = fK[kernel](z,t0); 
      dbi[i] = dK[kernel](z,t0)/h;
      Sbi += bi[i];
      Sdbi += dbi[i];
    }
    Sn1 = 0.0; // ||m||
    Sn2 = 0.0; // ||m||'
    for(i=0;i<n;i++){//reuse for mi and mi'
      bi[i] /= Sbi; // m
      t0 = dbi[i]*Sbi - bi[i]*Sdbi; // m'
      dbi[i] = t0/(Sbi*Sbi);
      Sn1 += bi[i]*bi[i];
      Sn2 += bi[i] * dbi[i];
    }
    Sn1 = sqrt(Sn1); //||m||
    Sn2 /= Sn1; // ||m||'
    z = 0.0; // ||T'||
    for(i=0;i<n;i++){
      t0 = dbi[i]*Sn1 - bi[i]*Sn2; // sum(T'^2)
      z += t0 * t0;
    }
    z /= Sn1*Sn1;

  }
  if((z<0)|(ISNAN(z))) z=0.000001;
  return sqrt(z);

}


double GLIKappa(double a, double b, 
		double (*f)(double, double*,double*,double,int,int,int),
		double *x,double *sig,double h,int n, int kernel, int stype)
{
  double integral = 0.0; 
   double c = 0.5 * (b - a);
   double d = 0.5 * (b + a);
   double dum;
   const double *pB = &B10[NOPZ10 - 1];
   const double *pA = &A10[NOPZ10 - 1];

   for (; pB >= B10; pA--, pB--) {
      dum = c * *pB;
      integral += *pA * ( (*f)(d - dum,x,sig,h,n,kernel,stype) 
			  + (*f)(d + dum,x,sig,h,n,kernel,stype) );
   }

   return c * integral;
}

double funKappa2(double t,double *x, double h, 
		int n, int kernel, int stype){

  FunKernel fK[1],dK[1];
  fK[0] = kgauss;
  dK[0] = dkgauss;

  int i;
  double Sn1,Sn2,deSn1,deSn2,dx,z,Sbi,Sdbi;
  double bi[n],dbi[n];
  double t0;
  double SLl,Sll,SLL;
  Sn1 = 0.0; Sn2 = 0.0; deSn1 = 0.0; deSn2 = 0.0;

  if(stype==0){
    for(i=0;i<n;i++){
      dx = t-x[i];
      z = dx/h;
      bi[i] = fK[kernel](z); //store K to compute bi later on
      dbi[i] = dK[kernel](z)/h;//store K' to compute bi later on
      t0 = bi[i] * dx;
      Sn1 += t0;
      Sn2 += t0 * dx;
      deSn1 += dbi[i] * dx + bi[i]; 
      deSn2 += dbi[i] * dx * dx + 2.0 * t0;
    }
    
    Sbi = 0.0; Sdbi = 0.0;
    for(i=0;i<n;i++){
      dx = t-x[i];
      dbi[i] = dbi[i]*(Sn2-Sn1*dx)+ bi[i]*(deSn2-Sn1-deSn1*dx);
      bi[i] *= Sn2-Sn1*dx;
      Sbi += bi[i];
      Sdbi += dbi[i];
    }
    SLl=0.0; Sll=0.0;SLL=0.0;
    for(i=0;i<n;i++){
      dx = t-x[i];
      dbi[i] = dbi[i]/Sbi-bi[i]*Sdbi/Sbi;//reuse dbi for dli
      bi[i] = bi[i] / Sbi;//reuse bi for li
      SLL += bi[i] * bi[i];
      SLl += bi[i] * dbi[i];
      Sll += dbi[i] * dbi[i];
    }
    z = Sll/SLL - SLl*SLl/SLL/SLL; 
  }else{
    // bi is the kernel K(), and dbi is K'()
    Sbi = 0.0; Sdbi = 0.0;
    for(i=0;i<n;i++){
      dx = t-x[i];
      z = dx/h;
      bi[i] = fK[kernel](z); 
      dbi[i] = dK[kernel](z)/h;
      Sbi += bi[i];
      Sdbi += dbi[i];
    }
    Sn1 = 0.0; // ||m||
    Sn2 = 0.0; // ||m||'
    for(i=0;i<n;i++){//reuse for mi and mi'
      bi[i] /= Sbi; // m
      t0 = dbi[i]*Sbi - bi[i]*Sdbi; // m'
      dbi[i] = t0/(Sbi*Sbi);
      Sn1 += bi[i]*bi[i];
      Sn2 += bi[i] * dbi[i];
    }
    Sn1 = sqrt(Sn1); //||m||
    Sn2 /= Sn1; // ||m||'
    z = 0.0; // ||T'||
    for(i=0;i<n;i++){
      t0 = dbi[i]*Sn1 - bi[i]*Sn2; // sum(T'^2)
      z += t0 * t0;
    }
    z /= Sn1*Sn1;

  }
  if((z<0)|(ISNAN(z))) z=0.000001;
  return sqrt(z);

}

double GLIKappa2(double a, double b, 
		double (*f)(double, double*,double,int,int,int),
		double *x,double h,int n, int kernel, int stype)
{
  double integral = 0.0; 
   double c = 0.5 * (b - a);
   double d = 0.5 * (b + a);
   double dum;
   const double *pB = &B10[NOPZ10 - 1];
   const double *pA = &A10[NOPZ10 - 1];

   for (; pB >= B10; pA--, pB--) {
      dum = c * *pB;
      integral += *pA * ( (*f)(d - dum,x,h,n,kernel,stype) 
			  + (*f)(d + dum,x,h,n,kernel,stype) );
   }

   return c * integral;
}


void menpr(double *y, double *x, double *sig, int *size, int *kernel, 
	 int *stype, double *bw, double *x0, int *ngrid, double *alpha,
	 double *ell, double *kappa, double *tubec)
{
  int i,j,m,n;
  n = size[0];  m = ngrid[0];
  double sigh[n], h = bw[0],fx[m],fk[n],b[n];
  double S1,S2,t1,bsum;

  FunKernel2 f[2];
  f[0] = ksupport;
  f[1] = klaplace;
  
  for(i=0;i<n;i++) sigh[i] = sig[i]/h;

  if(stype[0]==0){

    for(j=0;j<m;j++){
      for(i=0;i<n;i++)
	fk[i] = f[kernel[0]]((x[i]-x0[j])/h,sigh[i]);
      S1 = 0.0;  S2 = 0.0;
      for(i=0;i<n;i++){
	t1 = (x[i] - x0[j]);
	S1 += fk[i] * t1;
	S2 += fk[i] * t1 * t1;
      }
      for(i=0;i<n;i++){
	t1 = (x[i] - x0[j]);
	b[i] = fabs(fk[i]*(S2-t1*S1));
      }
      //  to compute ||l(x)||
      bsum = 0.0;
      t1 = 0.0;  // used to compute SS(b[i])
      fx[j] = 0.0;
      for(i=0;i<n;i++){
	bsum += b[i];
	t1 += b[i]*b[i];
	fx[j] += b[i] * y[i];
      }
      fx[j] /= bsum;
      ell[j] = sqrt(t1)/bsum;
      kappa[0] = GLIKappa(x0[0],x0[m-1],funKappa,x,sigh,h,n,kernel[0],stype[0]); 
      t1 = NRsolve(2.0, ftube, dftube, kappa[0], alpha[0]);
      tubec[0] = t1;
      ell[j] *= t1; 
    }
  }else{
    for(j=0;j<m;j++){
      for(i=0;i<n;i++)
	b[i] = f[kernel[0]]((x[i]-x0[j])/h,sigh[i]);
      //  to compute ||l(x)||
      bsum = 0.0;
      t1 = 0.0;  // used to compute SS(b[i])
      fx[j] = 0.0;
      for(i=0;i<n;i++){
	bsum += b[i];
	t1 += b[i]*b[i];
	fx[j] += b[i] * y[i];
      }
      fx[j] /= bsum;
      ell[j] = sqrt(t1)/bsum;
      kappa[0] = GLIKappa(x0[0],x0[m-1],funKappa,x,sigh,h,n,kernel[0],stype[0]); 
      t1 = NRsolve(2.0, ftube, dftube, kappa[0], alpha[0]);
      tubec[0] = t1;
      ell[j] *= t1; 
    }
    
  }
  
  for(j=0;j<m;j++){
    x0[j] = fx[j];
  }
}

void mehopt(double *y, double *x, double *sig, int *size, 
	  int *kernel, int *stype, double *bw,
	    int *m, double *hs, double *lscv)
{
  int i,j,k,n;
  n = size[0];
  double sigh[n], h = bw[0],fx[n],fk[n],b[n];
  double S1,S2,t1,bsum;
  double dh,h0,lcvscore,mse,lcvmax=9999999999999.;

  FunKernel2 f[2];
  f[0] = ksupport;
  f[1] = klaplace;

  h0=h;
  dh = 0.025*h;
  if(m[0]>1) h = 0.6*h; else h=h0-dh;
  
  for(k=0;k<m[0];k++){  //search for h, record as h0
    h += dh;
    for(i=0;i<n;i++) sigh[i] = sig[i]/h;
    lcvscore = 0.0;
    mse = 0.0;
    for(j=0;j<n;j++){
      for(i=0;i<n;i++)
	fk[i] = f[kernel[0]]((x[i]-x[j])/h,sigh[i]);
      if(stype[0] == 0){ 
	S1 = 0.0;  S2 = 0.0;
	for(i=0;i<n;i++){
	  t1 = (x[i] - x[j]);
	  S1 += fk[i] * t1;
	  S2 += fk[i] * t1 * t1;
	}
	for(i=0;i<n;i++){
	  t1 = (x[i] - x[j]);
	  b[i] = fabs(fk[i]*(S2-t1*S1));
	}
	
	bsum = 0.0;
	fx[j] = 0.0;
	for(i=0;i<n;i++){
	  bsum += b[i];
	  fx[j] += b[i] * y[i];
	}
	if(bsum>0.0){
	  fx[j] /= bsum;
	  mse += pow(y[j] - fx[j],2.0);
	  lcvscore += pow((y[j] - fx[j])/(1.- b[j]/bsum),2.0);
	}
      }else{
	bsum = 0.0;
	fx[j] = 0.;
	for(i=0;i<n;i++){
	  bsum += fk[i];
	  fx[j] += fk[i] * y[i];
	}
	if(bsum>0.0){
	  fx[j] /= bsum;
	  mse += pow(y[j] - fx[j],2.0);
	  lcvscore += pow((y[j] - fx[j])/(1.- fk[j]/bsum),2.0);
	}
      }
    }
    hs[k] = h;
    lscv[k] = lcvscore;
    if(lcvscore<lcvmax){
      lcvmax = lcvscore;
      h0 = h;
    }
  }
  if(m[0] > 1)  bw[0] = h0; else bw[0] = mse/n;
}

void hopt(double *y, double *x, int *size, 
	  int *kernel, int *stype, double *bw,
	  int *m, double *hs, double *lscv)
{
  int i,j,k,n;
  n = size[0];
  double h = bw[0],fx[n],fk[n],b[n];
  double S1,S2,t1,bsum;
  double dh,h0,lcvscore,lcvmax=9999999999999.;

  FunKernel f[1];
  f[0] = kgauss;

  h0=h;
  dh = 0.03*h;
  h = 0.5*h;

  for(k=0;k<m[0];k++){  //search for h, record as h0
    h += dh;
    lcvscore =0.0;

    for(j=0;j<n;j++){
      for(i=0;i<n;i++)
	fk[i] = f[kernel[0]]((x[i]-x[j])/h);
      if(stype[0] == 0){ 
	S1 = 0.0;  S2 = 0.0;
	for(i=0;i<n;i++){
	  t1 = (x[i] - x[j]);
	  S1 += fk[i] * t1;
	  S2 += fk[i] * t1 * t1;
	}
	for(i=0;i<n;i++){
	  t1 = (x[i] - x[j]);
	  b[i] = fabs(fk[i]*(S2-t1*S1));
	}
	
	bsum = 0.0;
	fx[j] = 0.0;
	for(i=0;i<n;i++){
	  bsum += b[i];
	  fx[j] += b[i] * y[i];
	}
	fx[j] /= bsum;
	lcvscore += pow((y[j] - fx[j])/(1.- b[j]/bsum),2.0);
      }else{
	bsum = 0.0;
	fx[j] = 0.;
	for(i=0;i<n;i++){
	  bsum += fk[i];
	  fx[j] += fk[i] * y[i];
	}
	fx[j] /= bsum;
	lcvscore += pow((y[j] - fx[j])/(1.- fk[j]/bsum),2.0);
      }
    }
    hs[k] = h;
    lscv[k] = lcvscore;
    if(lcvscore<lcvmax){
      lcvmax = lcvscore;
      h0 = h;
    }
  }
  bw[0] = h0;
}



void ddm(double *y, double *x, double *sig, int *size, int *kernel, 
	 int *stype, double *bw, double *x0, int *ngrid)
{
  int i,j,m,n;
  n = size[0];  m = ngrid[0];
  double h = bw[0],dx,dxh,t1;
  double sigh[n], fk[n], fdk[n], fddk[n]; 
  double b[n],db[n],ddb[n];
  double S1,dS1,ddS1,S2,dS2,ddS2;
  double sumk,sumk1,sumb,sumb1,sumb2;
  double mx[m];

  FunKernel2 f[2],df[2],ddf[2];
  f[0] = ksupport;
  f[1] = klaplace;
  df[0] = dksupport;
  df[1] = dklaplace;
  ddf[0] = ddksupport;
  ddf[1] = ddklaplace;
  
  for(i=0;i<n;i++) sigh[i] = sig[i]/h;

  for(j=0;j<m;j++){
    for(i=0;i<n;i++){
      dx = x0[j]-x[i];
      dxh = dx/h;
      fk[i] = f[kernel[0]](dxh,sigh[i]);
      fdk[i] = df[kernel[0]](dxh,sigh[i])/h;
      fddk[i] = ddf[kernel[0]](dxh,sigh[i])/h/h;      
    }
    if(stype[0]==0){
      S1=0.0;S2=0.0;dS1=0.0;dS2=0.0;ddS1=0.0;ddS2=0.0;
      sumk=0.0; sumk1=0.0;
      for(i=0;i<n;i++){
	sumk += fk[i];
	sumk1 += fdk[i];
	dx = x0[j]-x[i];
	t1 = fk[i]*dx;
	S1 += t1;
	t1 *= dx;
	S2 += t1;
	t1 = fdk[i] * dx;
	dS1 += t1;
	t1 *= dx;
	dS2 += t1;
	t1 = fddk[i] * dx;
	ddS1 += t1;
	t1 *= dx;
	ddS2 += t1;
      }
      ddS2 += 4.0 * dS1 + 2.0 * sumk;
      dS1 += sumk;
      dS2 += 2.0 * S1;
      ddS1 += 2.0 * sumk1;

      for(i=0;i<n;i++){
	dx = x0[j]-x[i];
	b[i] = fk[i] * (S2 - dx * S1);
	db[i] = fdk[i]*(S2-dx*S1)+fk[i]*(dS2-S1-dx*dS1);
	ddb[i] = fddk[i]*(S2-dx*S1) + 2.*fdk[i]*(dS2-S1-dx*dS1)
	  + fk[i]*(ddS2-2.0*dS1-dx*ddS1);
      }
    }else{
      for(i=0;i<n;i++){
	b[i] = fk[i];
	db[i] = fdk[i];
	ddb[i] = fddk[i];
      }
    }
    //  to compute ||l(x)||
    sumb = 0.0; sumb1=0.0; sumb2=0.0;
    for(i=0;i<n;i++){
      sumb += b[i];
      sumb1 += db[i];
      sumb2 += ddb[i];
    }
    t1 = ddb[i]*sumb*sumb -3.*db[i]*sumb*sumb1+2.*b[i]*sumb1*sumb1-b[i]*sumb2*sumb;
    mx[j] = t1/pow(sumb,3.0);
  }
  
  for(j=0;j<m;j++){
    x0[j] = mx[j];
  }
}


void npr(double *y, double *x, int *size, int *kernel, 
	 int *stype, double *bw, double *x0, int *ngrid, double *alpha,
	 double *ell, double *kappa, double *tubec)
{
  int i,j,m,n;
  n = size[0];  m = ngrid[0];
  double h = bw[0],fx[m],fk[n],b[n];
  double S1,S2,t1,bsum;

  FunKernel f[1];
  f[0] = kgauss;

  if(stype[0]==0){

    for(j=0;j<m;j++){
      for(i=0;i<n;i++)
	fk[i] = f[kernel[0]]((x[i]-x0[j])/h);
      S1 = 0.0;  S2 = 0.0;
      for(i=0;i<n;i++){
	t1 = (x[i] - x0[j]);
	S1 += fk[i] * t1;
	S2 += fk[i] * t1 * t1;
      }
      for(i=0;i<n;i++){
	t1 = (x[i] - x0[j]);
	b[i] = fabs(fk[i]*(S2-t1*S1));
      }
      //  to compute ||l(x)||
      bsum = 0.0;
      t1 = 0.0;  // used to compute SS(b[i])
      fx[j] = 0.0;
      for(i=0;i<n;i++){
	bsum += b[i];
	t1 += b[i]*b[i];
	fx[j] += b[i] * y[i];
      }
      fx[j] /= bsum;
      ell[j] = sqrt(t1)/bsum;

      kappa[0] = GLIKappa2(x0[0],x0[m-1],funKappa2,x,h,n,kernel[0],stype[0]); 
      t1 = NRsolve(2.0, ftube, dftube, kappa[0], alpha[0]);
      tubec[0] = t1;
      ell[j] *= t1; 
    }
  }else{
    for(j=0;j<m;j++){
      for(i=0;i<n;i++)
	b[i] = f[kernel[0]]((x[i]-x0[j])/h);
      //  to compute ||l(x)||
      bsum = 0.0;
      t1 = 0.0;  // used to compute SS(b[i])
      fx[j] = 0.0;
      for(i=0;i<n;i++){
	bsum += b[i];
	t1 += b[i]*b[i];
	fx[j] += b[i] * y[i];
      }
      fx[j] /= bsum;
      ell[j] = sqrt(t1)/bsum;
      kappa[0] = GLIKappa2(x0[0],x0[m-1],funKappa2,x,h,n,kernel[0],stype[0]); 
      t1 = NRsolve(2.0, ftube, dftube, kappa[0], alpha[0]);
      tubec[0] = t1;
      ell[j] *= t1; 
    }
    
  }
  
  for(j=0;j<m;j++){
    x0[j] = fx[j];
  }
}


void dm(double *y, double *x, double *sig, int *size, int *kernel, 
	 int *stype, double *bw, double *x0, int *ngrid)
{
  int i,j,m,n;
  n = size[0];  m = ngrid[0];
  double h = bw[0],dx,dxh,t1;
  double sigh[n], fk[n], fdk[n]; 
  double b[n],db[n];
  double S1,dS1,S2,dS2;
  double sumk,sumk1,sumb,sumb1,sumby, sumb1y;
  double mx[m];

  FunKernel2 f[2],df[2];
  f[0] = ksupport;
  f[1] = klaplace;
  df[0] = dksupport;
  df[1] = dklaplace;
  
  for(i=0;i<n;i++) sigh[i] = sig[i]/h;

  for(j=0;j<m;j++){
    for(i=0;i<n;i++){
      dx = x0[j]-x[i];
      dxh = dx/h;
      fk[i] = f[kernel[0]](dxh,sigh[i]);
      fdk[i] = df[kernel[0]](dxh,sigh[i])/h;
    }
    
    if(stype[0]==0){
      S1=0.0;S2=0.0;dS1=0.0;dS2=0.0;
      sumk=0.0; sumk1=0.0;
      for(i=0;i<n;i++){
	sumk += fk[i];
	sumk1 += fdk[i];
	dx = x0[j]-x[i];
	t1 = fk[i]*dx;
	S1 += t1;
	t1 *= dx;
	S2 += t1;
	t1 = fdk[i] * dx;
	dS1 += t1;
	t1 *= dx;
	dS2 += t1;
      }
      dS1 += sumk;
      dS2 += 2.0 * S1;

      for(i=0;i<n;i++){
	dx = x0[j]-x[i];
	b[i] = fk[i] * (S2 - dx * S1);
	db[i] = fdk[i]*(S2-dx*S1)+fk[i]*(dS2-S1-dx*dS1);
      }
    }else{
      for(i=0;i<n;i++){
	b[i] = fk[i];
	db[i] = fdk[i];
      }
    }
    //  to compute ||l(x)||
    sumb = 0.0; sumb1=0.0;sumby=0.0;sumb1y=0.0;
    for(i=0;i<n;i++){
      sumb += b[i];
      sumb1 += db[i];
      sumby += b[i]*y[i];
      sumb1y += db[i]*y[i];
    }

    t1 = sumb1y*sumb - sumby*sumb1;
    if(sumb != 0.0){
      mx[j] = t1/pow(sumb,2.0);
    }else{
      mx[j] = 0.0;
    }
  }

  for(j=0;j<m;j++){
    if(mx[j] > 10.0){
      x0[j] = 10.0;
    }else if(mx[j] < -10.0){
      x0[j] = -10.0;
    } else{
      x0[j] = mx[j];
    }
  }

}

void bias(double *y, double *x, double *sig, int *size, int *kernel, 
	 int *stype, double *bw, double *x0, int *ngrid)
{
  int i,j,m,n;
  n = size[0];  m = ngrid[0];
  double h = bw[0],dx,dxh,t1;
  double sigh[n], fk[n], fdk[n], fddk[n]; 
  double b[n],db[n],ddb[n];
  double S1,dS1,ddS1,S2,dS2,ddS2;
  double sumk,sumk1,sumb,sumb1,sumb2;
  double mx[m];

  FunKernel2 f[2],df[2],ddf[2];
  f[0] = ksupport;
  f[1] = klaplace;
  df[0] = dksupport;
  df[1] = dklaplace;
  ddf[0] = ddksupport;
  ddf[1] = ddklaplace;
  
  for(i=0;i<n;i++) sigh[i] = sig[i]/h;

  for(j=0;j<m;j++){
    for(i=0;i<n;i++){
      dx = x0[j]-x[i];
      dxh = dx/h;
      fk[i] = f[kernel[0]](dxh,sigh[i]);
      fdk[i] = df[kernel[0]](dxh,sigh[i]);
      fddk[i] = ddf[kernel[0]](dxh,sigh[i]);      
    }
    if(stype[0]==0){
      S1=0.0;S2=0.0;dS1=0.0;dS2=0.0;ddS1=0.0;ddS2=0.0;
      sumk=0.0; sumk1=0.0;
      for(i=0;i<n;i++){
	sumk += fk[i];
	sumk1 += fdk[i];
	dx = x0[j]-x[i];
	t1 = fk[i]*dx;
	S1 += t1;
	t1 *= dx;
	S2 += t1;
	t1 = fdk[i] * dx;
	dS1 += t1;
	t1 *= dx;
	dS2 += t1;
	t1 = fddk[i] * dx;
	ddS1 += t1;
	t1 *= dx;
	ddS2 += t1;
      }
      ddS2 += 4.0 * dS1 + 2.0 * sumk;
      dS1 += sumk;
      dS2 += 2.0 * S1;
      ddS1 += 2.0 * sumk1;

      for(i=0;i<n;i++){
	dx = x0[j]-x[i];
	b[i] = fk[i] * (S2 - dx * S1);
	db[i] = fdk[i]*(S2-dx*S1)+fk[i]*(dS2-S1-dx*dS1);
	ddb[i] = fddk[i]*(S2-dx*S1) + 2.*fdk[i]*(dS2-S1-dx*dS1)
	  + fk[i]*(ddS2-2.0*dS1-dx*ddS1);
      }
    }else{
      for(i=0;i<n;i++){
	b[i] = fk[i];
	db[i] = fdk[i];
	ddb[i] = fddk[i];
      }
    }
    //  to compute ||l(x)||
    sumb = 0.0; sumb1=0.0; sumb2=0.0;
    for(i=0;i<n;i++){
      sumb += b[i];
      sumb1 += db[i];
      sumb2 += ddb[i];
    }
    t1 = ddb[i]*sumb*sumb -3.*db[i]*sumb*sumb1+2.*b[i]*sumb1*sumb1-b[i]*sumb2*sumb;
    mx[j] = t1/pow(sumb,3.0);
  }
  
  for(j=0;j<m;j++){
    x0[j] = mx[j];
  }
}

/*

Codes for rounding error density estimation
 */


double GLInt7p(double a, double b,  
	       double (*fn)(double,double,double,double*,double*,double*,int),
	       double h, double g,double *half, double *w,double *f,int n)
{
   double integral = 0.0; 
   double c = 0.5 * (b - a);
   double d = 0.5 * (b + a);
   double dum;
   const double *pB = &B100[NOPZ100 - 1];
   const double *pA = &A100[NOPZ100 - 1];

   for (; pB >= B100; pA--, pB--) {
      dum = c * *pB;
      integral += *pA * ( (*fn)(d - dum,h,g,half,w,f,n) + 
			  (*fn)(d + dum,h,g,half,w,f,n) );
   }

   return c * integral;
}

void convunif(double *w, double *f, double *a, double *b, int *size, double *theta,
	      double *x0, int *n0){
  /* to compute the convolution density estimator using Gaussian
     kernel for heteroscedastic additive independenent uniform
     (rounding) errors.*/
  int n=size[0], M = n0[0],i,j; 
  double h = theta[0];
  double fx0[M];
  double za,zb,N=0.0;//N:=sum(f)
  
  for(i=0;i<n;i++){//compute the total number of observations
      N += f[i];
  }
  for(i=0;i<M;i++){
    fx0[i] = 0.0;
    for(j=0;j<n;j++){
      za = (x0[i]-w[j]-a[j])/h;
      zb = (x0[i]-w[j]-b[j])/h;
      fx0[i] += f[j]*(pnorm(za,0.0,1.0,1,0)-pnorm(zb,0.0,1.0,1,0))/(b[j]-a[j]);
    }
    x0[i] = fx0[i]/N;
  }
}

/*
The following codes are developed to find the bootstrap MISE optimal
bandwidth.  The common factor of 1/(2PIn^2) was disregarded.
 */

double fa(double t, double h, double g, double *H, double *W, double *f,int n){
  int i,j;
  double h2t2 = pow(h*t,2.0),g2t2=pow(g*h,2.0),tmp,sum1,sum2,res;
  sum1=0.;sum2=0.;tmp=0.;
  tmp = (1.-1./n)*exp(-(g2t2+h2t2))-2.*exp(-(.5*h2t2+g2t2))+exp(-g2t2);
  if(fabs(t)==0.0){
    for(j=0;j<n;j++){
      sum1 += cos(t*W[j])*f[j];
      sum2 += sin(t*W[j])*f[j];
    }
  }else{
    for(j=0;j<n;j++){
      res = sin(H[j]*t)/H[j]/t;
      sum1 +=  res * cos(t*W[j])*f[j];
      sum2 +=  res * sin(t*W[j])*f[j];
    }
  }
  res= tmp * (sum1*sum1+sum2*sum2);
  return res;
}

void hbmise2(double *x,double *f, double *w, int *size, double *hopt)
/* Input:
   (1) x[j]: the observations;
   (3) w[j]: = b[j]-a[j], which the support of the uniform error;
   (4) size, or n in the code: an integer shows the number of distinct values in y[.]
   (5) f: frequencies
   Output: 
   (1) hopt: the initial bandwidth, which will also be the output.
     
   Derived parameters:
   (1) h is the bandwidth, which is searched over a defined grid by
   minimizing the BMISE.  We set the grid number 100.  The initial
   bandwidth h0 was selected by using Silverman's estimate
   (h=1.06*sigma/n^.2) or the dpik or dpih in package KernSmooth,
   and we search over the range from h0/10 to k*h0.  Where
   k=1+max(bj-aj)^2/12/sd(y). We can simply select 2. If the
   bandwidth is on the boundary, expand the search.

   Controls
   (1) The integration range is set to t \in [0,U], where U>4/h=40/h0.
   (2) iter is used to control the maximum search loop.  Maximum 10.

   Remarks: check whether erf() is a build-in function.  If yes, use
   it directly.

   */
{
  int ngrid = 150, n=size[0],i,j,nsum=0;
  double hinit,h0,h,hstep,h2;
  double xbar,s,tmp0,tmp1,tmp2,mise, fint,fb;
  double w2[n],w1_2[n],w1_2sq[n];
  double mise0=9999999999.;
  Fun7p fn[1];
  fn[0] = fa; //To compute part A to be integrated.

  tmp1 = 0.0;
  tmp2 = 0.0;
  for(i=0;i<n;i++){
    nsum += f[i];
    tmp1 = tmp1 + x[i]*f[i];
    tmp2 = tmp2 + x[i] * x[i]*f[i];
    w2[i] = pow(w[i],2.0);
    w1_2[i] = w[i]/2.0;
    w1_2sq[i]=pow(w1_2[i],2.0);
  }
  xbar = tmp1 / nsum;
  s = sqrt((tmp2 - nsum * xbar * xbar) / (nsum -1.));
  // initialize bandwidths
  h0 = hopt[0];
  h = h0/30.;
  h2 = h * h;
  hstep = 1.5*h0/ngrid;

  for(i=0;i<ngrid;i++){
    h += hstep;
    fb = .5;
    tmp2 = 0.1/h0; //used as step for t
    tmp1 = 0.00001; // initial t value
    for(j=0;j<10000;j++){
      if(sin(0.5*tmp1) == 0.0){
	tmp1 = tmp1 + 0.00001;
      }
      fb += pow(exp(-h*tmp1) * tmp1 / sin(0.5 * tmp1),2.0);
      tmp1 += tmp2;
    }

    fint = GLInt7p(0.0,1000/h0,fn[0],h,h0,w1_2,x,f,n);
    mise =  fb * tmp2 * nsum;
    tmp1 = (mise+4.*fint/nsum/nsum);//   /2./pow(nsum,2.0)/M_PI;
    if(tmp1<mise0){
      hopt[0] = h;
      mise0 = tmp1;
    }
  }
  //  hopt[0] = h0;
}
 
/*  End of BMISE bandwidth selector */


void hbmise(double *x,double *f, double *w, int *size, double *hopt)
/* Input: This bandwidth selector is very similar to hbmise.  hbmise2
   is for CME, while hbmise is for BME.  Due to the numerical
   difficulty of 'singularity', we compute the second term differently
   by Remann sum.
   */
{
  int ngrid = 150, n=size[0],i,j,nsum=0;
  double hinit,h0,h,hstep,h2;
  double xbar,s,tmp0,tmp1,tmp2,mise, fint,fb;
  double w2[n],w1_2[n],w1_2sq[n];
  double mise0=9999999999.;
  Fun7p fn[1];
  fn[0] = fa; //To compute part A to be integrated.

  tmp1 = 0.0;
  tmp2 = 0.0;
  for(i=0;i<n;i++){
    nsum += f[i];
    tmp1 = tmp1 + x[i]*f[i];
    tmp2 = tmp2 + x[i] * x[i]*f[i];
    w2[i] = pow(w[i],2.0);
    w1_2[i] = w[i]/2.0;
    w1_2sq[i]=pow(w1_2[i],2.0);
  }
  xbar = tmp1 / nsum;
  s = sqrt((tmp2 - nsum * xbar * xbar) / (nsum -1.));
  // initialize bandwidths
  h0 = hopt[0];
  h = h0/30.;
  h2 = h * h;
  hstep = 1.5*h0/ngrid;

  for(i=0;i<ngrid;i++){
    h += hstep;
    fb = 0.0;
    for(j=0;j<n;j++){
      tmp1 = exp(-w1_2sq[j]/h2);
      tmp2 = w1_2[j]/h*M_SQRT_PI*erf(w1_2[j]/h);
      fb += fabs(1./w1_2sq[j]*(tmp1+tmp2-1.))*f[j];
    }
    fint = GLInt7p(0.0,1000/h0,fn[0],h,h,w1_2,x,f,n);
    mise=M_SQRT_PI*h*fb;
    tmp1 = (mise+2*fint/nsum/nsum)/2./pow(nsum,2.0)/M_PI;
    if(tmp1<mise0){
      hopt[0] = h;
      mise0 = tmp1;
    }
  }
  //use the estimated bw as g to see whether we can improve the optimal bw selection
  h0 = hopt[0];
  h = h0/30.;
  h2 = h * h;
  hstep = 1.5*h0/ngrid;

  for(i=0;i<ngrid;i++){
    h += hstep;
    fb = 0.0;
    for(j=0;j<n;j++){
      tmp1 = exp(-w1_2sq[j]/h2);
      tmp2 = w1_2[j]/h*M_SQRT_PI*erf(w1_2[j]/h);
      fb += fabs(1./w1_2sq[j]*(tmp1+tmp2-1.))*f[j];
    }
    fint = GLInt7p(0.0,1000/h0,fn[0],h,h,w1_2,x,f,n);
    mise=M_SQRT_PI*h*fb;
    tmp1 = (mise+2*fint/nsum/nsum)/2./pow(nsum,2.0)/M_PI;
    if(tmp1<mise0){
      hopt[0] = h;
      mise0 = tmp1;
    }
  }
  //  hopt[0] = h0;
}
 
/*  End of BMISE bandwidth selector */
