#' Run BEAST2
#' @inheritParams default_params_doc
#' @return The text sent to \code{STDOUT} and \code{STDERR}.
#'   It will create the files with names
#'   \code{output_log_filename}, \code{output_trees_filenames}
#'   and \code{output_state_filenames}
#' @export
#' @examples
#'   library(testthat)
#'
#'   if (is_beast2_installed() && is_on_ci()) {
#'
#'     output_log_filename <- tempfile(fileext = ".log")
#'     output_trees_filenames <- tempfile(fileext = ".trees")
#'     output_state_filename <- tempfile(fileext = ".xml.state")
#'
#'     expect_false(file.exists(output_log_filename))
#'     expect_false(file.exists(output_trees_filenames))
#'     expect_false(file.exists(output_state_filename))
#'
#'     output <- run_beast2(
#'       input_filename = get_beastier_path("2_4.xml"),
#'       output_log_filename = output_log_filename,
#'       output_trees_filenames = output_trees_filenames,
#'       output_state_filename = output_state_filename
#'     )
#'
#'     expect_true(length(output) > 40)
#'     expect_true(file.exists(output_log_filename))
#'     expect_true(file.exists(output_trees_filenames))
#'     expect_true(file.exists(output_state_filename))
#'   }
#' @author Richèl J.C. Bilderbeek
run_beast2 <- function(
  input_filename,
  output_log_filename = create_default_log_filename(
    input_filename,
    beast2_path,
    verbose
  ),
  output_trees_filenames = create_default_trees_filenames(
    input_filename,
    beast2_path,
    verbose
  ),
  output_state_filename = tempfile(
    pattern = "beastier_", fileext = ".xml.state"
  ),
  rng_seed = NA,
  n_threads = NA,
  use_beagle = FALSE,
  overwrite = TRUE,
  beast2_working_dir = tempfile(pattern = "beast2_tmp_folder"),
  beast2_path = get_default_beast2_path(),
  verbose = FALSE
) {
  if (verbose) {
    print(paste("input_filename:", input_filename))
    print(paste("output_log_filename:", output_log_filename))
    print(paste("output_trees_filenames:", output_trees_filenames))
    print(paste("output_state_filename:", output_state_filename))
  }
  if (is_win_bin_path(beast2_path)) {
    stop("Cannot use the Windows executable BEAST2.exe in scripts")
  }
  # Add the full path of the input file
  input_filename_full <- input_filename
  if (basename(input_filename) == input_filename) {
    input_filename_full <- file.path(getwd(), input_filename)
  }
  output_state_filename_full <- output_state_filename
  if (basename(output_state_filename) == output_state_filename) {
    output_state_filename_full <- file.path(getwd(), output_state_filename)
  }

  check_input_filename(input_filename_full) # nolint internal function
  check_beast2_path(beast2_path) # nolint internal function
  check_input_filename_validity( # nolint internal function
    input_filename = input_filename_full,
    beast2_path = beast2_path,
    verbose = verbose
  )

  if (any(file.exists(output_log_filename)) && overwrite == FALSE) {
    stop("Will not overwrite 'output_log_filename' ('",
      output_log_filename, "') with 'overwrite' is FALSE"
    )
  }
  if (any(file.exists(output_trees_filenames)) && overwrite == FALSE) {
    stop("Will not overwrite 'output_trees_filenames' ('",
      output_trees_filenames, "') with 'overwrite' is FALSE"
    )
  }

  # These are files created by BEAST2
  beast_log_filename <- create_default_log_filename(
    input_filename = input_filename_full,
    beast2_path = beast2_path,
    verbose = verbose
  )
  if (any(file.exists(beast_log_filename)) && overwrite == FALSE) {
    stop("Cannot overwrite the .log file created by BEAST2 ('",
      beast_log_filename, "') with 'overwrite' is FALSE"
    )
  }
  beast_trees_filename <- create_default_trees_filenames(
    input_filename = input_filename_full,
    beast2_path = beast2_path,
    verbose = verbose
  )
  if (any(file.exists(beast_trees_filename)) && overwrite == FALSE) {
    stop("Cannot overwrite the .trees files created by BEAST2 ('",
      beast_trees_filename, "') with 'overwrite' is FALSE"
    )
  }

  check_rng_seed(rng_seed) # nolint beastier function

  alignment_ids <- get_alignment_ids(input_filename_full) # nolint internal function

  if (length(output_trees_filenames) != length(alignment_ids)) {
    stop(
      "'output_trees_filenames' must have as much elements as ",
      "'input_filename' has alignments. 'output_trees_filenames' has ",
      length(output_trees_filenames), " elements, 'input_filename' has ",
      length(alignment_ids), " alignments"
    )
  }

  testit::assert(length(input_filename_full) == 1)
  testit::assert(length(output_state_filename) == 1)
  testit::assert(length(rng_seed) == 1)
  testit::assert(length(n_threads) == 1)
  testit::assert(length(use_beagle) == 1)
  testit::assert(length(overwrite) == 1)
  testit::assert(length(beast2_path) == 1)

  cmd <- beastier::create_beast2_run_cmd(
    input_filename = input_filename_full,
    output_state_filename = output_state_filename_full,
    rng_seed = rng_seed,
    n_threads = n_threads,
    use_beagle = use_beagle,
    overwrite = overwrite,
    beast2_path = beast2_path
  )

  if (verbose == TRUE) {
    print(paste("cmd:", paste0(cmd, collapse = " ")))
  }

  # Move working directory to temporary folder
  work_in_tmp_folder <- TRUE
  if (work_in_tmp_folder) {
    cur_wd <- getwd()
    tmp_wd <- beast2_working_dir
    # Do not warning if the folder already exists
    dir.create(tmp_wd, showWarnings = FALSE)
    setwd(tmp_wd)
  }

  output <- system2(
    command = cmd[1],
    args = cmd[-1],
    stdout = TRUE,
    stderr = TRUE
  )


  if (length(output) == 1) {
    stop(
      "Command '", paste0(cmd, collapse = " "), "' failed ",
      "with error '", output, "'"
    )
  }

  # Copying done, back to original working directory
  if (work_in_tmp_folder) {
    setwd(cur_wd)
  }

  ##############################################################################
  # The filenames as created by BEAST2
  ##############################################################################
  actual_log_filename <- file.path(
    tmp_wd,
    basename(
      create_default_log_filename(
        input_filename = input_filename_full,
        beast2_path = beast2_path
      )
    )
  )
  if (!file.exists(output_log_filename)) {
    testit::assert(file.exists(actual_log_filename))
    file.rename(from = actual_log_filename, to = output_log_filename)
  }

  for (i in seq_along(output_trees_filenames)) {
    to <- output_trees_filenames[i]
    if (!file.exists(to)) {
      actual_trees_filename <- file.path(
        tmp_wd,
        basename(
          create_default_trees_filenames(
            input_filename = input_filename_full,
            beast2_path = beast2_path,
            verbose = verbose
          )[i]
        )
      )
      testit::assert(file.exists(actual_trees_filename))
      file.rename(from = actual_trees_filename, to = to)
    }
  }

  if (verbose) {
    print(paste0("output_log_filename ('", output_log_filename, "'): ",
      file.exists(output_log_filename))
    )
    print(paste0("output_state_filename ('", output_state_filename, "'): ",
      file.exists(output_state_filename))
    )
    print(paste0("output_trees_filenames ('", output_trees_filenames, "'): ",
      file.exists(output_trees_filenames))
    )
  }

  testit::assert(file.exists(output_log_filename))
  testit::assert(file.exists(output_state_filename_full))
  testit::assert(all(file.exists(output_trees_filenames)))

  output
}
