% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BIOMOD_RangeSize.R
\name{BIOMOD_RangeSize}
\alias{BIOMOD_RangeSize}
\alias{BIOMOD_RangeSize,data.frame,data.frame-method}
\alias{BIOMOD_RangeSize,SpatRaster,SpatRaster-method}
\title{Analyze the range size differences between projections of species distribution models}
\usage{
BIOMOD_RangeSize(proj.current, proj.future)

\S4method{BIOMOD_RangeSize}{data.frame,data.frame}(proj.current, proj.future)

\S4method{BIOMOD_RangeSize}{SpatRaster,SpatRaster}(proj.current, proj.future)
}
\arguments{
\item{proj.current}{a \code{data.frame}, \code{\link[raster:stack]{RasterLayer}}
or \code{\link[terra:rast]{SpatRaster}} object containing the initial binary projection(s)
of the (ensemble) species distribution model(s)}

\item{proj.future}{a \code{data.frame}, \code{\link[raster:stack]{RasterLayer}}
or \code{\link[terra:rast]{SpatRaster}} object containing the final binary projection(s)
of the (ensemble) species distribution model(s)}
}
\value{
A \code{list} containing two objects :
\describe{
\item{Compt.By.Species}{a \code{data.frame} containing the summary of range change for each
comparison
\itemize{
\item{\code{Loss} : }{number of pixels predicted to be lost}
\item{\code{Stable0} : }{number of pixels not currently occupied and not predicted to be}
\item{\code{Stable1} : }{number of pixels currently occupied and predicted to remain
occupied}
\item{\code{Gain} : }{number of pixels predicted to be gained}
\item{\code{PercLoss} : }{percentage of pixels currently occupied and predicted to be lost
(\code{Loss / (Loss + Stable1)})}
\item{\code{PercGain} : }{percentage of pixels predicted to be gained compare to the
number of pixels currently occupied (\code{Gain / (Loss + Stable1)})}
\item{\code{SpeciesRangeChange} : }{percentage of pixels predicted to change (loss or gain)
compare to the number of pixels currently occupied (\code{PercGain - PercLoss})}
\item{\code{CurrentRangeSize} : }{number of pixels currently occupied}
\item{\code{FutureRangeSize0Disp} : }{number of pixels predicted to be occupied, assuming
no migration}
\item{\code{FutureRangeSize1Disp} : }{number of pixels predicted to be occupied, assuming
migration}
}
}
\item{Diff.By.Pixel}{an object in the same form than the input data (\code{proj.current} and
\code{proj.future}) and containing a value for each point/pixel of each comparison among :
\itemize{
\item \code{-2} : predicted to be lost
\item \code{-1} : predicted to remain occupied
\item \code{0} : predicted to remain unoccupied
\item \code{1} : predicted to be gained
}
}
}
}
\description{
This function allows to calculate the absolute number of locations (pixels) lost,
stable and gained, as well as the corresponding relative proportions, between two (or more)
binary projections of (ensemble) species distribution models (\emph{which can represent new
time scales or environmental scenarios for example}).
}
\details{
Note that \bold{this function is only relevant to compare binary projections, made on the
same area with the same resolution}. \cr

Comparison between \code{proj.current} and \code{proj.future} depends
on the number of projection in both objects:\tabular{lll}{
   \code{proj.current} \tab \code{proj.future} \tab \bold{Comparison} \cr
   \bold{1 projection} (\emph{e.g. data.frame with 1 column, SpatRaster with 1 layer}) \tab \bold{1 projection}  (\emph{e.g. data.frame with 1 column, SpatRaster with 1 layer}) \tab comparison of both projection  (\emph{e.g. current vs future conditions for the same model ; current vs current condition for two different models}) \cr
   \bold{\code{n} projections}  (\emph{e.g. data.frame with n column, SpatRaster with n layer}) \tab \bold{\code{n} projections}  (\emph{e.g. data.frame with n column, SpatRaster with n layer}) \tab comparing projection \code{i} in \code{proj.current} to projection \code{i} in \code{proj.future}  (\emph{e.g. comparing current vs future condition for n models}) \cr
   \bold{\code{1} projection}   (\emph{e.g. data.frame with 1 column, SpatRaster with 1 layer}) \tab \bold{\code{n} projections}  (\emph{e.g. data.frame with n column, SpatRaster with n layer}) \tab comparing projection in \code{proj.current} to each projection in \code{proj.future}  (\emph{e.g. comparing current vs n different future condition (e.g. climate change scenario) for 1 model}) \cr
}


\code{Diff.By.Pixel} object is obtained by applying the simple following formula :
\deqn{proj.future - 2 * proj.current}
}
\examples{
library(terra)

# Load species occurrences (6 species available)
data(DataSpecies)
head(DataSpecies)

# Select the name of the studied species
myRespName <- 'GuloGulo'

# Get corresponding presence/absence data
myResp <- as.numeric(DataSpecies[, myRespName])

# Get corresponding XY coordinates
myRespXY <- DataSpecies[, c('X_WGS84', 'Y_WGS84')]

# Load environmental variables extracted from BIOCLIM (bio_3, bio_4, bio_7, bio_11 & bio_12)
data(bioclim_current)
myExpl <- terra::rast(bioclim_current)

\dontshow{
myExtent <- terra::ext(0,30,45,70)
myExpl <- terra::crop(myExpl, myExtent)
}

# --------------------------------------------------------------- #
file.out <- paste0(myRespName, "/", myRespName, ".AllModels.models.out")
if (file.exists(file.out)) {
  myBiomodModelOut <- get(load(file.out))
} else {

  # Format Data with true absences
  myBiomodData <- BIOMOD_FormatingData(resp.var = myResp,
                                       expl.var = myExpl,
                                       resp.xy = myRespXY,
                                       resp.name = myRespName)

  # Create default modeling options
  myBiomodOptions <- BIOMOD_ModelingOptions()

  # Model single models
  myBiomodModelOut <- BIOMOD_Modeling(bm.format = myBiomodData,
                                      modeling.id = 'AllModels',
                                      models = c('RF', 'GLM'),
                                      bm.options = myBiomodOptions,
                                      nb.rep = 2,
                                      data.split.perc = 80,
                                      metric.eval = c('TSS','ROC'),
                                      var.import = 3,
                                      do.full.models = FALSE,
                                      seed.val = 42)
}

models.proj <- get_built_models(myBiomodModelOut, algo = "RF")
  # Project single models
  myBiomodProj <- BIOMOD_Projection(bm.mod = myBiomodModelOut,
                                    proj.name = 'CurrentRangeSize',
                                    new.env = myExpl,
                                    models.chosen = models.proj,
                                    metric.binary = 'all',
                                    build.clamping.mask = TRUE)


# --------------------------------------------------------------- #
# Load environmental variables extracted from BIOCLIM (bio_3, bio_4, bio_7, bio_11 & bio_12)
data(bioclim_future)
myExplFuture <- terra::rast(bioclim_future)
\dontshow{
myExtent <- terra::ext(0,30,45,70)
myExplFuture <- terra::crop(myExplFuture, myExtent)
}
# Project onto future conditions
myBiomodProjectionFuture <- BIOMOD_Projection(bm.mod = myBiomodModelOut,
                                              proj.name = 'FutureRangeSize',
                                              new.env = myExplFuture,
                                              models.chosen = models.proj,
                                              metric.binary = 'TSS')

# Load current and future binary projections
CurrentProj <- get_predictions(myBiomodProj,
                               metric.binary = "TSS",
                               model.as.col = TRUE)
FutureProj <- get_predictions(myBiomodProjectionFuture,
                               metric.binary = "TSS",
                               model.as.col = TRUE)
# Compute differences
myBiomodRangeSize <- BIOMOD_RangeSize(proj.current = CurrentProj, proj.future = FutureProj)

myBiomodRangeSize$Compt.By.Models
plot(myBiomodRangeSize$Diff.By.Pixel)

# Represent main results 
bm_PlotRangeSize(bm.range = myBiomodRangeSize)


}
\seealso{
\code{\link{BIOMOD_Projection}}, \code{\link{BIOMOD_EnsembleForecasting}},
\code{\link{bm_PlotRangeSize}}

Other Main functions: 
\code{\link{BIOMOD_CrossValidation}()},
\code{\link{BIOMOD_EnsembleForecasting}()},
\code{\link{BIOMOD_EnsembleModeling}()},
\code{\link{BIOMOD_FormatingData}()},
\code{\link{BIOMOD_LoadModels}()},
\code{\link{BIOMOD_ModelingOptions}()},
\code{\link{BIOMOD_Modeling}()},
\code{\link{BIOMOD_PresenceOnly}()},
\code{\link{BIOMOD_Projection}()},
\code{\link{BIOMOD_Tuning}()}
}
\author{
Wilfried Thuiller, Damien Georges, Bruno Lafourcade
}
\concept{Main functions}
\keyword{"species}
\keyword{change"}
\keyword{gain}
\keyword{loss}
\keyword{projections}
\keyword{range}
