% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BIOMOD_EnsembleForecasting.R
\name{BIOMOD_EnsembleForecasting}
\alias{BIOMOD_EnsembleForecasting}
\title{Project ensemble species distribution models onto new environment}
\usage{
BIOMOD_EnsembleForecasting(
  bm.em,
  bm.proj = NULL,
  proj.name = NULL,
  new.env = NULL,
  new.env.xy = NULL,
  models.chosen = "all",
  metric.binary = NULL,
  metric.filter = NULL,
  compress = TRUE,
  nb.cpu = 1,
  na.rm = TRUE,
  ...
)
}
\arguments{
\item{bm.em}{a \code{\link{BIOMOD.ensemble.models.out}} object returned by the 
\code{\link{BIOMOD_EnsembleModeling}} function}

\item{bm.proj}{a \code{\link{BIOMOD.projection.out}} object returned by the 
\code{\link{BIOMOD_Projection}} function}

\item{proj.name}{(\emph{optional, default} \code{NULL}) \cr 
If \code{bm.proj = NULL}, a \code{character} corresponding to the name (ID) of the 
projection set (\emph{a new folder will be created within the simulation folder with this 
name})}

\item{new.env}{(\emph{optional, default} \code{NULL}) \cr 
If \code{bm.proj = NULL}, a \code{matrix}, \code{data.frame} or
 \code{\link[terra:rast]{SpatRaster}} object containing the new 
explanatory variables (in columns or layers, with names matching the
 variables names given to the \code{\link{BIOMOD_FormatingData}} function to build 
\code{bm.mod}) that will be used to project the species distribution model(s)
\cr \emph{Note that old format from \pkg{raster} are still supported such as 
\code{RasterStack} objects. }}

\item{new.env.xy}{(\emph{optional, default} \code{NULL}) \cr 
If \code{new.env} is a \code{matrix} or a \code{data.frame}, a 2-columns \code{matrix} or 
\code{data.frame} containing the corresponding \code{X} and \code{Y} coordinates that will 
be used to project the ensemble species distribution model(s)}

\item{models.chosen}{a \code{vector} containing model names to be kept, must be either 
\code{all} or a sub-selection of model names that can be obtained with the 
\code{\link{get_built_models}} function}

\item{metric.binary}{(\emph{optional, default} \code{NULL}) \cr 
A \code{vector} containing evaluation metric names to be used to transform prediction values 
into binary values based on models evaluation scores obtained with the 
\code{\link{BIOMOD_Modeling}} function. Must be among \code{all} (same evaluation metrics than 
those of \code{modeling.output}) or \code{POD}, \code{FAR}, \code{POFD}, \code{SR}, 
\code{ACCURACY}, \code{BIAS}, \code{ROC}, \code{TSS}, \code{KAPPA}, \code{OR}, \code{ORSS}, 
\code{CSI}, \code{ETS}, \code{BOYCE}, \code{MPA}}

\item{metric.filter}{(\emph{optional, default} \code{NULL}) \cr 
A \code{vector} containing evaluation metric names to be used to transform prediction values 
into filtered values based on models evaluation scores obtained with the 
\code{\link{BIOMOD_Modeling}} function. Must be among \code{all} (same evaluation metrics than 
those of \code{modeling.output}) or \code{POD}, \code{FAR}, \code{POFD}, \code{SR}, 
\code{ACCURACY}, \code{BIAS}, \code{ROC}, \code{TSS}, \code{KAPPA}, \code{OR}, \code{ORSS}, 
\code{CSI}, \code{ETS}, \code{BOYCE}, \code{MPA}}

\item{compress}{(\emph{optional, default} \code{TRUE}) \cr 
A \code{logical} or a \code{character} value defining whether and how objects should be 
compressed when saved on hard drive, must be either \code{TRUE}, \code{FALSE}, \code{xz} or 
\code{gzip} (see Details)}

\item{nb.cpu}{(\emph{optional, default} \code{1}) \cr 
An \code{integer} value corresponding to the number of computing resources to be used to 
parallelize the single models computation}

\item{na.rm}{(\emph{optional, default} \code{TRUE}) \cr
A boolean defining whether Ensemble Model projection should ignore \code{NA}
in Individual Model projection. Argument ignored by EWmean ensemble algorithm.}

\item{\ldots}{(\emph{optional, see Details})}
}
\value{
A \code{\link{BIOMOD.projection.out}} object containing models projections, or links to saved 
outputs. \cr Models projections are stored out of \R (for memory storage reasons) in 
\code{proj.name} folder created in the current working directory :
\enumerate{
  \item the output is a \code{data.frame} if \code{new.env} is a \code{matrix} or a 
  \code{data.frame}
  \item it is a \code{\link[terra:rast]{SpatRaster}} if \code{new.env} 
  is a \code{\link[terra:rast]{SpatRaster}} (or several
  \code{\link[terra:rast]{SpatRaster}} objects, if \code{new.env} is too large)
  \item raw projections, as well as binary and filtered projections (if asked),
   are saved in the \code{proj.name} folder
}
}
\description{
This function allows to project ensemble models built with the 
\code{\link{BIOMOD_EnsembleModeling}} function onto new environmental data 
(\emph{which can represent new areas, resolution or time scales for example}).
}
\details{
If \code{models.chosen = 'all'}, projections are done for all calibration and pseudo absences 
runs if applicable. \cr These projections may be used later by the 
\code{\link{BIOMOD_EnsembleForecasting}} function. \cr \cr

If \code{build.clamping.mask = TRUE}, a raster file will be saved within the projection 
folder. This mask values will correspond to the number of variables in each pixel that are out 
of their calibration / validation range, identifying locations where predictions are uncertain. 
\cr \cr

\code{...} can take the following values :
\itemize{
  \item \code{on_0_1000} : a \code{logical} value defining whether \code{0 - 1} 
  probabilities are to be converted to \code{0 - 1000} scale to save memory on backup
  \item \code{do.stack} : a \code{logical} value defining whether all projections are to be 
  saved as one \code{SpatRaster} object or several \code{SpatRaster} files (\emph{the 
  default if projections are too heavy to be all loaded at once in memory})
  \item \code{keep.in.memory} : a \code{logical} value defining whether all projections are 
  to be kept loaded at once in memory, or only links pointing to hard drive are to be returned
  \item \code{output.format} : a \code{character} value corresponding to the projections 
  saving format on hard drive, must be either \code{.grd}, \code{.img}, \code{.tif} or \code{.RData} (the 
  default if \code{new.env} is given as \code{matrix} or \code{data.frame})
}
}
\examples{

library(terra)
# Load species occurrences (6 species available)
data(DataSpecies)
head(DataSpecies)

# Select the name of the studied species
myRespName <- 'GuloGulo'

# Get corresponding presence/absence data
myResp <- as.numeric(DataSpecies[, myRespName])

# Get corresponding XY coordinates
myRespXY <- DataSpecies[, c('X_WGS84', 'Y_WGS84')]

# Load environmental variables extracted from BIOCLIM (bio_3, bio_4, bio_7, bio_11 & bio_12)
data(bioclim_current)
myExpl <- terra::rast(bioclim_current)

\dontshow{
myExtent <- terra::ext(0,30,45,70)
myExpl <- terra::crop(myExpl, myExtent)
}

 
# --------------------------------------------------------------- #
file.out <- paste0(myRespName, "/", myRespName, ".AllModels.models.out")
if (file.exists(file.out)) {
  myBiomodModelOut <- get(load(file.out))
} else {

  # Format Data with true absences
  myBiomodData <- BIOMOD_FormatingData(resp.var = myResp,
                                       expl.var = myExpl,
                                       resp.xy = myRespXY,
                                       resp.name = myRespName)

  # Model single models
  myBiomodModelOut <- BIOMOD_Modeling(bm.format = myBiomodData,
                                      modeling.id = 'AllModels',
                                      models = c('RF', 'GLM'),
                                      CV.strategy = 'random',
                                      CV.nb.rep = 2,
                                      CV.perc = 0.8,
                                      OPT.strategy = 'bigboss',
                                      metric.eval = c('TSS','ROC'),
                                      var.import = 3,
                                      seed.val = 42)
}


file.proj <- paste0(myRespName, "/proj_Current/", myRespName, ".Current.projection.out")
if (file.exists(file.proj)) {
  myBiomodProj <- get(load(file.proj))
} else {

  # Project single models
  myBiomodProj <- BIOMOD_Projection(bm.mod = myBiomodModelOut,
                                    proj.name = 'Current',
                                    new.env = myExpl,
                                    models.chosen = 'all',
                                    build.clamping.mask = TRUE)
}


file.EM <- paste0(myRespName, "/", myRespName, ".AllModels.ensemble.models.out")
if (file.exists(file.EM)) {
  myBiomodEM <- get(load(file.EM))
} else {

  # Model ensemble models
  myBiomodEM <- BIOMOD_EnsembleModeling(bm.mod = myBiomodModelOut,
                                        models.chosen = 'all',
                                        em.by = 'all',
                                        em.algo = c('EMmean', 'EMca'),
                                        metric.select = c('TSS'),
                                        metric.select.thresh = c(0.7),
                                        metric.eval = c('TSS', 'ROC'),
                                        var.import = 3,
                                        seed.val = 42)
}


# --------------------------------------------------------------- #
# Project ensemble models (from single projections)
myBiomodEMProj <- BIOMOD_EnsembleForecasting(bm.em = myBiomodEM, 
                                             bm.proj = myBiomodProj,
                                             models.chosen = 'all',
                                             metric.binary = 'all',
                                             metric.filter = 'all')

# Project ensemble models (building single projections)
myBiomodEMProj <- BIOMOD_EnsembleForecasting(bm.em = myBiomodEM,
                                             proj.name = 'CurrentEM',
                                             new.env = myExpl,
                                             models.chosen = 'all',
                                             metric.binary = 'all',
                                             metric.filter = 'all')
myBiomodEMProj
plot(myBiomodEMProj)


}
\seealso{
\code{\link{BIOMOD_FormatingData}}, \code{\link{bm_ModelingOptions}}, 
\code{\link{BIOMOD_Modeling}}, \code{\link{BIOMOD_EnsembleModeling}}, 
\code{\link{BIOMOD_RangeSize}}

Other Main functions: 
\code{\link{BIOMOD_EnsembleModeling}()},
\code{\link{BIOMOD_FormatingData}()},
\code{\link{BIOMOD_LoadModels}()},
\code{\link{BIOMOD_Modeling}()},
\code{\link{BIOMOD_Projection}()},
\code{\link{BIOMOD_RangeSize}()}
}
\author{
Wilfried Thuiller, Damien Georges, Robin Engler
}
\concept{Main functions}
\keyword{models}
\keyword{projection}
