% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/braidModel.R
\name{evalBraidModel}
\alias{evalBraidModel}
\title{Evaluate the BRAID response surface model}
\usage{
evalBraidModel(DA, DB, bpar, calcderivs = FALSE)
}
\arguments{
\item{DA}{A vector of concentrations of drug A in a combination (values 0
and \code{Inf} are permitted). Must be length 1 or the same length as \code{DB}.}

\item{DB}{A vector of concentrations of drug B in a combination (values 0
and \code{Inf} are permitted). Must be length 1 or the same length as \code{DA}.}

\item{bpar}{A BRAID response surface parameter vector (see Details)}

\item{calcderivs}{Primarily used by fitting functions for non-linear
optimization.  If \code{FALSE} (the default), the function returns a vector of
response values; if \code{TRUE}, it returns a list including the partial
derivatives of the BRAID parameters.}
}
\value{
If \code{calcderivs} is \code{FALSE}, a numeric vector the same length as \code{DA}
and/or \code{DB} with the predicted BRAID response surface values.  If
\code{calcderivs} is \code{TRUE}, a list with two elements: \code{value}, containing the
response surface values, and \code{derivatives}, a matrix with as many rows as
\code{value} has elements, and nine columns containing the partial derivatives of
the response surface with respect to the nine BRAID response surface
parameters
}
\description{
Evaluate the BRAID response surface model
}
\details{
The BRAID response model is, in total, described by nine response surface
parameters.  A BRAID parameter vector should uniquely determine all these
values. They are
\itemize{
\item IDMA: The dose of median effect of drug A, also called the EC50
\item IDMB: The dose of median effect of drug B
\item na: The Hill slope, or sigmoidicity, of drug A
\item nb: The Hill slope of drug B
\item kappa: The BRAID interaction parameter, indicating additivity (kappa = 0),
antagonism (2 < kappa < 0), or synergy (kappa > 0)
\item E0: The minimal effect, the effect observed when neither drug is present
\item EfA: The maximal effect of drug A, the effect theoretically observed when
drug B is absent and drug A is present at infinite concentration
\item EfB: The maximal effect of drug B,
\item Ef: The maximal effect of the combination, theoretically observed when
both drugs are present at infinite concentration. It may be (but often is
not) further from E0 than either EfA or EfB.
}

In many cases, however, it is easier to specify only some of the final three
parameters.  \link{braidrm} functions therefore support BRAID parameter vectors
of length 7 (in which the sixth and seventh values are assumed to be E0 and
Ef, and EfA and EfB are assumed to be equal to Ef), length 8 (in which the
seventh and eighth values are EfA and EfB, and Ef is assumed to be equal to
whichever of these two values is further from E0), or the full length 9
parameter vector.
}
\examples{
concentrations <- c(0, 2^(-3:3))
surface <- data.frame(
    concA = rep(concentrations,each=length(concentrations)),
    concB = rep(concentrations,times=length(concentrations))
)

surface$additive <- evalBraidModel(
    surface$concA,
    surface$concB,
    c(1, 1, 3, 3, 0, 0, 100, 100, 100)
)

surface$synergy <- evalBraidModel(
    surface$concA,
    surface$concB,
    c(1, 1, 3, 3, 2, 0, 80, 90)
)

surface$antagonism <- evalBraidModel(
    surface$concA,
    surface$concB,
    c(1, 1, 3, 3, -1, 0, 100)
)

head(surface)
}
