% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/causalCmprsk-functions.R
\name{fit.cox}
\alias{fit.cox}
\title{Cox-based estimation of ATE corresponding to the target population}
\usage{
fit.cox(
  df,
  X,
  E,
  A,
  C = NULL,
  wtype = "unadj",
  cens = 0,
  conf.level = 0.95,
  bs = FALSE,
  nbs.rep = 400,
  seed = 17,
  parallel = FALSE,
  verbose = FALSE
)
}
\arguments{
\item{df}{a data frame that includes time-to-event \code{X}, type of event \code{E},
a treatment indicator \code{A}  and covariates \code{C}.}

\item{X}{a character string specifying the name of the time-to-event variable in \code{df}.}

\item{E}{a character string specifying the name of the "event type" variable in \code{df}.}

\item{A}{a character specifying the name of the treatment/exposure variable.
It is assumed that \code{A} is a numeric binary indicator with 0/1 values, where \code{A}=1
is assumed a treatment group, and \code{A}=0 a control group.}

\item{C}{a vector of character strings with variable names (potential confounders)
in the logistic regression model for Propensity Scores, i.e. P(A=1|C=c).
The default value of \code{C} is NULL corresponding to \code{wtype}="unadj"
that will estimate treatment effects in the raw (observed) data.}

\item{wtype}{a character string variable indicating the type of weights that will define the target
population for which the ATE will be estimated.
The default is "unadj" - this will not adjust for possible
treatment selection bias and will not use propensity scores weighting. It can be used, for example,
in data from a randized controlled trial (RCT) where there is no need for emulation of baseline randomization.
Other possible values are "stab.ATE", "ATE", "ATT", "ATC" and "overlap".
See Table 1 from Li, Morgan, and Zaslavsky (2018).
"stab.ATE" is defined as P(A=a)/P(A=a|C=c) - see Hernan et al. (2000).}

\item{cens}{an integer value in \code{E} that corresponds to censoring times recorded in \code{X}.
By default \code{fit.nonpar} assumes \code{cens}=0}

\item{conf.level}{the confidence level that will be used in the bootstrap confidence intervals.
The default is 0.95}

\item{bs}{a logical flag indicating whether to perform bootstrap in order to obtain confidence intervals. There are no
analytical confidence intervals in \code{fit.nonpar}}

\item{nbs.rep}{number of bootstrap replications}

\item{seed}{the random seed for the bootstrap, in order to make the results reproducible}

\item{parallel}{a logical flag indicating whether to perform bootstrap sequentially or in parallel
using 2 cores simultaneously. The default value is FALSE}

\item{verbose}{a logical flag indicating whether to show a progress of bootstrap.
The progress bar is shown only for sequential bootstrap computation.
The default value is FALSE.}
}
\value{
A list with the following fields:
\tabular{ll}{
\code{time}  \tab \cr a vector of time points for which all the parameters are estimated \cr
\code{trt.0} \tab \cr a list of estimates of the absolute counterfactual parameters
corresponding to \code{A}=0 and the type of event \code{E}. \code{trt.0} has the number of
 fields as the number of different types of events in the data set.
For each type of event there is a list of estimates:  \cr }
\itemize{
\item{\code{CumHaz}} a vector of cumulative hazard estimates
\item{\code{CIF}} a vector of cumulative incidence functions
\item{\code{RMT}} a vector of restricted mean time estimates}
\tabular{ll}{
\code{trt.1} \tab \cr a list of estimates of the absolute counterfactual parameters
corresponding to \code{A}=1 and the type of event \code{E}. \code{trt.1} has the number of
 fields as the number of different types of events  in the data set.
For each type of event there is a list of estimates:  \cr }
\itemize{
\item{\code{CumHaz}} a vector of cumulative hazard estimates
\item{\code{CIF}} a vector of cumulative incidence functions
\item{\code{RMT}} a vector of restricted mean time estimates}
\tabular{ll}{
\code{trt.eff} \tab \cr a list of estimates of the treatment effect measures
corresponding to the type of event \code{E}. \code{trt.eff} has the number of
 fields as the number of different types of events  in the data set.
For each type of event there is a list of estimates: \cr}
\itemize{
\item{\code{log.CumHazR}} an estimate of the log of the hazard ratio. It is a scalar since the Cox model is assumed.
\item{\code{RD}} a vector of time-varying risk difference between two treatment arms
\item{\code{RR}} a vector of time-varying risk ratio between two treatment arms
\item{\code{ATE.RMT}} a vector of the time-varying restricted mean time difference
 between two treatment arms}
}
\description{
Implements Cox-based estimation  of ATE assuming a structural proportional hazards model for two potential outcomes.
It provides three measures of treatment effects on time-to-event outcomes:
(1) cause-specific hazard ratios which are time-dependent measures under a nonparametric model,
(2) risk-based measures such as cause-specific risk differences and cause-specific risk ratios, and
(3) restricted-mean-time differences which quantify how much time on average was lost (or gained)
due to treatment by some specified time point.
Please see our package vignette for more details.
}
\examples{
# create a data set
n <- 1000
set.seed(7)
c1 <- runif(n)
c2 <- as.numeric(runif(n)< 0.2)
set.seed(77)
cf.m.T1 <- rweibull(n, shape=1, scale=exp(-(-1 + 2*c1)))
cf.m.T2 <-  rweibull(n, shape=1, scale=exp(-(1 + 1*c2)))
cf.m.T <- pmin( cf.m.T1, cf.m.T2)
cf.m.E <- rep(0, n)
cf.m.E[cf.m.T1<=cf.m.T2] <- 1
cf.m.E[cf.m.T2<cf.m.T1] <- 2
set.seed(77)
cf.s.T1 <- rweibull(n, shape=1, scale=exp(-1*c1 ))
cf.s.T2 <-  rweibull(n, shape=1, scale=exp(-2*c2))
cf.s.T <- pmin( cf.s.T1, cf.s.T2)
cf.s.E <- rep(0, n)
cf.s.E[cf.s.T1<=cf.s.T2] <- 1
cf.s.E[cf.s.T2<cf.s.T1] <- 2
exp.z <- exp(0.5 + 1*c1 - 1*c2)
pr <- exp.z/(1+exp.z)
TRT <- ifelse(runif(n)< pr, 1, 0)
X <- ifelse(TRT==1, cf.m.T, cf.s.T)
E <- ifelse(TRT==1, cf.m.E, cf.s.E)
covs.names <- c("c1", "c2")
data <- data.frame(X=X, E=E, TRT=TRT, c1=c1, c2=c2)
wei <- get.weights(data, "TRT", covs.names, wtype = "overlap")
hist(wei$ps[data$TRT==1], col="red", breaks = seq(0,1,0.05))
par(new=TRUE)
hist(wei$ps[data$TRT==0], col="blue", breaks = seq(0,1,0.05))
# Cox-based estimation:
res.cox.ATE <- fit.cox(df=data, X="X", E="E", A="TRT", C=covs.names, wtype="stab.ATE")
cox.pe <- get.pointEst(res.cox.ATE, 0.5)
cox.pe$trt.eff[[1]]$RD #-0.1894038

# please see our package vignette for practical examples

}
\references{
F. Li, K.L. Morgan, and A.M. Zaslavsky. 2018. Balancing Covariates via Propensity Score Weighting. Journal of the American Statistical Association 113 (521): 390–400.

M. Hernan, K.L. Morgan, and A.M. Zaslavsky. 2000. Balancing Covariates via Propensity Score Weighting. Journal of the American Statistical Association 113 (521): 390–400.
}
\seealso{
\code{\link{fit.nonpar}}, \code{\link{get.pointEst}}, \code{\link{causalCmprsk}}
}
