% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/await.R
\name{await}
\alias{await}
\alias{await_all}
\title{Call a function repeatedly until a status is reached.}
\usage{
await(f, ..., .status_key = "state", .success_states = c("succeeded",
  "success"), .error_states = c("failed", "cancelled"),
  .timeout = NULL,
  .interval = getOption("civis.default_polling_interval"),
  .verbose = FALSE)

await_all(f, .x, .y = NULL, ..., .status_key = "state",
  .success_states = c("succeeded", "success"),
  .error_states = c("failed", "cancelled"), .timeout = NULL,
  .interval = NULL, .verbose = FALSE)
}
\arguments{
\item{f}{function to be called repeatedly until a status is reached.}

\item{...}{arguments to \code{f}}

\item{.status_key}{The name of the element of the list returned by \code{f} containing the status.
For most Civis API endpoints, this is the default, \code{"state"}.}

\item{.success_states}{list of states indicating remote work has completed successfully.
For most Civis API endpoints, this set of states is the default, \code{c("succeeded", "success")}.}

\item{.error_states}{list of states indicating remote work is in an error state. For most Civis
API endpoints, this set of states is the default, \code{c("failed", "cancelled")}.}

\item{.timeout}{Number of seconds after which to timeout.}

\item{.interval}{The interval for retries (in seconds). If \code{NULL} (default), use exponentially increasing
intervals with jitter (see 'Details')}

\item{.verbose}{Print the status of \code{f} at a given retry with the retry time (default \code{FALSE})}

\item{.x}{a vector of values to be passed to \code{f}}

\item{.y}{a vector of values to be passed to \code{f} (default \code{NULL})}
}
\description{
\code{await} repeatedly calls a Civis API endpoint such as \code{scripts_get_sql_runs}
that monitors the status of a script, job, import, or model. It blocks until the function
returns a result with a successful or error status.
If the script, job, import or model results in an error state,
\code{await} throws an error with useful debugging information.

\code{await_all} is a vectorized version of \code{await}. It repeatedly calls a Civis API endpoint for
all values of a vector, e.g. a vector of script, job, import, run, or model ids. It blocks until
all calls have returned a result with a given status, and silently captures jobs that return
errors.
}
\details{
\code{await} and \code{await_all} can wrap Civis API endpoints in \code{generated_client.R}.
The default values for \code{.status_key}, \code{.success_states}, and \code{.error_states}
are suitable for most endpoints. The final status of \code{f} can be obtained using
\code{\link{get_status}}.

If an error state is reached, \code{await} throws a \code{civis_await_error}.
\code{await_all} silently captures and returns a \code{civis_await_error} for any job
reaching an error state as an element in the list of results.

If \code{.timeout} is specified and the job fails to reach a success state
within the time limit, \code{await} throws a \code{civis_timeout_error}.
Likewise, \code{await_all} throws a \code{civis_timeout_error} if all jobs fail to
reach a success state within the time limit.

These errors can be caught using \code{try} or \code{tryCatch}.
Useful debugging information can be returned using \code{\link{get_error}} and \code{\link{fetch_logs}}.

The set of possible states for jobs on Civis platform are:
\code{"succeeded"}, \code{"success"}, \code{"failed"}, \code{"queued"}, \code{"running"},
and \code{"cancelled"}.

Unless \code{.interval} is specified, retries are attempted with exponentially increasing intervals using
\code{.25 * (1.2^i)) + runif(1, 0, .2)}, where \code{i} is the index of the current retry.
Approximate intervals for a given number of retries are as follows:
\itemize{
\item{1-5: .5s}
\item{6-10: 1-5s}
\item{11-19: 5-10s}
\item{20-29: 10s - 1m}
}
The polling interval can be set to a fixed value globally with
\code{options("civis.default_polling_interval" = INTERVAL_IN_SECONDS)}.
}
\section{Functions}{
\itemize{
\item \code{await_all}: Call a function repeatedly for all values of a vector until all have reached a completed status
}}

\examples{
\dontrun{

   # Executing a query
   q_id <- queries_post(db_id, query, n_rows, cred_id)[["id"]]
   r <- await(queries_get, id = q_id)
   get_status(r)

   r <- tryCatch(await(queries_get, id = q_id), error = function(e) e)
   get_error(r)

   r <- try(await(queries_get, id = q_id))
   get_error(r)

   jobs <- c(1234, 5678)
   runs <- c(1234, 5678)
   rs <- await_all(scripts_get_r_runs, .x = jobs, .y = runs)
}
}
\seealso{
\code{\link{get_status}, \link{get_error}, \link{fetch_logs}}
}
