get_trans_df <- function(rec) {
  return(map_df(
    xmlChildren(rec[["transporters"]]),
    ~ get_organizm_rec(.x, xmlValue(rec["drugbank-id"][[1]]))
  ))
}

get_trans_actions_df <- function(rec) {
  return(map_df(xmlChildren(rec[["transporters"]]), ~ drug_sub_df(.x,
    "actions",
    id = "id"
  )))
}

get_trans_articles_df <- function(rec) {
  return(map_df(
    xmlChildren(rec[["transporters"]]),
    ~ drug_sub_df(.x, "references", seconadary_node = "articles", id = "id")
  ))
}

get_trans_textbooks_df <- function(rec) {
  return(map_df(
    xmlChildren(rec[["transporters"]]),
    ~ drug_sub_df(.x, "references", seconadary_node = "textbooks", id = "id")
  ))
}

get_trans_links_df <- function(rec) {
  return(map_df(
    xmlChildren(rec[["transporters"]]),
    ~ drug_sub_df(.x, "references", seconadary_node = "links", id = "id")
  ))
}

get_trans_poly_df <- function(rec) {
  return(map_df(xmlChildren(rec[["transporters"]]), ~ get_polypeptide_rec(.x)))
}

get_trans_poly_ex_identity_df <-
  function(rec) {
    return(map_df(
      xmlChildren(rec[["transporters"]]),
      ~ get_poly_ex_identity(.x)
    ))
  }

get_trans_poly_syn_df <- function(rec) {
  return(map_df(
    xmlChildren(rec[["transporters"]]),
    ~ get_polypeptide_syn(.x)
  ))
}

get_trans_poly_pfams_df <- function(rec) {
  return(map_df(
    xmlChildren(rec[["transporters"]]),
    ~ get_polypeptide_pfams(.x)
  ))
}

get_trans_poly_go_df <- function(rec) {
  return(map_df(
    xmlChildren(rec[["transporters"]]),
    ~ get_polypeptide_go(.x)
  ))
}

#' Extracts the drug transporters actions element and return data as tibble.
#'
#' \code{transporters actions} returns tibble of drug
#'  transporters
#' actions elements.
#'
#' This functions extracts the transporters actions element of drug node in
#' drugbank
#' xml database with the option to save it in a predefined database via
#' passed database connection. It takes two optional arguments to
#' save the returned tibble in the database \code{save_table} and
#' \code{database_connection}.
#' It must be called after \code{\link{read_drugbank_xml_db}} function like
#' any other parser function.
#' If \code{\link{read_drugbank_xml_db}} is called before for any reason, so
#' no need to call it again before calling this function.
#'
#' @param save_table boolean, save table in database if true.
#' @param save_csv boolean, save csv version of parsed tibble if true
#' @param csv_path location to save csv files into it, default is current
#'  location, save_csv must be true
#' @param override_csv override existing csv, if any, in case it is true in
#'  the new parse operation
#' @param database_connection DBI connection object that holds a connection to
#' user defined database. If \code{save_table} is enabled without providing
#' value for this function an error will be thrown.
#'
#' @return drug transporters actions node attributes date frame
#' @family transporters
#' @examples
#' \dontrun{
#' # return only the parsed tibble
#' transporters_actions()
#'
#' # will throw an error, as database_connection is NULL
#' transporters_actions(save_table = TRUE)
#'
#' # save parsed tibble as csv if it does not exist in current location
#' # and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_actions(save_csv = TRUE)
#'
#' # save in database, save parsed tibble as csv if it does not exist
#' # in current location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_actions(save_table = TRUE, save_csv = TRUE)
#'
#' # save parsed tibble as csv if it does not exist in given location
#' # and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_actions(save_csv = TRUE, csv_path = TRUE)
#'
#' # save parsed tibble as csv if it does not exist in current location
#' #  and return parsed tibble.
#' # If the csv exist override it and return it.
#' transporters_actions(
#'   save_csv = TRUE, csv_path = TRUE,
#'   override = TRUE
#' )
#' }
#' @export
transporters_actions <-
  function(save_table = FALSE,
           save_csv = FALSE,
           csv_path = ".",
           override_csv = FALSE,
           database_connection = NULL) {
    check_parameters_validation(save_table, database_connection)
    path <-
      get_dataset_full_path("drug_trans_actions", csv_path)
    if (!override_csv & file.exists(path)) {
      drug_trans_actions <- readr::read_csv(path)
    } else {
      drug_trans_actions <-
        map_df(pkg_env$children, ~ get_trans_actions_df(.x)) %>%
        unique()

      write_csv(drug_trans_actions, save_csv, csv_path)
    }

    if (nrow(drug_trans_actions) > 0) {
      colnames(drug_trans_actions) <- c("action", "transporter_id")
    }


    if (save_table) {
      save_drug_sub(
        con = database_connection,
        df = drug_trans_actions,
        table_name = "drug_trans_actions",
        save_table_only = TRUE
      )
    }
    return(drug_trans_actions %>% as_tibble())
  }


#' Extracts the drug transporters articles element and return data as
#' tibble.
#'
#' \code{transporters articles} returns tibble of drug
#' transporters articles elements.
#'
#' This functions extracts the transporters articles element of drug node in
#'  drugbank
#' xml database with the option to save it in a predefined database via
#' passed database connection. It takes two optional arguments to
#' save the returned tibble in the database \code{save_table} and
#' \code{database_connection}.
#' It must be called after \code{\link{read_drugbank_xml_db}} function like
#' any other parser function.
#' If \code{\link{read_drugbank_xml_db}} is called before for any reason, so
#' no need to call it again before calling this function.
#'
#' @param save_table boolean, save table in database if true.
#' @param save_csv boolean, save csv version of parsed tibble if true
#' @param csv_path location to save csv files into it, default is current
#' location, save_csv must be true
#' @param override_csv override existing csv, if any, in case it is true in the
#'  new parse operation
#' @param database_connection DBI connection object that holds a connection to
#' user defined database. If \code{save_table} is enabled without providing
#' value for this function an error will be thrown.
#'
#' @return drug transporters articles node attributes date frame
#' @family transporters
#' @examples
#' \dontrun{
#' # return only the parsed tibble
#' transporters_articles()
#'
#' # will throw an error, as database_connection is NULL
#' transporters_articles(save_table = TRUE)
#'
#' # save parsed tibble as csv if it does not exist in current location
#' # and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_articles(save_csv = TRUE)
#'
#' # save in database, save parsed tibble as csv if it does not exist
#' # in current location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_articles(save_table = TRUE, save_csv = TRUE)
#'
#' # save parsed tibble as csv if it does not exist in given location
#' # and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_articles(save_csv = TRUE, csv_path = TRUE)
#'
#' # save parsed tibble as csv if it does not exist in current
#' # location and return parsed tibble.
#' # If the csv exist override it and return it.
#' transporters_articles(
#'   save_csv = TRUE, csv_path = TRUE,
#'   override = TRUE
#' )
#' }
#' @export
transporters_articles <-
  function(save_table = FALSE,
           save_csv = FALSE,
           csv_path = ".",
           override_csv = FALSE,
           database_connection = NULL) {
    check_parameters_validation(save_table, database_connection)
    path <-
      get_dataset_full_path("drug_trans_articles", csv_path)
    if (!override_csv & file.exists(path)) {
      drug_trans_articles <- readr::read_csv(path)
    } else {
      drug_trans_articles <-
        map_df(pkg_env$children, ~ get_trans_articles_df(.x)) %>%
        unique()

      write_csv(drug_trans_articles, save_csv, csv_path)
    }

    if (save_table) {
      save_drug_sub(
        con = database_connection,
        df = drug_trans_articles,
        table_name = "drug_trans_articles",
        save_table_only = TRUE
      )
    }
    return(drug_trans_articles %>% as_tibble())
  }


#' Extracts the drug transporters textbooks element and return data as
#'  tibble.
#'
#' \code{transporters_textbooks} returns tibble of drug
#' transporters
#'  textbooks elements.
#'
#' This functions extracts the transporters textbooks element of drug node in
#' drugbank
#' xml database with the option to save it in a predefined database via
#' passed database connection. It takes two optional arguments to
#' save the returned tibble in the database \code{save_table} and
#' \code{database_connection}.
#' It must be called after \code{\link{read_drugbank_xml_db}} function like
#' any other parser function.
#' If \code{\link{read_drugbank_xml_db}} is called before for any reason, so
#' no need to call it again before calling this function.
#'
#' @param save_table boolean, save table in database if true.
#' @param save_csv boolean, save csv version of parsed tibble if true
#' @param csv_path location to save csv files into it, default is current
#' location, save_csv must be true
#' @param override_csv override existing csv, if any, in case it is true in the
#'  new parse operation
#' @param database_connection DBI connection object that holds a connection to
#' user defined database. If \code{save_table} is enabled without providing
#' value for this function an error will be thrown.
#'
#' @return drug transporters textbooks node attributes date frame
#' @family transporters
#' @examples
#' \dontrun{
#' # return only the parsed tibble
#' transporters_textbooks()
#'
#' # will throw an error, as database_connection is NULL
#' transporters_textbooks(save_table = TRUE)
#'
#' # save in database in SQLite in memory database and return parsed tibble
#' sqlite_con <- DBI::dbConnect(RSQLite::SQLite())
#' transporters_textbooks(save_table = TRUE, database_connection = sqlite_con)
#'
#' # save parsed tibble as csv if it does not exist in current
#' # location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_textbooks(save_csv = TRUE)
#'
#' # save in database, save parsed tibble as csv if it does not exist
#' # in current location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_textbooks(save_table = TRUE, save_csv = TRUE)
#'
#' # save parsed tibble as csv if it does not exist in given location
#' #  and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_textbooks(save_csv = TRUE, csv_path = TRUE)
#'
#' # save parsed tibble as csv if it does not exist in current
#' # location and return parsed tibble.
#' # If the csv exist override it and return it.
#' transporters_textbooks(
#'   save_csv = TRUE, csv_path = TRUE,
#'   override = TRUE
#' )
#' }
#' @export
transporters_textbooks <-
  function(save_table = FALSE,
           save_csv = FALSE,
           csv_path = ".",
           override_csv = FALSE,
           database_connection = NULL) {
    check_parameters_validation(save_table, database_connection)
    path <-
      get_dataset_full_path("drug_trans_textbooks", csv_path)
    if (!override_csv & file.exists(path)) {
      drug_trans_textbooks <- readr::read_csv(path)
    } else {
      drug_trans_textbooks <-
        map_df(pkg_env$children, ~ get_trans_textbooks_df(.x)) %>%
        unique()

      write_csv(drug_trans_textbooks, save_csv, csv_path)
    }

    if (save_table) {
      save_drug_sub(
        con = database_connection,
        df = drug_trans_textbooks,
        table_name = "drug_trans_textbooks",
        save_table_only = TRUE
      )
    }
    return(drug_trans_textbooks %>% as_tibble())
  }


#' Extracts the drug transporters links element and return data as tibble.
#'
#' \code{transporters_links} returns tibble of drug
#' transporters_links elements.
#'
#' This functions extracts the transporters links element of drug node in
#' drugbank
#' xml database with the option to save it in a predefined database via
#' passed database connection. It takes two optional arguments to
#' save the returned tibble in the database \code{save_table} and
#' \code{database_connection}.
#' It must be called after \code{\link{read_drugbank_xml_db}} function like
#' any other parser function.
#' If \code{\link{read_drugbank_xml_db}} is called before for any reason, so
#' no need to call it again before calling this function.
#'
#' @param save_table boolean, save table in database if true.
#' @param save_csv boolean, save csv version of parsed tibble if true
#' @param csv_path location to save csv files into it, default is current
#' location, save_csv must be true
#' @param override_csv override existing csv, if any, in case it is true in the
#'  new parse operation
#' @param database_connection DBI connection object that holds a connection to
#' user defined database. If \code{save_table} is enabled without providing
#' value for this function an error will be thrown.
#'
#' @return drug transporters links node attributes date frame
#' @family transporters
#' @examples
#' \dontrun{
#' # return only the parsed tibble
#' transporters_links()
#'
#' # will throw an error, as database_connection is NULL
#' transporters_links(save_table = TRUE)
#'
#' # save in database in SQLite in memory database and return parsed tibble
#' sqlite_con <- DBI::dbConnect(RSQLite::SQLite())
#' transporters_textbooks(save_table = TRUE, database_connection = sqlite_con)
#'
#' # save parsed tibble as csv if it does not exist in current
#' # location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_links(save_csv = TRUE)
#'
#' # save in database, save parsed tibble as csv if it does not exist
#' # in current location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_links(save_table = TRUE, save_csv = TRUE)
#'
#' # save parsed tibble as csv if it does not exist in given
#' # location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_links(save_csv = TRUE, csv_path = TRUE)
#'
#' # save parsed tibble as csv if it does not exist in current
#' # location and return parsed tibble.
#' # If the csv exist override it and return it.
#' transporters_links(
#'   save_csv = TRUE, csv_path = TRUE,
#'   override = TRUE
#' )
#' }
#' @export
transporters_links <-
  function(save_table = FALSE,
           save_csv = FALSE,
           csv_path = ".",
           override_csv = FALSE,
           database_connection = NULL) {
    check_parameters_validation(save_table, database_connection)
    path <-
      get_dataset_full_path("drug_trans_links", csv_path)
    if (!override_csv & file.exists(path)) {
      drug_trans_links <- readr::read_csv(path)
    } else {
      drug_trans_links <-
        map_df(pkg_env$children, ~ get_trans_links_df(.x)) %>%
        unique()

      write_csv(drug_trans_links, save_csv, csv_path)
    }

    if (save_table) {
      save_drug_sub(
        con = database_connection,
        df = drug_trans_links,
        table_name = "drug_trans_links",
        save_table_only = TRUE
      )
    }
    return(drug_trans_links %>% as_tibble())
  }


#' Extracts the drug transporters polypeptides element and return data
#'  as tibble.
#'
#' \code{transporters_polypeptide} returns tibble of
#' transporters polypeptides groups elements.
#'
#' This functions extracts the transporters polypeptides element of drug
#' node in \strong{DrugBank}
#' xml database with the option to save it in a predefined database via
#' passed database connection. It takes two optional arguments to
#' save the returned tibble in the database \code{save_table} and
#' \code{database_connection}.
#' It must be called after \code{\link{read_drugbank_xml_db}} function like
#' any other parser function.
#' If \code{\link{read_drugbank_xml_db}} is called before for any reason, so
#' no need to call it again before calling this function.
#'
#' @param save_table boolean, save table in database if true.
#' @param save_csv boolean, save csv version of parsed tibble if true
#' @param csv_path location to save csv files into it, default is current
#'  location, save_csv must be true
#' @param override_csv override existing csv, if any, in case it is true in the
#'  new parse operation
#' @param database_connection DBI connection object that holds a connection to
#' user defined database. If \code{save_table} is enabled without providing
#' value for this function an error will be thrown.
#'
#' @return drug transporters polypeptides node attributes date frame
#' @family transporters
#' @examples
#' \dontrun{
#' # return only the parsed tibble
#' transporters_polypeptide()
#'
#' # will throw an error, as database_connection is NULL
#' transporters_polypeptide(save_table = TRUE)
#'
#' # save in database in SQLite in memory database and return parsed tibble
#' sqlite_con <- DBI::dbConnect(RSQLite::SQLite())
#' transporters_textbooks(save_table = TRUE, database_connection = sqlite_con)
#'
#' # save parsed tibble as csv if it does not exist in current
#' # location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_polypeptide(save_csv = TRUE)
#'
#' # save in database, save parsed tibble as csv if it does not
#' # exist in current location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_polypeptide(save_table = TRUE, save_csv = TRUE)
#'
#' # save parsed tibble as csv if it does not exist in given
#' # location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_polypeptide(save_csv = TRUE, csv_path = TRUE)
#'
#' # save parsed tibble as csv if it does not exist in current
#' # location and return parsed tibble.
#' # If the csv exist override it and return it.
#' transporters_polypeptide(
#'   save_csv = TRUE, csv_path = TRUE,
#'   override = TRUE
#' )
#' }
#' @export
transporters_polypeptide <-
  function(save_table = FALSE,
           save_csv = FALSE,
           csv_path = ".",
           override_csv = FALSE,
           database_connection = NULL) {
    check_parameters_validation(save_table, database_connection)
    path <-
      get_dataset_full_path("drug_trans_polys", csv_path)
    if (!override_csv & file.exists(path)) {
      drug_trans_polys <- readr::read_csv(path)
    } else {
      drug_trans_polys <-
        map_df(pkg_env$children, ~ get_trans_poly_df(.x)) %>%
        unique()

      write_csv(drug_trans_polys, save_csv, csv_path)
    }

    if (save_table) {
      save_drug_sub(
        con = database_connection,
        df = drug_trans_polys,
        table_name = "drug_trans_polys",
        save_table_only = TRUE,
        field_types = list(
          general_function =
            paste("varchar(",
              max(
                nchar(drug_trans_polys$general_function),
                na.rm = TRUE
              ), ")",
              sep = ""
            ),
          specific_function =
            paste("varchar(",
              max(
                nchar(drug_trans_polys$specific_function),
                na.rm = TRUE
              ), ")",
              sep = ""
            ),
          amino_acid_sequence =
            paste("varchar(",
              max(
                nchar(drug_trans_polys$amino_acid_sequence),
                na.rm = TRUE
              ), ")",
              sep = ""
            ),
          gene_sequence = paste("varchar(max)", sep = "")
        )
      )
    }
    return(drug_trans_polys %>% as_tibble())
  }


#' Extracts the drug transporters polypeptides external identifiers
#'  element and return data as tibble.
#'
#' \code{transporters_polypep_ex_ident}
#' returns tibble of drug transporters polypeptides external identifiers
#' elements.
#'
#' This functions extracts the transporters polypeptides external
#'  identifiers element of drug node in \strong{DrugBank}
#' xml database with the option to save it in a predefined database via
#' passed database connection. It takes two optional arguments to
#' save the returned tibble in the database \code{save_table} and
#' \code{database_connection}.
#' It must be called after \code{\link{read_drugbank_xml_db}} function like
#' any other parser function.
#' If \code{\link{read_drugbank_xml_db}} is called before for any reason, so
#' no need to call it again before calling this function.
#'
#' @param save_table boolean, save table in database if true.
#' @param save_csv boolean, save csv version of parsed tibble if true
#' @param csv_path location to save csv files into it, default is current
#'  location, save_csv must be true
#' @param override_csv override existing csv, if any, in case it is true in the
#'  new parse operation
#' @param database_connection DBI connection object that holds a connection to
#' user defined database. If \code{save_table} is enabled without providing
#' value for this function an error will be thrown.
#'
#' @return drug transporters polypeptides external identifiers
#'  node attributes date frame
#' @family transporters
#' @examples
#' \dontrun{
#' # return only the parsed tibble
#' transporters_polypep_ex_ident()
#'
#' # will throw an error, as database_connection is NULL
#' transporters_polypep_ex_ident(save_table = TRUE)
#'
#' # save in database in SQLite in memory database and return parsed tibble
#' sqlite_con <- DBI::dbConnect(RSQLite::SQLite())
#' transporters_textbooks(save_table = TRUE, database_connection = sqlite_con)
#'
#' # save parsed tibble as csv if it does not exist in current
#' # location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_polypep_ex_ident(save_csv = TRUE)
#'
#' # save in database, save parsed tibble as csv if it does not exist
#' # in current location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_polypep_ex_ident(
#'   save_table = TRUE, save_csv = TRUE
#' )
#'
#' # save parsed tibble as csv if it does not exist in given location
#' # and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_polypep_ex_ident(
#'   save_csv = TRUE, csv_path = TRUE
#' )
#'
#' # save parsed tibble as csv if it does not exist in current
#' # location and return parsed tibble.
#' # If the csv exist override it and return it.
#' transporters_polypep_ex_ident(
#'   save_csv = TRUE, csv_path = TRUE, override = TRUE
#' )
#' }
#' @export
transporters_polypep_ex_ident <-
  function(save_table = FALSE,
           save_csv = FALSE,
           csv_path = ".",
           override_csv = FALSE,
           database_connection = NULL) {
    check_parameters_validation(save_table, database_connection)
    path <-
      get_dataset_full_path(
        "drug_trans_poly_ex_identity",
        csv_path
      )
    if (!override_csv & file.exists(path)) {
      drug_trans_poly_ex_identity <-
        readr::read_csv(path)
    } else {
      drug_trans_poly_ex_identity <-
        map_df(
          pkg_env$children,
          ~ get_trans_poly_ex_identity_df(.x)
        ) %>%
        unique()

      write_csv(
        drug_trans_poly_ex_identity,
        save_csv,
        csv_path
      )
    }

    if (save_table) {
      save_drug_sub(
        con = database_connection,
        df = drug_trans_poly_ex_identity,
        table_name = "drug_trans_polys_external_identifiers",
        save_table_only = TRUE
      )
    }
    return(drug_trans_poly_ex_identity %>% as_tibble())
  }


#' Extracts the drug transporters polypeptides syn
#' element and return data as tibble.
#'
#' \code{transporters_polypeptide_syn} returns data
#' frame of drug transporters polypeptides syn elements.
#'
#' This functions extracts the transporters polypeptides syn
#'  element of drug node in \strong{DrugBank}
#' xml database with the option to save it in a predefined database via
#' passed database connection. It takes two optional arguments to
#' save the returned tibble in the database \code{save_table} and
#' \code{database_connection}.
#' It must be called after \code{\link{read_drugbank_xml_db}} function like
#' any other parser function.
#' If \code{\link{read_drugbank_xml_db}} is called before for any reason, so
#' no need to call it again before calling this function.
#'
#' @param save_table boolean, save table in database if true.
#' @param save_csv boolean, save csv version of parsed tibble if true
#' @param csv_path location to save csv files into it, default is current
#' location, save_csv must be true
#' @param override_csv override existing csv, if any, in case it is true in the
#'  new parse operation
#' @param database_connection DBI connection object that holds a connection to
#' user defined database. If \code{save_table} is enabled without providing
#' value for this function an error will be thrown.
#'
#' @return drug transporters polypeptides syn node attributes date frame
#' @family transporters
#' @examples
#' \dontrun{
#' # return only the parsed tibble
#' transporters_polypeptide_syn()
#'
#' # will throw an error, as database_connection is NULL
#' transporters_polypeptide_syn(save_table = TRUE)
#'
#' # save in database in SQLite in memory database and return parsed tibble
#' sqlite_con <- DBI::dbConnect(RSQLite::SQLite())
#' transporters_textbooks(save_table = TRUE, database_connection = sqlite_con)
#'
#' # save parsed tibble as csv if it does not exist in current
#' # location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_polypeptide_syn(save_csv = TRUE)
#'
#' # save in database, save parsed tibble as csv if it does not
#' # exist in current location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_polypeptide_syn(
#'   save_table = TRUE,
#'   save_csv = TRUE
#' )
#'
#' # save parsed tibble as csv if it does not exist in given
#' # location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_polypeptide_syn(
#'   save_csv = TRUE,
#'   csv_path = TRUE
#' )
#'
#' # save parsed tibble as csv if it does not exist in current
#' # location and return parsed tibble.
#' # If the csv exist override it and return it.
#' transporters_polypeptide_syn(
#'   save_csv = TRUE,
#'   csv_path = TRUE, override = TRUE
#' )
#' }
#' @export
transporters_polypeptide_syn <-
  function(save_table = FALSE,
           save_csv = FALSE,
           csv_path = ".",
           override_csv = FALSE,
           database_connection = NULL) {
    check_parameters_validation(save_table, database_connection)
    path <-
      get_dataset_full_path("drug_trans_poly_syn", csv_path)
    if (!override_csv & file.exists(path)) {
      drug_trans_poly_syn <- readr::read_csv(path)
    } else {
      drug_trans_poly_syn <-
        map_df(
          pkg_env$children,
          ~ get_trans_poly_syn_df(.x)
        ) %>%
        unique()

      write_csv(
        drug_trans_poly_syn,
        save_csv,
        csv_path
      )
    }

    if (save_table) {
      save_drug_sub(
        con = database_connection,
        df = drug_trans_poly_syn,
        table_name = "drug_trans_polys_syn",
        save_table_only = TRUE
      )
    }
    return(drug_trans_poly_syn %>% as_tibble())
  }


#' Extracts the drug transporters polypeptides pfams element
#' and return data as tibble.
#'
#' \code{transporters_polypeptide_pfams} returns tibble
#'  of drug transporters polypeptides pfams elements.
#'
#' This functions extracts the transporters polypeptides pfams
#'  element of drug node in \strong{DrugBank}
#' xml database with the option to save it in a predefined database via
#' passed database connection. It takes two optional arguments to
#' save the returned tibble in the database \code{save_table} and
#' \code{database_connection}.
#' It must be called after \code{\link{read_drugbank_xml_db}} function like
#' any other parser function.
#' If \code{\link{read_drugbank_xml_db}} is called before for any reason, so
#' no need to call it again before calling this function.
#'
#' @param save_table boolean, save table in database if true.
#' @param save_csv boolean, save csv version of parsed tibble if true
#' @param csv_path location to save csv files into it, default is current
#' location, save_csv must be true
#' @param override_csv override existing csv, if any, in case it is true in the
#'  new parse operation
#' @param database_connection DBI connection object that holds a connection to
#' user defined database. If \code{save_table} is enabled without providing
#' value for this function an error will be thrown.
#'
#' @return drug transporters polypeptides pfams node attributes date frame
#' @family transporters
#' @examples
#' \dontrun{
#' # return only the parsed tibble
#' transporters_polypeptide_pfams()
#'
#' # will throw an error, as database_connection is NULL
#' transporters_polypeptide_pfams(save_table = TRUE)
#'
#' # save in database in SQLite in memory database and return parsed tibble
#' sqlite_con <- DBI::dbConnect(RSQLite::SQLite())
#' transporters_textbooks(save_table = TRUE, database_connection = sqlite_con)
#'
#' # save parsed tibble as csv if it does not exist in current
#' # location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_polypeptide_pfams(save_csv = TRUE)
#'
#' # save in database, save parsed tibble as csv if it does not
#' # exist in current location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_polypeptide_pfams(
#'   save_table = TRUE,
#'   save_csv = TRUE
#' )
#'
#' # save parsed tibble as csv if it does not exist in given
#' # location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_polypeptide_pfams(save_csv = TRUE, csv_path = TRUE)
#'
#' # save parsed tibble as csv if it does not exist in current
#' # location and return parsed tibble.
#' # If the csv exist override it and return it.
#' transporters_polypeptide_pfams(
#'   save_csv = TRUE, csv_path = TRUE,
#'   override = TRUE
#' )
#' }
#' @export
transporters_polypeptide_pfams <-
  function(save_table = FALSE,
           save_csv = FALSE,
           csv_path = ".",
           override_csv = FALSE,
           database_connection = NULL) {
    check_parameters_validation(save_table, database_connection)
    path <-
      get_dataset_full_path("drug_trans_polys_pfams", csv_path)
    if (!override_csv & file.exists(path)) {
      drug_trans_polys_pfams <- readr::read_csv(path)
    } else {
      drug_trans_polys_pfams <-
        map_df(
          pkg_env$children,
          ~ get_trans_poly_pfams_df(.x)
        ) %>%
        unique()

      write_csv(drug_trans_polys_pfams, save_csv, csv_path)
    }


    if (save_table) {
      save_drug_sub(
        con = database_connection,
        df = drug_trans_polys_pfams,
        table_name = "drug_trans_polys_pfams",
        save_table_only = TRUE
      )
    }
    return(drug_trans_polys_pfams %>% as_tibble())
  }


#' Extracts the drug transporters polypeptides go
#' classifiers element and return data as tibble.
#'
#' \code{transporters_polypeptide_go}
#' returns tibble of drug transporters polypeptides
#' go classifiers elements.
#'
#' This functions extracts the transporters polypeptides go
#'  classifiers element of drug node in \strong{DrugBank}
#' xml database with the option to save it in a predefined database via
#' passed database connection. It takes two optional arguments to
#' save the returned tibble in the database \code{save_table} and
#' \code{database_connection}.
#' It must be called after \code{\link{read_drugbank_xml_db}} function like
#' any other parser function.
#' If \code{\link{read_drugbank_xml_db}} is called before for any reason, so
#' no need to call it again before calling this function.
#'
#' @param save_table boolean, save table in database if true.
#' @param save_csv boolean, save csv version of parsed tibble if true
#' @param csv_path location to save csv files into it, default is current
#' location, save_csv must be true
#' @param override_csv override existing csv, if any, in case it is true in the
#' new parse operation
#' @param database_connection DBI connection object that holds a connection to
#' user defined database. If \code{save_table} is enabled without providing
#' value for this function an error will be thrown.
#'
#' @return drug transporters polypeptides go classifiers node attributes
#' date frame
#' @family transporters
#' @examples
#' \dontrun{
#' # return only the parsed tibble
#' transporters_polypeptide_go()
#'
#' # will throw an error, as database_connection is NULL
#' transporters_polypeptide_go(save_table = TRUE)
#'
#' # save in database in SQLite in memory database and return parsed tibble
#' sqlite_con <- DBI::dbConnect(RSQLite::SQLite())
#' transporters_textbooks(save_table = TRUE, database_connection = sqlite_con)
#'
#' # save parsed tibble as csv if it does not exist in current
#' # location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_polypeptide_go(save_csv = TRUE)
#'
#' # save in database, save parsed tibble as csv if it does not exist in
#' #  current location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_polypeptide_go(
#'   save_table = TRUE,
#'   save_csv = TRUE
#' )
#'
#' # save parsed tibble as csv if it does not exist in given location
#' # and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters_polypeptide_go(
#'   save_csv = TRUE,
#'   csv_path = TRUE
#' )
#'
#' # save parsed tibble as csv if it does not exist in current
#' # location and return parsed tibble.
#' # If the csv exist override it and return it.
#' transporters_polypeptide_go(
#'   save_csv = TRUE, csv_path = TRUE, override = TRUE
#' )
#' }
#' @export
transporters_polypeptide_go <-
  function(save_table = FALSE,
           save_csv = FALSE,
           csv_path = ".",
           override_csv = FALSE,
           database_connection = NULL) {
    check_parameters_validation(save_table, database_connection)
    path <-
      get_dataset_full_path(
        "drug_trans_polys_go",
        csv_path
      )
    if (!override_csv & file.exists(path)) {
      drug_trans_polys_go <-
        readr::read_csv(path)
    } else {
      drug_trans_polys_go <-
        map_df(
          pkg_env$children,
          ~ get_trans_poly_go_df(.x)
        ) %>%
        unique()

      write_csv(
        drug_trans_polys_go,
        save_csv,
        csv_path
      )
    }


    if (save_table) {
      save_drug_sub(
        con = database_connection,
        df = drug_trans_polys_go,
        table_name = "drug_trans_polys_go",
        save_table_only = TRUE
      )
    }
    return(drug_trans_polys_go %>% as_tibble())
  }




#' Extracts the drug transporters element and return data as tibble.
#'
#' \code{transporters} returns tibble of drug transporters
#' elements.
#'
#' This functions extracts the transporters element of drug node in
#'  \strong{DrugBank}
#' xml database with the option to save it in a predefined database via
#' passed database connection. It takes two optional arguments to
#' save the returned tibble in the database \code{save_table} and
#' \code{database_connection}.
#' It must be called after \code{\link{read_drugbank_xml_db}} function like
#' any other parser function.
#' If \code{\link{read_drugbank_xml_db}} is called before for any reason, so
#' no need to call it again before calling this function.
#'
#' @param save_table boolean, save table in database if true.
#' @param save_csv boolean, save csv version of parsed tibble if true
#' @param csv_path location to save csv files into it, default is current
#' location, save_csv must be true
#' @param override_csv override existing csv, if any, in case it is true in the
#'  new parse operation
#' @param database_connection DBI connection object that holds a connection to
#' user defined database. If \code{save_table} is enabled without providing
#' value for this function an error will be thrown.
#'
#' @return drug transporters node attributes date frame
#' @family transporters
#' @examples
#' \dontrun{
#' # return only the parsed tibble
#' transporters()
#'
#' # will throw an error, as database_connection is NULL
#' transporters(save_table = TRUE)
#'
#' # save in database in SQLite in memory database and return parsed tibble
#' sqlite_con <- DBI::dbConnect(RSQLite::SQLite())
#' transporters_textbooks(save_table = TRUE, database_connection = sqlite_con)
#'
#' # save parsed tibble as csv if it does not exist in
#' # current location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters(save_csv = TRUE)
#'
#' # save in database, save parsed tibble as csv if it does not
#' # exist in current location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters(save_table = TRUE, save_csv = TRUE,
#' database_connection = sqlite_con)
#'
#' # save parsed tibble as csv if it does not exist in given
#' # location and return parsed tibble.
#' # If the csv exist before read it and return its data.
#' transporters(save_csv = TRUE, csv_path = TRUE)
#'
#' # save parsed tibble as csv if it does not exist in current
#' # location and return parsed tibble.
#' # If the csv exist override it and return it.
#' transporters(save_csv = TRUE, csv_path = TRUE, override = TRUE)
#' }
#' @export
transporters <-
  function(save_table = FALSE,
           save_csv = FALSE,
           csv_path = ".",
           override_csv = FALSE,
           database_connection = NULL) {
    check_parameters_validation(save_table, database_connection)
    path <-
      get_dataset_full_path("drug_transporters", csv_path)
    if (!override_csv & file.exists(path)) {
      drug_transporters <- readr::read_csv(path)
    } else {
      drug_transporters <-
        map_df(pkg_env$children, ~ get_trans_df(.x)) %>%
        unique()

      write_csv(drug_transporters, save_csv, csv_path)
    }


    if (save_table) {
      save_drug_sub(
        con = database_connection,
        df = drug_transporters,
        table_name = "drug_transporters",
        foreign_key = "parent_key"
      )
    }
    return(drug_transporters %>% as_tibble())
  }
