#' Model data getter
#'
#' Get the data with which the distributional regression model of interest was
#' estimated (see \link{distreg_checker} for a list of supported object
#' classes). By default, only explanatory variables are returned.
#' @importFrom methods is
#' @importFrom stats model.frame
#' @param model A gamlss or bamlss object.
#' @param dep If TRUE, then only the dependent variable is returned.
#' @param varname Variable name in character form that should be returned. If
#'   this is specified, only the desired variable is returned.
#' @return A data.frame object if dep or varname is not specified, otherwise a
#'   vector.
#' @examples
#' library("betareg")
#'
#' # Get some data
#' beta_dat <- model_fam_data(fam_name = "betareg")
#'
#' # Estimate model
#' betamod <- betareg(betareg ~ ., data = beta_dat)
#'
#' # Get data
#' model_data(betamod)
#' @export
model_data <- function(model, dep = FALSE, varname = NULL) {

  # Check first if supported distributional regression model
  if (!distreg_checker(model))
    stop("Specified model is not a supported distributional regression object. \nSee ?distreg_checker for details")

  if (dep & !is.null(varname))
    stop("Combination dep = TRUE and a specified varname is not possible.")

  # GAMLSS
  if (is(model, "gamlss")) {

    # Put all together
    data_model <- model.frame(model)

    # Put dep variable in
    all_data <- cbind(model$y, data_model)
    dep_name <- as.character(model$mu.formula)[2] # this works, because in gamlss we do not have multivariate responses
    colnames(all_data)[1] <- dep_name

    # Here we check wether we have splines or identical columns
    all_data <- gamlss_data_cleaner(all_data)

    # Make data.frame out of this
    return_object <- as.data.frame(all_data)

    # If dep then return only dependent variable
    if (dep & is.null(varname))
      return_object <- return_object[[dep_name]]

    # If varname then return varname
    if (!dep & !is.null(varname))
      return_object <- return_object[[varname]]

    # If no varname but no dep then don't return with dep
    if (!dep & is.null(varname))
      return_object <- return_object[, !colnames(return_object) %in% dep_name]

  }

  # BAMLSS
  if (is(model, "bamlss")) {
    return_object <- model$model.frame

    # Return dependent variable if wanted
    if (dep & is.null(varname))
      return_object <- c(model$y)[[1]]

    # Return a specific variable
    if (!dep & !is.null(varname))
      return_object <- return_object[[varname]]

    # If dep is true but varname not specified then return without dep
    if (!dep & is.null(varname))
      return_object <- return_object[, -1]

  }

  # Betareg / Betatree
  if (is(model, "betareg") | is(model, "betatree")) {
    return_object <- model.frame(model)

    # Return dependent variable if wanted
    if (dep & is.null(varname))
      return_object <- return_object[[1]]

    if (!dep) {
      if (!is.null(varname))
        return_object <- return_object[[varname]]
      else if (is.null(varname))
        return_object <- return_object[, c(-1)]
    }
  }
  return(return_object)
}

#' Internal: Function to obtain all explanatory variables used to fit
#'   a model, without the dependent variables
#' @importFrom methods is
#' @keywords internal
expl_vars <- function(model) {
  all_data <- model_data(model)

  # GAMLSS
  if (is(model, "gamlss")) {
    dep_name <- as.character(model$mu.formula)[2]
    expl_vars <- all_data[, !colnames(all_data) %in% dep_name, drop = FALSE]
  }

  # BAMLSS
  if (is(model, "bamlss")) {
    dep_name <- colnames(model$y)
    expl_vars <- all_data[, !colnames(all_data) %in% dep_name, drop = FALSE]
  }
  return(expl_vars)
}

#' GAMLSS expl_data cleaner
#'
#' This checks whether we have spline column names and/or duplicate columns
#' @keywords internal
gamlss_data_cleaner <- function(temp_df) {
  cnames <- colnames(temp_df)

  # Clean of spline and other functions
  broken_up_list <- strsplit(cnames, "[(]|[)|,]")
  new_cnames <- sapply(broken_up_list, FUN = function(x) {
    if (length(x) != 1)
      return(x[2])
    else
      return(x[1])
  })

  # Assign new colnames
  new_df <- temp_df
  colnames(new_df) <- new_cnames

  # Only retain unique columns
  new_df <- new_df[, unique(new_cnames)]

  # Return it
  return(new_df)
}
