% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tbl_dbi.R
\name{normality.tbl_dbi}
\alias{normality.tbl_dbi}
\title{Performs the Shapiro-Wilk test of normality}
\usage{
\method{normality}{tbl_dbi}(.data, ..., sample = 5000, in_database = FALSE, collect_size = Inf)
}
\arguments{
\item{.data}{a tbl_dbi.}

\item{...}{one or more unquoted expressions separated by commas.
You can treat variable names like they are positions.
Positive values select variables; negative values to drop variables.
If the first expression is negative, normality() will automatically start with all variables.
These arguments are automatically quoted and evaluated in a context where column names
represent column positions.
They support unquoting and splicing.}

\item{sample}{the number of samples to perform the test.}

\item{in_database}{Specifies whether to perform in-database operations. 
If TRUE, most operations are performed in the DBMS. if FALSE, 
table data is taken in R and operated in-memory. Not yet supported in_database = TRUE.}

\item{collect_size}{a integer. The number of data samples from the DBMS to R. 
Applies only if in_database = FALSE.

See vignette("EDA") for an introduction to these concepts.}
}
\value{
An object of the same class as .data.
}
\description{
The normality() performs Shapiro-Wilk test of normality 
of numerical(INTEGER, NUMBER, etc.) column of the DBMS table through tbl_dbi.
}
\details{
This function is useful when used with the \code{\link{group_by}}
function of the dplyr package. If you want to test by level of the categorical
data you are interested in, rather than the whole observation,
you can use group_tf as the group_by function.
This function is computed \code{\link{shapiro.test}} function.
}
\section{Normality test information}{

The information derived from the numerical data test is as follows.

\itemize{
\item statistic : the value of the Shapiro-Wilk statistic.
\item p_value : an approximate p-value for the test. This is said in
Roystion(1995) to be adequate for p_value < 0.1.
\item sample : the numer of samples to perform the test.
The number of observations supported by the stats::shapiro.test function is 3 to 5000.
}
}

\examples{
\donttest{
library(dplyr)

# Generate data for the example
carseats <- ISLR::Carseats
carseats[sample(seq(NROW(carseats)), 20), "Income"] <- NA
carseats[sample(seq(NROW(carseats)), 5), "Urban"] <- NA

# connect DBMS
con_sqlite <- DBI::dbConnect(RSQLite::SQLite(), ":memory:")

# copy carseats to the DBMS with a table named TB_CARSEATS
copy_to(con_sqlite, carseats, name = "TB_CARSEATS", overwrite = TRUE)

# Using pipes ---------------------------------
# Normality test of all numerical variables
con_sqlite \%>\% 
  tbl("TB_CARSEATS") \%>\% 
  normality()

# Positive values select variables, and In-memory mode and collect size is 200
con_sqlite \%>\% 
  tbl("TB_CARSEATS") \%>\% 
  normality(Sales, Price, collect_size  = 200)

# Positions values select variables
con_sqlite \%>\% 
  tbl("TB_CARSEATS") \%>\% 
  normality(1)

# Using pipes & dplyr -------------------------
# Test all numerical variables by 'ShelveLoc' and 'US',
# and extract only those with 'ShelveLoc' variable level is "Good".
con_sqlite \%>\% 
  tbl("TB_CARSEATS") \%>\% 
  group_by(ShelveLoc, US) \%>\%
  normality() \%>\%
  filter(ShelveLoc == "Good")

# extract only those with 'Urban' variable level is "Yes",
# and test 'Sales' by 'ShelveLoc' and 'US'
con_sqlite \%>\% 
  tbl("TB_CARSEATS") \%>\% 
  filter(Urban == "Yes") \%>\%
  group_by(ShelveLoc, US) \%>\%
  normality(Sales)

# Test log(Income) variables by 'ShelveLoc' and 'US',
# and extract only p.value greater than 0.01.

# SQLite extension functions for log
RSQLite::initExtension(con_sqlite)

con_sqlite \%>\% 
  tbl("TB_CARSEATS") \%>\% 
  mutate(log_income = log(Income)) \%>\%
  group_by(ShelveLoc, US) \%>\%
  normality(log_income) \%>\%
  filter(p_value > 0.01)
}

}
\seealso{
\code{\link{normality.data.frame}}, \code{\link{diagnose_numeric.tbl_dbi}}, \code{\link{describe.tbl_dbi}}, \code{\link{plot_normality.tbl_dbi}}.
}
