#' @title Show an interactive visual network representation
#'   of your drake project.
#' \lifecycle{maturing}
#' @description To save time for repeated plotting,
#' this function is divided into
#' [drake_graph_info()] and [render_drake_graph()].
#' @export
#' @seealso [render_drake_graph()], [sankey_drake_graph()],
#'   [drake_ggraph()], [text_drake_graph()]
#' @return A `visNetwork` graph.
#' @inheritParams drake_graph_info
#' @inheritParams render_drake_graph
#' @examples
#' \dontrun{
#' isolate_example("Quarantine side effects.", {
#' if (suppressWarnings(require("knitr"))) {
#' load_mtcars_example() # Get the code with drake_example("mtcars").
#' config <- drake_config(my_plan)
#' # Plot the network graph representation of the workflow.
#' if (requireNamespace("visNetwork", quietly = TRUE)) {
#' vis_drake_graph(config, width = '100%') # The width is passed to visNetwork
#' make(my_plan) # Run the project, build the targets.
#' vis_drake_graph(config) # The red nodes from before are now green.
#' # Plot a subgraph of the workflow.
#' vis_drake_graph(
#'   config,
#'   from = c("small", "reg2"),
#'   to = "summ_regression2_small"
#' )
#' }
#' }
#' })
#' }
vis_drake_graph <- function(
  config,
  file = character(0),
  selfcontained = FALSE,
  build_times = "build",
  digits = 3,
  targets_only = FALSE,
  font_size = 20,
  layout = NULL,
  main = NULL,
  direction = NULL,
  hover = FALSE,
  navigationButtons = TRUE, # nolint
  from = NULL, mode = c("out", "in", "all"),
  order = NULL,
  subset = NULL,
  ncol_legend = 1,
  full_legend = FALSE,
  make_imports = TRUE,
  from_scratch = FALSE,
  group = NULL,
  clusters = NULL,
  show_output_files = TRUE,
  collapse = TRUE,
  ...
) {
  log_msg("begin vis_drake_graph()", config = config)
  on.exit(log_msg("end vis_drake_graph()", config = config), add = TRUE)
  assert_pkg("visNetwork")
  graph_info <- drake_graph_info(
    config = config,
    from = from,
    mode = mode,
    order = order,
    subset = subset,
    build_times = build_times,
    digits = digits,
    targets_only = targets_only,
    font_size = font_size,
    make_imports = make_imports,
    from_scratch = from_scratch,
    full_legend = full_legend,
    group = group,
    clusters = clusters,
    show_output_files = show_output_files,
    hover = hover
  )
  if (is.null(main)) {
    main <- graph_info$default_title
  }
  render_drake_graph(
    graph_info,
    file = file,
    selfcontained = selfcontained,
    layout = layout,
    direction = direction,
    navigationButtons = navigationButtons, # nolint
    hover = hover,
    main = main,
    ncol_legend = ncol_legend,
    full_legend = full_legend,
    collapse = collapse,
    ...
  )
}

#' @title Render a visualization using the data frames
#'   generated by [drake_graph_info()].
#' \lifecycle{maturing}
#' @description This function is called inside
#' [vis_drake_graph()], which typical users
#' call more often.
#' @export
#' @seealso [vis_drake_graph()], [sankey_drake_graph()],
#'   [drake_ggraph()]
#' @return A `visNetwork` graph.
#'
#' @inheritParams drake_graph_info
#'
#' @param graph_info List of data frames generated by
#'   [drake_graph_info()].
#'   There should be 3 data frames: `nodes`, `edges`,
#'   and `legend_nodes`.
#'
#' @param file Name of a file to save the graph.
#'   If `NULL` or `character(0)`, no file is saved and
#'   the graph is rendered and displayed within R.
#'   If the file ends in a `.png`, `.jpg`, `.jpeg`, or `.pdf` extension,
#'   then a static image will be saved. In this case,
#'   the webshot package and PhantomJS are required:
#'   `install.packages("webshot"); webshot::install_phantomjs()`.
#'   If the file does not end in a `.png`, `.jpg`, `.jpeg`, or `.pdf`
#'   extension, an HTML file will be saved, and you can open the
#'   interactive graph using a web browser.
#'
#' @param layout Deprecated.
#'
#' @param selfcontained Logical, whether
#'   to save the `file` as a self-contained
#'   HTML file (with external resources base64 encoded) or a file with
#'   external resources placed in an adjacent directory. If `TRUE`,
#'   pandoc is required. The `selfcontained` argument only applies
#'   to HTML files. In other words, if `file` is a
#'   PNG, PDF, or JPEG file, for instance,
#'   the point is moot.
#'
#' @param direction Deprecated.
#'
#' @param navigationButtons Logical, whether to add navigation buttons with
#'   `visNetwork::visInteraction(navigationButtons = TRUE)`
#'
#' @param hover Logical, whether to show the command that generated the target
#'   when you hover over a node with the mouse. For imports, the label does not
#'   change with hovering.
#'
#' @param main Character string, title of the graph.
#'
#' @param ncol_legend Number of columns in the legend nodes.
#'   To remove the legend entirely, set `ncol_legend` to `NULL` or `0`.
#'
#' @param collapse Logical, whether to allow nodes to collapse
#'   if you double click on them.
#'   Analogous to `visNetwork::visOptions(collapse = TRUE)` or
#'   `visNetwork::visOptions(collapse = TRUE)`.
#'
#' @param ... Arguments passed to `visNetwork()`.
#'
#' @examples
#' \dontrun{
#' isolate_example("Quarantine side effects.", {
#' if (suppressWarnings(require("knitr"))) {
#' load_mtcars_example() # Get the code with drake_example("mtcars").
#' if (requireNamespace("visNetwork", quietly = TRUE)) {
#' # Instead of jumpting right to vis_drake_graph(), get the data frames
#' # of nodes, edges, and legend nodes.
#' config <- drake_config(my_plan) # Internal configuration list
#' vis_drake_graph(config) # Jump straight to the interactive graph.
#' # Get the node and edge info that vis_drake_graph() just plotted:
#' graph <- drake_graph_info(config)
#' # You can pass the data frames right to render_drake_graph()
#' # (as in vis_drake_graph()) or you can create
#' # your own custom visNewtork graph.
#' render_drake_graph(graph, width = '100%') # Width is passed to visNetwork.
#' }
#' }
#' })
#' }
render_drake_graph <- function(
  graph_info, file = character(0),
  layout = NULL,
  direction = NULL,
  hover = TRUE,
  main = graph_info$default_title,
  selfcontained = FALSE,
  navigationButtons = TRUE, # nolint
  ncol_legend = 1,
  collapse = TRUE,
  ...
) {
  assert_pkg("visNetwork")
  if (!hover) {
    graph_info$nodes$title <- NULL
  }
  if (!is.null(direction)) {
    # 2019-04-16 # nolint
    warning(
      "the `direction` argument of `vis_drake_graph()` ",
      "and `render_drake_graph()` is deprecated.",
      call. = FALSE
    )
  }
  if (!is.null(layout)) {
    # 2019-04-16 # nolint
    warning(
      "the `layout` argument of `vis_drake_graph()` ",
      "and `render_drake_graph()` is deprecated.",
      call. = FALSE
    )
  }
  out <- visNetwork::visNetwork(
    nodes = graph_info$nodes,
    edges = graph_info$edges,
    main = main,
    ...
  )
  out <- visNetwork::visNodes(out, physics = FALSE)
  if (collapse) {
    out <- visNetwork::visOptions(out, collapse = TRUE)
  }
  if (length(ncol_legend) && ncol_legend > 0) {
    out <- visNetwork::visLegend(
      graph = out,
      useGroups = FALSE,
      addNodes = graph_info$legend_nodes,
      ncol = ncol_legend
    )
  }
  sugiyama <- nrow(graph_info$edges) &&
    nrow(graph_info$nodes) > 10 &&
    abs(diff(range(graph_info$nodes$x))) > 0.1 &&
    abs(diff(range(graph_info$nodes$x))) > 0.1
  if (sugiyama) {
    out <- visNetwork::visIgraphLayout(
      graph = out,
      physics = FALSE,
      randomSeed = 2017,
      layout = "layout_with_sugiyama"
    )
  } else {
    out <- visNetwork::visHierarchicalLayout(out, direction = "LR")
  }
  out$x$nodes$x <- graph_info$nodes$x
  out$x$nodes$y <- graph_info$nodes$y
  if (navigationButtons) { # nolint
    out <- visNetwork::visInteraction(out, navigationButtons = TRUE) # nolint
  }
  if (length(file)) {
    file <- path.expand(file)
    if (is_image_filename(file)) {
      assert_pkg("webshot")
      url <- file.path(random_tempdir(), "tmp.html")
      visNetwork::visSave(graph = out, file = url, selfcontained = FALSE)
      webshot::webshot(url = url, file = file)
    } else {
      visNetwork::visSave(graph = out, file = file,
        selfcontained = selfcontained)
    }
    return(invisible())
  }
  out
}

is_image_filename <- function(x) {
  tolower(file_extn(x)) %in% c("jpg", "jpeg", "pdf", "png")
}

file_extn <- function(x) {
  x <- basename(x)
  x <- strsplit(x, split = ".", fixed = TRUE)
  x <- unlist(x)
  x <- rev(x)
  x[1]
}
