% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/generic_functions.R
\name{get_kernel}
\alias{get_kernel}
\title{Obtain the Drifting semi-Markov kernel}
\usage{
get_kernel(obj, t, u, v, l, klim = 100)
}
\arguments{
\item{obj}{An object that inherits from the S3
classes \code{dsmm},
\code{dsmm_fit_parametric}, or
\code{dsmm_fit_nonparametric},
\code{dsmm_nonparametric} or \code{dsmm_parametric}.}

\item{t}{Optional, but recommended. Positive integer specifying
the instance \eqn{t} of the visited states.}

\item{u}{Optional. Can be either of the two options below:
\itemize{
  \item Character specifying the previous state \eqn{u}, e.g. \code{u = "a"}.
  \item Positive integer, specifying a state in the state space \eqn{E}.
  For example, if \eqn{E = \{a, c, g, t\}} and \code{u = 1}, it corresponds
  to the state \eqn{a}, if \code{u = 2}, it corresponds to the state \eqn{c}.
}}

\item{v}{Optional. Can be either of the two options below:
\itemize{
  \item Character specifying the next state \eqn{v}, e.g. \code{v = "c"}.
  \item Positive integer, specifying a state in the state space \eqn{E}.
  For example, if \eqn{E = \{a, c, g, t\}} and \code{v = 3}, it corresponds
  to the state \eqn{c}, if \code{v = 4}, it corresponds to the state \eqn{t}.
}}

\item{l}{Optional. Positive integer specifying the sojourn time \eqn{l}
that is spent in the previous state \eqn{u}.}

\item{klim}{Optional. Positive integer. Used only when \code{obj} inherits
from the S3 classes \code{dsmm_parametric} or \code{dsmm_fit_parametric}.
Specifies the time horizon used to approximate the \eqn{d + 1} sojourn time
distributions if \eqn{f} is drifting, or just \eqn{1} sojourn time
distribution if \eqn{f} is \emph{not drifting}.
Default value is 100.

A larger value will result in a considerably larger
kernel, which has dimensions of \eqn{s \times s \times klim \times (n + 1)},
which will increase the memory requirements and will slow down considerably
the \code{simulate.dsmm()} method.
However, this will lead to better estimations through \code{fit_dsmm()}.
(\link{dsmm_parametric}, \link{fit_dsmm}, \link{simulate.dsmm})}
}
\value{
An array with dimensions of
\eqn{s \times s \times k_{max} \times (n + 1)}, giving the
value of the Drifting semi-Markov kernel \eqn{q_{\frac{t}{n}}(u,v,l)} for
the corresponding \eqn{(u,v,l,t)}. If any of \eqn{u,v,l} or \eqn{t} were
specified, their dimension in the array becomes 1.
}
\description{
This is a generic method that computes and returns the Drifting
semi-Markov kernel as a numerical array of dimensions
\eqn{s \times s \times k_{max} \times (n + 1)}.
}
\details{
The Drifting semi-Markov kernel is given as the probability that,
given at the instance \eqn{t} the previous state
is \eqn{u}, the next state state \eqn{v} will be reached
with a sojourn time of \eqn{l}:
\deqn{q_{\frac{t}{n}}(u,v,l) = P(J_{t}=v,X_{t}=l|J_{t-1}=u),}
where \eqn{n} is the model size, defined as the length of the embedded
Markov chain \eqn{(J_{t})_{t\in \{0,\dots,n\}}} minus the last state,
\eqn{J_t} is the visited state at the instant \eqn{t} and
\eqn{X_{t} = S_{t}-S_{t-1}} is the sojourn time of the state \eqn{J_{t-1}}.
Specifically, it is given as the sum of a linear combination:
\deqn{q_{\frac{t}{n}}(u,v,l)=
     \sum_{i = 0}^{d}A_{i}(t)q_{\frac{i}{d}}(u,v,l),}
where \eqn{A_i, i = 0, \dots, d} are \eqn{d + 1} polynomials with degree
\eqn{d} that satisfy certain conditions (see \link{dsmmR}) and
\eqn{q_{\frac{i}{d}}(u,v,l), i = 0, \dots, d}
are \eqn{d + 1} semi-Markov kernels.
Three possible model specifications are described below.
We will use the exponentials \eqn{(1), (2), (3)} to distinguish between
the Drifting semi-Markov kernel \eqn{q_\frac{t}{n}} and the
semi-Markov kernels \eqn{q_\frac{i}{d}} used in
Model 1, Model 2 and Model 3.

\strong{\emph{Model 1}}

In this case, both \eqn{p} and \eqn{f} are "drifting" between \eqn{d + 1}
fixed points of the model, hence the "drifting" in Drifting semi-Markov
Models. Therefore, the semi-Markov kernels \eqn{q_{\frac{i}{d}}^{\ (1)}} are
equal to:

\deqn{q_{\frac{i}{d}}^{\ (1)}(u,v,l) =
     {p_{\frac{i}{d}}(u,v)}{f_{\frac{i}{d}}(u,v,l)},}

where for \eqn{i = 0, \dots, d} we have \eqn{d + 1} Markov Transition
matrices \eqn{p_{\frac{i}{d}}(u,v)}, and \eqn{d + 1} sojourn time
distributions \eqn{f_{\frac{i}{d}}(u,v,l)}, where \eqn{d} is the
polynomial degree.

Thus, the Drifting semi-Markov kernel will be equal to:

\deqn{q_{\frac{t}{n}}^{\ (1)}(u,v,l) =
\sum_{i = 0}^{d} A_i(t)q_{\frac{i}{d}}^{\ (1)}(u,v,l) =
\sum_{i = 0}^{d} A_i(t)p_{\frac{i}{d}}(u,v)f_{\frac{i}{d}}(u,v,l)
}


\strong{\emph{Model 2}}

In this case, \eqn{p} is drifting and \eqn{f} \strong{is not drifting}.
Therefore, the semi-Markov kernels \eqn{q_{\frac{i}{d}}^{\ (2)}} are
equal to:
\deqn{q_{\frac{i}{d}}^{\ (2)}(u,v,l)={p_{\frac{i}{d}}(u,v)}{f(u,v,l)}.}

Thus, the Drifting semi-Markov kernel will be equal to:
\deqn{q_{\frac{t}{n}}^{\ (2)}(u,v,l) =
\sum_{i = 0}^{d} A_i(t) q_{\frac{i}{d}}^{\ (2)}(u,v,l) =
\sum_{i = 0}^{d} A_i(t) p_{\frac{i}{d}}(u,v)f(u,v,l)
}


\strong{\emph{Model 3}}

In this case, \eqn{f} is drifting and \eqn{p} \strong{is not drifting}.

Therefore, the semi-Markov kernels \eqn{q_{\frac{i}{d}}^{\ (3)}}
are now described as:
\deqn{q_{\frac{i}{d}}^{\ (3)}(u,v,l)={p(u,v)}{f_{\frac{i}{d}}(u,v,l)}.}

Thus, the Drifting semi-Markov kernel will be equal to:
\deqn{q_{\frac{t}{n}}^{\ (3)}(u,v,l) =
\sum_{i = 0}^{d} A_i(t) q_{\frac{i}{d}}^{\ (3)}(u,v,l) =
\sum_{i = 0}^{d} A_i(t) p(u,v)f_{\frac{i}{d}}(u,v,l)
}
}
\examples{
# Setup.
states <- c("Rouen", "Bucharest", "Samos", "Aigio", "Marseille")
seq <- create_sequence(states, probs = c(0.3, 0.1, 0.1, 0.3, 0.2))
obj_model_2 <- fit_dsmm(
    sequence = seq,
    states = states,
    degree = 3,
    f_is_drifting = FALSE,
    p_is_drifting = TRUE
)


# Get the kernel.
kernel_model_2 <- get_kernel(obj_model_2)
cat(paste0("If no further arguments are made, kernel has dimensions ",
           "for all u, v, l, t:\n",
           "(s, s, k_max, n + 1) = (",
           paste(dim(kernel_model_2), collapse = ", "), ")"))


# Specifying `t`.
kernel_model_2_t <- get_kernel(obj_model_2, t = 100)
# kernel_model_2_t[ , , , t = 100]
cat(paste0("If we specify t, the kernel has dimensions for ",
           "all the remaining u, v, l:\n(s, s, k_max) = (",
           paste(dim(kernel_model_2_t), collapse = ", "), ")"))


# Specifying `t` and `u`.
kernel_model_2_tu <- get_kernel(obj_model_2, t = 2, u = "Aigio")
# kernel_model_2_tu["Aigio", , , t = 2]
cat(paste0("If we specify t and u, the kernel has dimensions for ",
           "all the remaining v, l:\n(s, k_max) = (",
           paste(dim(kernel_model_2_tu), collapse = ", "), ")"))


# Specifying `t`, `u` and `v`.
kernel_model_2_tuv <- get_kernel(obj_model_2, t = 3,
                                 u = "Rouen", v = "Bucharest")
# kernel_model_2_tuv["Rouen", "Bucharest", , t = 3]
cat(paste0("If we specify t, u and v, the kernel has dimensions ",
           "for all l:\n(k_max) = (",
           paste(length(kernel_model_2_tuv), collapse = ", "), ")"))

# It is possible to ask for any valid combination of `u`, `v`, `l` and `t`.
}
\seealso{
For the objects required to calculate this kernel:
\link{fit_dsmm}, \link{parametric_dsmm}, \link{nonparametric_dsmm}.

For sequence simulation through this kernel: \link{simulate.dsmm}.

For the theoretical background of Drifting semi-Markov models: \link{dsmmR}.
}
