% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/projection.R
\name{dspline_solve}
\alias{dspline_solve}
\title{Discrete spline projection}
\usage{
dspline_solve(v, k, xd, knot_idx, basis = c("N", "B", "H"), mat)
}
\arguments{
\item{v}{Vector to be values to be projected, one value per design point.}

\item{k}{Order for the discrete spline space. Must be >= 0.}

\item{xd}{Design points. Must be sorted in increasing order, and have length
at least \code{k+1}.}

\item{knot_idx}{Vector of indices, a subset of \code{(k+1):(n-1)} where \code{n = length(xd)}, that indicates which design points should be used as knot
points for the discrete spline space. Must be sorted in increasing order.}

\item{basis}{String, one of \code{"N"}, \code{"B"}, or \code{"H"}, indicating which basis
representation is to be used for the least squares computation. The default
is \code{"N"}, the discrete B-spline basis. See details for more information.}

\item{mat}{Matrix to use for the least squares computation. If missing, the
default, then the matrix will be formed according to the \code{basis} argument.
See details for more information.}
}
\value{
List with components \code{sol}: the least squares solution; \code{fit}: the
least squares fit; and \code{mat}: the basis matrix used for the least squares
problem (only present if the input argument \code{mat} is missing).
}
\description{
Projects a sequence of values onto the space of discrete splines a given
order, with respect to given design points, and given knot points.
}
\details{
This function minimizes the least squares criterion
\deqn{
  \|v - M \beta\|_2^2
  }
over coefficient vectors \eqn{\beta}; that is, it computes
\deqn{
  \hat\beta = (M^T M)^{-1} M^T v
  }
for a vector \eqn{v} and basis matrix \eqn{M}. The basis matrix \eqn{M} is
specified via the \code{basis} argument, which allows for three options. The
default is \code{"N"}, in which case the discrete B-spline basis matrix from
\code{\link[=n_mat]{n_mat()}} is used, with the \code{knot_idx} argument set accordingly. This is
generally the \strong{most stable and efficient} option: it leads to a banded,
well-conditioned linear system.  Bandedness means that the least squares
projection can be computed in \eqn{O(nk^2)} operations. See Section 8.4 of
Tibshirani (2020) for numerical experiments investigating conditioning.

The option \code{"H"} means that the falling factorial basis matrix from \code{\link[=h_mat]{h_mat()}}
is used, with the \code{col_idx} argument set appropriately. This option should
be \strong{avoided in general} since it leads to a linear system that is neither
sparse nor well-conditioned.

The option \code{"B"} means that the extended discrete derivative matrix from
\code{\link[=b_mat]{b_mat()}}, with \code{tf_weighting = TRUE}, is used to compute the least squares
solution from projecting onto the falling factorial basis. The fact this is
possible stems from a special inverse relationship between the discrete
derivative matrix and the falling factorial basis matrix. While this option
leads to a banded linear system, this system tends to have worse
conditioning than that using the discrete B-spline representation. However,
it is essentially always preferable to the \code{"H"} option, and it produces
the same solution (coefficients in the falling factorial basis expansion).

\strong{Note 1:} the basis matrix to be used in the least squares problem can be
passed in directly via the \code{mat} argument (which saves on the cost of
computing it in the first place). Even when \code{mat} not missing, the \code{basis}
argument must still be used to specify which type of basis matrix is being
passed in, as the downstream computations differ depending on the type.

\strong{Note 2:} when \code{mat} is not missing and \code{basis = "B"}, the matrix being
passed in must be the \strong{entire} extended discrete derivative matrix, as
returned by \code{b_mat()} with \code{row_idx = NULL} (the default), and not some
row-subsetted version. This is because both the rows corresponding to the
knots in the discrete spline space and the complementary set of roles play
a role in computing the solution. See Section 8.1 of Tibshirani (2020).
}
\examples{
xd = 1:100 / 100
knot_idx = 1:9 * 10
y = sin(2 * pi * xd) + rnorm(100, 0, 0.2)
yhat = res = dspline_solve(y, 2, xd, knot_idx)$fit
plot(xd, y, pch = 16, col = "gray60")
points(xd, yhat, col = "firebrick")
abline(v = xd[knot_idx], lty = 2)
}
\references{
Tibshirani (2020), "Divided differences, falling factorials, and
discrete splines: Another look at trend filtering and related problems",
Sections 8.1 and 8.4.
}
\seealso{
\code{\link[=dspline_interp]{dspline_interp()}} for interpolation within the "canonical" space of
discrete splines.
}
