% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dtm_main.r
\name{run.dtm}
\alias{run.dtm}
\title{Run Dynamic TOPMODEL using the catchment areal groupings (response units) for a discretisation.}
\usage{
run.dtm(groups, weights, rain, routing, qobs = NULL, qt0 = 1e-04,
  pe = NULL, dt = NULL, ntt = 1, ichan = 1, i.out = ichan[1],
  vchan = 1000, vof = 100, dqds = NULL, sim.start = NA, sim.end = NA,
  disp.par = disp.par())
}
\arguments{
\item{groups}{Data frame of ngroup areal group definitions along with their hydrological parameters.}

\item{weights}{The flux distribution (weighting) ngroup*ngroup matrix. Usually generated by the discretise method.}

\item{rain}{A time series of rainfall data in m/hr. One column per gauge if multiple gauges used.  Use aggregate_obs to apply a different time interval to this and the other observation data.}

\item{routing}{Channel routing table comprises a two-column data.frame or matrix. Its first column should be average flow distance to the outlet in m, the second the proportions of the catchment channel network within each distance category. Can be generated by make.routing.table}

\item{qobs}{Optional time series of observation data}

\item{qt0}{Initial specific discharge (m/hr)}

\item{pe}{Time series of potential evapotranspiration, at the same time step as rainfall data}

\item{dt}{Time step (hours). Defaults to the interval used by the rainfall data}

\item{ntt}{Number of inner time steps used in subsurface routing algorithm.}

\item{ichan}{Integer index of the "channel" group. Defaults to 1}

\item{i.out}{For multi-channel systems, the index of the outlet reach}

\item{vchan}{Default channel routing velocity (m/hr)}

\item{vof}{Default overland flow routing velocity (m/hr).}

\item{dqds}{Function to supplies a custom flux-storage relationship as the kinematic wave celerity. If not supplied then exponential relationship used}

\item{sim.start}{Optional start time for simulation in any format that can be coerced into a POSIXct instance. Defaults to start of rainfall data}

\item{sim.end}{Optional end time of simulation in any format that can be coerced into a POSIXct instance. Defaults to end of rainfall datA}

\item{disp.par}{List of graphical routing parameters. A set of defaults are retrieved by calling disp.par()}
}
\value{
A list containing run output and input data. These include

qsim: time series of specific discharges (m/hr) at the specified time interval. can be converted to absolutre discharges by multiying by catch.area

catch.area: the catchment area in m^2, calculated from the areas in the groups table

data.in: the parameters supplied as input to the call to run.dtm

sim.start: start of simulation

sim.end: end time of simulation

fluxes: a list comprising, for each response unit the specific base flows qbf, specific upslope inputs qin, drainage fluxes quz, and any overland flow qof, all in m/hr

storages: a list comprising, for each response unit, root zone and unsaturated storage and total storage deficit (all m)
}
\description{
Run Dynamic TOPMODEL using the catchment areal groupings (response units) for a discretisation.
}
\details{
The grouping (HRU) table may be generated by the discretise method and includes each indexed channel as separate group. See Metcalfe et al. (2015) for descriptions of the parameters maintained in this table.

Evapotranspiration input can be generated using the approx.pe.ts method

If disp.par$graphics.show = T then the output will be displayed graphically whilst the simulation is in progress. Otherwise simulated specific discharges
}
\examples{
\dontrun{
require(dynatopmodel)
data(brompton)
# September 2012 storm event
# The response is sensitive to the size of the channels, but many are small.
# Set an overall width of 2m.
chans <- build_chans(dem=brompton$dem, drn=brompton$drn, chan.width=2)

# discretisation by reverse distance from nearest channel. The raster brompton$flowdists
# gives the D8 flow pathway distance for every area in the catchment
layers <- addLayer(brompton$dem, 2000-brompton$flowdists)
disc  <- discretise(layers, cuts=c(flowdists=5), chans=chans, area.thresh=3/100)

Network routing table
routing <- build_routing_table(brompton$dem, chans)

# Here we apply the same parameter values to all groups. Suggest applying smaller m and td values to
# the closest areas to simulate a fast response due to the artificial drainage.
# It would also be possible to supply a custom transmissivity profile that has
# a discontinuity at the depth of the drains
groups <- disc$groups
groups$m <- 0.011
groups$td <-  42
# a very high transmissivity prevents saturation flow as there appears be little
groups$ln_t0 <- 18
groups$srz_max <- 0.1
# initial root zone storage
groups$srz0 <- 0.87
# quite slow channel flow, which might be expected with the shallow and reedy
# reaches in this catchment
groups$vchan <- 750

# Observations at a 15 minute time step
dt <- 0.25
obs <- list(rain=brompton$rain,
pe=brompton$pe,
qobs=brompton$qobs)
obs <- aggregate_obs(obs, dt=dt)

# parameters for graphics output
par <- disp.par(int.time=24)

# Note max.q in mm/hr
par$max.q <- 1000*max(obs$qobs, na.rm=TRUE)
sim.start <- "2012-09-23"
sim.end <- "2012-10-01"

# Ensure output goes to a new window
options("device"="X11")
# take initial discharge from the observations
qt0 <- as.numeric(obs$qobs[sim.start][1])

# Run the model across the September 2012 storm event using 2 inner time steps
and a 15 minute interval
 storm.run <- run.dtm(groups=groups,
   weights=disc$weights,
   rain=obs$rain,
   pe=obs$pe,
   qobs=obs$qobs,
   qt0=qt0,
   sim.start=sim.start,
   sim.end=sim.end,
   routing=routing,
   disp.par=par,
   ntt=2)
 # show run statistics
 cat("NSE=", NSE(storm.run$qsim, storm.run$qobs))
 cat("Time at peak =", format(time_at_peak(storm.run$qsim)))
}
}
\author{
Peter Metcalfe
}
\references{
Beven, K., & Freer, J. (2001). A dynamic topmodel. Hydrological processes, 15(10), 1993-2011.

Metcalfe, P., Beven, K., & Freer, J. (2015). Dynamic TOPMODEL: A new implementation in R and its sensitivity to time and space steps. Environmental Modelling & Software, 72, 155-172.
}

