% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/load.R
\name{load_raster}
\alias{load_raster}
\title{Load eBird Status and Trends raster data cubes}
\usage{
load_raster(
  path,
  product = c("abundance", "count", "occurrence", "percent-population"),
  period = c("weekly", "seasonal", "full-year"),
  metric = NULL,
  resolution = c("hr", "mr", "lr")
)
}
\arguments{
\item{path}{character; directory that the Status and Trends data for a given
species was downloaded to. This path is returned by \code{ebirdst_download()}
or \code{get_species_path()}.}

\item{product}{character; Status and Trends product to load: occurrence,
count, relative abundance, or percent of population. See Details for a
detailed explanation of each of these products.}

\item{period}{character; temporal period of the estimation. The Status and
Trends models make predictions for each week of the year; however, as a
convenience, data are also provided summarized at the seasonal or annual
("full-year") level.}

\item{metric}{character; by default, the weekly products provide estimates of
the median value (\code{metric = "median"}) and the summarized products are the
cell-wise mean across the weeks within the season (\code{metric = "mean"}).
However, additional variants exist for some of the products. For the weekly
relative abundance, confidence intervals are provided: specify \code{metric = "lower"} to get the 10th quantile or \code{metric = "upper"} to get the 90th
quantile. For the seasonal and annual products, the cell-wise maximum
values across weeks can be obtained with \code{metric = "max"}.}

\item{resolution}{character; the resolution of the raster data to load. The
default is to load the native ~3 km resolution (\code{"hr"}); however, for some
applications 9 km (\code{"mr"}) or 27 km (\code{"lr"}) data may be suitable.}
}
\value{
For the weekly cubes, a \link[terra:SpatRaster-class]{SpatRaster} with 52
layers for the given product, labeled by week. Seasonal cubes will have up
to four layers labeled according to the seasons. The full-year products
will have a single layer.
}
\description{
Each of the eBird Status and Trends raster products is packaged as a GeoTIFF
file representing predictions on a regular grid. The core products are
occurrence, count, relative abundance, and percent of population. This
function loads one of the available data products into R as a
\link[terra:SpatRaster-class]{SpatRaster} object.
}
\details{
The core Status and Trends data products provide weekly estimates
across a regular spatial grid. They are packaged as rasters with 52 layers,
each corresponding to estimates for a week of the year, and we refer to
them as "cubes" (e.g. the "relative abundance cube"). All estimates are the
median expected value for a standard 1km, 1 hour eBird Traveling Count by
an expert eBird observer at the optimal time of day and for optimal weather
conditions to observe the given species. These products are:
\itemize{
\item \code{occurrence}: the expected probability (0-1) of occurrence a species.
\item \code{count}: the expected count of a species, conditional on its occurrence at
the given location.
\item \code{abundance}: the expected relative abundance of a species, computed as the product of
the probability of occurrence and the count conditional on occurrence.
\item \code{percent-population}: the percent of the total relative abundance within
each cell. This is a derived product calculated by dividing each cell value
in the relative abundance raster with the total abundance summed across all
cells.
}

In addition to these weekly data cubes, this function provides access to data
summarized over different periods. Seasonal cubes are produced by taking the
cell-wise mean or max across the weeks within each season. The boundary dates
for each season are species specific and are available in \code{ebirdst_runs}, and
if a season failed review no associated layer will be included in the cube.
In addition, full-year summaries provide the mean or max across all weeks of
the year that fall within a season that passed review. Note that this is not
necessarily all 52 weeks of the year. For example, if the estimates for the
non-breeding season failed expert review for a given species, the full-year
summary for that species will not include the weeks that would fall within
the non-breeding season.
}
\examples{
\dontrun{
# download example data
path <- ebirdst_download("example_data")
# or get the path if you already have the data downloaded
path <- get_species_path("example_data")

# weekly relative abundance
# note that only low resolution (lr) data are available for the example data
abd_weekly <- load_raster(path, "abundance", resolution = "lr")

# the weeks for each layer are stored in the layer names
names(abd_weekly)
# and can be converted to Date objects with
parse_raster_dates(abd_weekly)

# max seasonal abundance
abd_seasonal <- load_raster(path, "abundance",
                            period = "seasonal", metric = "max",
                            resolution = "lr")
# available seasons in stack
names(abd_seasonal)
# subset to just breeding season abundance
abd_seasonal[["breeding"]]
}
}
