% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/trajectoryComparison.R
\name{trajectoryComparison}
\alias{trajectoryComparison}
\alias{segmentDistances}
\alias{trajectoryDistances}
\alias{trajectoryConvergence}
\alias{trajectoryShifts}
\title{Trajectory comparison}
\usage{
segmentDistances(x, distance.type = "directed-segment", add = TRUE)

trajectoryDistances(
  x,
  distance.type = "DSPD",
  symmetrization = "mean",
  add = TRUE
)

trajectoryConvergence(x, symmetric = FALSE, add = TRUE)

trajectoryShifts(x, add = TRUE)
}
\arguments{
\item{x}{An object of class \code{\link{trajectories}}.}

\item{distance.type}{The type of distance index to be calculated (see section Details).}

\item{add}{Flag to indicate that constant values should be added (local transformation) to correct triplets of distance values that do not fulfill the triangle inequality.}

\item{symmetrization}{Function used to obtain a symmetric distance, so that DSPD(T1,T2) = DSPD(T2,T1) (e.g., \code{mean}, \code{max} or \code{min}). If \code{symmetrization = NULL} then the symmetrization is not conducted and the output dissimilarity matrix is not symmetric.}

\item{symmetric}{A logical flag to indicate a symmetric convergence comparison of trajectories.}
}
\value{
Function \code{trajectoryDistances} returns an object of class \code{\link{dist}} containing the distances between trajectories (if \code{symmetrization = NULL} then the object returned is of class \code{matrix}).

Function \code{segmentDistances} list with the following elements:
\itemize{
\item{\code{Dseg}: Distance matrix between segments.}
\item{\code{Dini}: Distance matrix between initial points of segments.}
\item{\code{Dfin}: Distance matrix between final points of segments.}
\item{\code{Dinifin}: Distance matrix between initial points of one segment and the final point of the other.}
\item{\code{Dfinini}: Distance matrix between final points of one segment and the initial point of the other.}
}
Function \code{trajectoryConvergence} returns a list with two elements:
\itemize{
\item{\code{tau}: A matrix with the statistic (Mann-Kendall's tau) of the convergence/divergence test between trajectories. If \code{symmetric=TRUE} then the matrix is square. Otherwise the statistic of the test of the row trajectory approaching the column trajectory.}
\item{\code{p.value}: A matrix with the p-value of the convergence/divergence test between trajectories. If \code{symmetric=TRUE} then the matrix is square. Otherwise the p-value indicates the test of the row trajectory approaching the column trajectory.}
}
Function \code{trajectoryShifts} returns an object of class \code{\link{data.frame}} describing trajectory shifts (i.e. advances and delays). The columns of the \code{\link{data.frame}} are:
\itemize{
\item{\code{reference}: the site (trajectory) that is taken as reference for shift evaluation.}
\item{\code{site}: the target site (trajectory) for which shifts have been computed.}
\item{\code{survey}: the target trajectory survey for which shift is computed.}
\item{\code{time}: the time corresponding to target trajectory survey.}
\item{\code{timeRef}: the time associated to the projected ecological state onto the reference trajectory.}
\item{\code{shift}: the time difference between the time of the target survey and the time of projected ecological state onto the reference trajectory. Positive values mean faster
trajectories and negative values mean slower trajectories.}
}
}
\description{
Functions to compare pairs of trajectories or trajectory segments.
\itemize{
\item{Function \code{segmentDistances} calculates the distance between pairs of trajectory segments.}
\item{Function \code{trajectoryDistances} calculates the distance between pairs of trajectories.}
\item{Function \code{trajectoryConvergence} performs the Mann-Kendall trend test on the distances between trajectories (symmetric test) or the distance between points of one trajectory to the other.}
\item{Function \code{trajectoryShifts} calculates trajectory shifts (i.e. advances and delays) between trajectories assumed to follow a similar path but with different speeds or time lags.}
}
}
\details{
Ecological Trajectory Analysis (ETA) is a framework to analyze dynamics of ecosystems described as trajectories in a chosen space of multivariate resemblance (De \enc{Cáceres}{Caceres} et al. 2019).
ETA takes trajectories as objects to be analyzed and compared geometrically.

The input distance matrix \code{d} should ideally be metric. That is, all subsets of distance triplets should fulfill the triangle inequality (see utility function \code{\link{is.metric}}).
All ETA functions that require metricity include a parameter '\code{add}', which by default is TRUE, meaning that whenever the triangle inequality is broken the minimum constant required to fulfill it is added to the three distances.
If such local (an hence, inconsistent across triplets) corrections are not desired, users should find another way modify \code{d} to achieve metricity, such as PCoA, metric MDS or non-metric MDS (see vignette 'Introduction to Ecological Trajectory Analysis').
If parameter '\code{add}' is set to FALSE and problems of triangle inequality exist, ETA functions may provide missing values in some cases where they should not.

The resemblance between trajectories is done by adapting concepts and procedures used for the analysis of trajectories in space (i.e. movement data) (Besse et al. 2016).

Parameter \code{distance.type} is the type of distance index to be calculated which for function \code{segmentDistances} has the following options (Besse et al. 2016, De \enc{Cáceres}{Caceres} et al. 2019:
\itemize{
\item{\code{Hausdorff}: Hausdorff distance between two segments.}
\item{\code{directed-segment}: Directed segment distance (default).}
\item{\code{PPA}: Perpendicular-parallel-angle distance.}
}
In the case of function \code{trajectoryDistances} the following values are possible (De \enc{Cáceres}{Caceres} et al. 2019):
\itemize{
\item{\code{Hausdorff}: Hausdorff distance between two trajectories.}
\item{\code{SPD}: Segment Path Distance.}
\item{\code{DSPD}: Directed Segment Path Distance (default).}
\item{\code{TSPD}: Time-Sensitive Path Distance (experimental).}
}

Function \code{trajectoryShifts} is intended to be used to compare trajectories that are assumed to follow a similar pathway. The function
evaluates shifts (advances or delays) due to different trajectory speeds or the existence of time lags between them. This is done using calls to \code{\link{trajectoryProjection}}.
Whenever the projection of a given target state on the reference trajectory does not exist the shift cannot be evaluated (missing values are returned).
}
\examples{
#Description of sites and surveys
sites <- c("1","1","1","1","2","2","2","2","3","3","3","3")
surveys <- c(1,2,3,4,1,2,3,4,1,2,3,4)
  
#Raw data table
xy<-matrix(0, nrow=12, ncol=2)
xy[2,2]<-1
xy[3,2]<-2
xy[4,2]<-3
xy[5:6,2] <- xy[1:2,2]
xy[7,2]<-1.5
xy[8,2]<-2.0
xy[5:6,1] <- 0.25
xy[7,1]<-0.5
xy[8,1]<-1.0
xy[9:10,1] <- xy[5:6,1]+0.25
xy[11,1] <- 1.0
xy[12,1] <-1.5
xy[9:10,2] <- xy[5:6,2]
xy[11:12,2]<-c(1.25,1.0)
  
#Draw trajectories
trajectoryPlot(xy, sites, surveys,  
               traj.colors = c("black","red", "blue"), lwd = 2)

#Distance matrix
d <- dist(xy)
d
  
#Trajectory data
x <- defineTrajectories(d, sites, surveys)

#Distances between trajectory segments
segmentDistances(x, distance.type = "Hausdorff")
segmentDistances(x, distance.type = "directed-segment")

#Distances between trajectories
trajectoryDistances(x, distance.type = "Hausdorff")
trajectoryDistances(x, distance.type = "DSPD")
  
#Trajectory convergence/divergence
trajectoryConvergence(x)

#### Example of trajectory shifts
#Description of sites and surveys
sites2 <- c("1","1","1","1","2","2","2","2","3","3","3","3")
times2 <- c(1,2,3,4,1,2,3,4,1,2,3,4)
  
#Raw data table
xy2<-matrix(0, nrow=12, ncol=2)
xy2[2,2]<-1
xy2[3,2]<-2
xy2[4,2]<-3
xy2[5:8,1] <- 0.25
xy2[5:8,2] <- xy2[1:4,2] + 0.5 # States are all shifted with respect to site "1"
xy2[9:12,1] <- 0.5
xy2[9:12,2] <- xy2[1:4,2]*1.25  # 1.25 times faster than site "1"
  
#Draw trajectories
trajectoryPlot(xy2, sites2,  
               traj.colors = c("black","red", "blue"), lwd = 2)

#Trajectory data
x2 <- defineTrajectories(dist(xy2), sites = sites2, times = times2)

#Check that the third trajectory is faster
trajectorySpeeds(x2)

#Trajectory shifts
trajectoryShifts(x2)
}
\references{
Besse, P., Guillouet, B., Loubes, J.-M. & François, R. (2016). Review and perspective for distance based trajectory clustering. IEEE Trans. Intell. Transp. Syst., 17, 3306–3317.

De \enc{Cáceres}{Caceres} M, Coll L, Legendre P, Allen RB, Wiser SK, Fortin MJ, Condit R & Hubbell S. (2019).
Trajectory analysis in community ecology. Ecological Monographs 89, e01350.
}
\seealso{
\code{\link{trajectoryMetrics}}, \code{\link{trajectoryPlot}}, \code{\link{transformTrajectories}}, \code{\link{trajectoryProjection}}
}
\author{
Miquel De \enc{Cáceres}{Caceres}, CREAF

Nicolas Djeghri, UBO
}
