% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stability_validation.R
\name{stability_validation}
\alias{stability_validation}
\title{Stability validation of ego-Temporal Exponential Random Graph Model (ego-TERGM) fit.}
\usage{
stability_validation(ego_tergm_fit = NULL, splitting_probability = 0.5,
  seed = 12345, R = 10, forking = FALSE, ncpus = 1, steps = 50,
  tol = 1e-06)
}
\arguments{
\item{ego_tergm_fit}{The output of a previously fit ego-TERGM fit using the ego_tergm function.  This is the model that stability validation will be performed on.}

\item{splitting_probability}{A value from 0 to 1 that determines the probability that any given network is assigned to the comparison group.}

\item{seed}{The seed set to replicate analysis for pseudorandom number generator.}

\item{R}{The number of bootstrap replications that should be used for the estimation of a bootstrapped MPLE estimated TERGM for model initialization.  Defaults to 10.}

\item{forking}{If parallelization via forking should be used (TRUE) or if no parallel processing should be used (FALSE).  Currently, sockets are not supported.}

\item{ncpus}{The number of CPUs that should should be used for estimation, defaults to 1.}

\item{steps}{The number of default EM steps that should be taken, defaults to 50.}

\item{tol}{The difference in parameter estimates between EM iterations to determine if the algorithm has converged.  Defaults to 1e-6.}
}
\value{
Returns comparison_table (a matrix of cross-tabulation results to compare common cluster assignments or if incompatible a table of relative proportions sorted by value to allow for comparisons under set incompatibility and label switching), networks_sampled (which networks were included in the new validation sample),
        comparison_lambda (the matrix of role assignments for validation networks), comparison_group.theta (centroids for validation networks),
        comparison_EE.BIC (Salter-Townshend and Murphy BIC that doesn't penalize for longitudinal networks for validation networks),
        comparison_TS.BIC (BIC that penalizes for longitudinal networks for validation networks), comparison_role_assignments (role assignments for validation networks),
        and comparison_ego_nets (validation ego-networks).  Note that labels may switch.
}
\description{
This function examines the stability of ego-TERGM results to the pooling window.  It takes some proportion of the window's network and compares the result of a model fit on these time periods to the original fit.
}
\examples{
\donttest{
# Code from xergm.common and their preparation of the Knecht network
library(xergm.common)
set.seed(1)

data("knecht")

for (i in 1:length(friendship)) {
 rownames(friendship[[i]]) <- paste("Student.", 1:nrow(friendship[[i]]), sep="")
 colnames(friendship[[i]]) <- paste("Student.", 1:nrow(friendship[[i]]), sep="")
}
rownames(primary) <- rownames(friendship[[1]])
colnames(primary) <- colnames(friendship[[1]])
sex <- demographics$sex
names(sex) <- rownames(friendship[[1]])
# step 2: imputation of NAs and removal of absent nodes:
friendship <- xergm.common::handleMissings(friendship, na = 10, method = "remove")
friendship <- xergm.common::handleMissings(friendship, na = NA, method = "fillmode")
# step 3: add nodal covariates to the networks
for (i in 1:length(friendship)) {
  s <- xergm.common::adjust(sex, friendship[[i]])
  friendship[[i]] <- network::network(friendship[[i]])
  friendship[[i]] <- network::set.vertex.attribute(friendship[[i]], "sex", s)
  idegsqrt <- sqrt(sna::degree(friendship[[i]], cmode = "indegree"))
  friendship[[i]] <- network::set.vertex.attribute(friendship[[i]],
                                                   "idegsqrt", idegsqrt)
  odegsqrt <- sqrt(sna::degree(friendship[[i]], cmode = "outdegree"))
  friendship[[i]] <- network::set.vertex.attribute(friendship[[i]],
                                                   "odegsqrt", odegsqrt)
}
sapply(friendship, network::network.size)
net <- friendship
rm(list=setdiff(ls(), "net"))

ego_tergm_fit <- ego_tergm(net = net,
                          form = c("edges", "mutual", "triangle",
                                   "nodeicov('idegsqrt')", "nodeocov('odegsqrt')",
                                   "nodematch('sex')"),
                          core_size = 1,
                          min_size = 5,
                          roles = 3,
                          add_drop = TRUE,
                          directed = TRUE,
                          edge_covariates = FALSE,
                          seed = 12345,
                          R = 10,
                          forking = FALSE,
                          ncpus = 1,
                          steps = 50,
                          tol = 1e-06)

stability_check <- stability_validation(ego_tergm_fit = ego_tergm_fit, seed = 614)
print(stability_check$comparison_table)
}
}
\references{
{
 Campbell, Benjamin W. (2018):
 Inferring Latent Roles in Longitudinal Networks.
 \emph{Political Analysis} 26(3): 292-311.  \url{https://doi.org/10.1017/pan.2018.20}
}
}
\keyword{ego-TERGM}
\keyword{validation}
