% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ebp.R
\name{ebp}
\alias{ebp}
\title{Empirical Best Prediction for disaggregated indicators}
\usage{
ebp(fixed, pop_data, pop_domains, smp_data, smp_domains, L, pov_line = NULL,
  transformation = "box.cox", interval = c(-1, 2), MSE = FALSE,
  B = NULL, parallel_mode = "local", cpus = 1, custom_indicator = NULL,
  na.rm = FALSE)
}
\arguments{
\item{fixed}{a two-sided linear formula object describing the
fixed-effects part of the nested error linear regression model with the
dependent variable on the left of a ~ operator and the explanatory
variables on the right, separated by + operators. The argument corresponds
to the argument \code{fixed} in function \code{\link[nlme]{lme}}.}

\item{pop_data}{a data frame that needs to comprise the variables
named on the right of the ~ operator in \code{fixed}, i.e. the explanatory
variables, and \code{pop_domains}.}

\item{pop_domains}{a character string containing the name of a variable that
indicates domains in the population data. The variable can be numeric or
a factor but needs to be of the same class as the variable named in
\code{smp_domains}.}

\item{smp_data}{a data frame that needs to comprise all variables named in
\code{fixed} and \code{smp_domains}.}

\item{smp_domains}{a character string containing the name of a variable
that indicates domains in the sample data. The variable can be numeric or a
factor but needs to be of the same class as the variable named in
\code{pop_domains}.}

\item{L}{a number determining the number of Monte-Carlo simulations.}

\item{pov_line}{a number defining a poverty line. A poverty line is
needed for calculation e.g. of head count ratios and poverty gaps. The 
argument defaults to \code{NULL}. In this case the poverty line is set to 60\% 
of the median of the variable that is selected as dependent variable similary 
to the At-risk-of-poverty rate used in the EU (see also \cite{Social Protection 
Committee 2001}). However, any desired poverty line can be chosen.}

\item{transformation}{a character string. Three different transformation
types for the dependent variable can be chosen (i) no transformation ("no");
(ii) log transformation ("log"); (iii) Box-Cox transformation ("box.cox").
Defaults to \code{NULL}.}

\item{interval}{a numeric vector containing a lower and upper limit
determining an interval for the estimation of the optimal parameter. Defaults
to c(-1,2).}

\item{MSE}{if TRUE, MSE estimates using a parametric bootstrap approach
are calculated (see also \cite{Gonzalez-Manteiga et al. (2008)}). Defaults
to \code{FALSE}.}

\item{B}{a number determining the number of bootstrap populations in the
parametric bootstrap approach (see also \cite{Gonzalez-Manteiga et al. (2008)})
used in the MSE estimation. Defaults to \code{NULL}.}

\item{parallel_mode}{modus of parallelisation, defaults to local. For details see \code{\link[parallelMap]{parallelStart}}}

\item{cpus}{number determining the kernels that are used for the 
parallelization. Defaults to 1. For details see \code{\link[parallelMap]{parallelStart}}}

\item{custom_indicator}{a list of functions containing the indicators to be
calculated additionaly, such functions must and must only depend on the
population income vector \code{y} and the poverty line \code{pov_line}. 
Defaults to \code{NULL}.}

\item{na.rm}{if TRUE, observations with \code{NA} values are deleted from the 
population and sample data. For the EBP procedure complete observations  
are required. Defaults to \code{FALSE}.}
}
\value{
An object of class "emdi" that provides estimators for regional
disaggregated indicators and optionally corresponding MSE estimates. Generic
functions such as \code{\link{estimators}}, \code{\link{print}}, 
\code{\link{plot}}, and \code{\link{summary}} have methods that can be used
to obtain further information. See \code{\link{emdiObject}} for descriptions
of components of objects of class "emdi".
}
\description{
Function \code{ebp} estimates indicators using the Empirical Best Prediction
approach by \cite{Molina and Rao (2010)}. Point predictions of indicators are
obtained by Monte-Carlo approximations. Additionally, mean squared error (MSE)
estimation can be conducted by using a parametric bootstrap approach (see
also \cite{Gonzalez-Manteiga et al. (2008)}). The unit level model of
\cite{Battese, Harter and Fuller (1988)} is fitted by REML method and one of
three different transformation types for the dependent variable can be chosen.
}
\details{
For Monte-Carlo approximations and in the parametric bootstrap
approach random number generation is used. In order to specify seeds use
\code{\link{set.seed}}. \cr \cr
The set of predefined indicators includes the mean, median, four further quantiles
(10\%, 25\%, 75\% and 90\%), head count ratio, poverty gap, Gini coefficient
and the quintile share ratio.
}
\examples{
# Loading data - population and sample data
data("eusilcA_pop")
data("eusilcA_pop")

# Example with default setting but na.rm=TRUE
set.seed(100); emdi_model <- ebp( fixed = eqIncome ~ gender + eqsize + cash + 
self_empl + unempl_ben + age_ben + surv_ben + sick_ben + dis_ben + rent + 
fam_allow + house_allow + cap_inv + tax_adj, pop_data = eusilcA_pop,
pop_domains = "district", smp_data = eusilcA_smp, smp_domains = "district",
L= 1, na.rm = TRUE)


# Example with two additional indicators
set.seed(100); emdi_model <- ebp( fixed = eqIncome ~ gender + eqsize + cash + 
self_empl + unempl_ben + age_ben + surv_ben + sick_ben + dis_ben + rent + 
fam_allow + house_allow + cap_inv + tax_adj, pop_data = eusilcA_pop,
pop_domains = "district", smp_data = eusilcA_smp, smp_domains = "district",
pov_line = 10722.66, transformation = "box.cox", L= 1, MSE = TRUE, B = 1,
custom_indicator = list( my_max = function(y, pov_line){max(y)},
my_min = function(y, pov_line){min(y)}), na.rm = TRUE, cpus = 1)
}
\references{
Battese, G.E., Harter, R.M. and Fuller, W.A. (1988). An Error-Components
Model for Predictions of County Crop Areas Using Survey and Satellite Data.
Journal of the American Statistical Association, Vol.83, No. 401, 28-36. \cr \cr
Gonzalez-Manteiga, W. et al. (2008). Bootstrap mean squared error of
a small-area EBLUP. Journal of Statistical Computation and Simulation,
78:5, 443-462. \cr \cr
Molina, I. and Rao, J.N.K. (2010). Small area estimation of poverty
indicators. The Canadian Journal of Statistics, Vol. 38, No.3, 369-385. \cr \cr
Social Protection Committee (2001). Report on indicators in the field of
poverty and social exclusions, Technical Report, European Union.
}
\seealso{
\code{\link{emdiObject}}, \code{\link[nlme]{lme}},
\code{\link{estimators.emdi}}, \code{\link{print.emdi}}, \code{\link{plot.emdi}},
\code{\link{summary.emdi}}
}

