\name{Independence E-test}
\alias{indep.e}
\alias{indep.etest}
\title{ Energy Statistic Tests of Independence}
\description{
 Computes the multivariate nonparametric E-statistic and test of independence.}
\usage{
indep.e(x, y)
indep.etest(x, y, R=199)
}
\arguments{
  \item{x}{ matrix: first sample, observations in rows}
  \item{y}{ matrix: second sample, observations in rows}
  \item{R}{ number of replicates}
}
\details{
 Computes the coefficient \eqn{\mathcal I}{I_n} and performs a nonparametric 
 \eqn{\mathcal E}{E}-test of independence. The test decision is obtained via 
 bootstrap, with \code{R} replicates.
 The sample sizes (number of rows) of the two samples must agree, and
 samples must not contain missing values. The statistic 
 \eqn{\mathcal E = n \mathcal I^2}{E = I^2} is a ratio of V-statistics based 
 on interpoint distances \eqn{\|x_{i}-y_{j}\|}{||x_{i}-y_{j}||}. 
 See the reference below for details. 
 
}
\value{
 The value of the test statistic is returned by \code{indep.e}.
 The function \code{indep.etest} returns a list with class 
 \code{htest} containing
 \item{method}{description of test}
 \item{statistic}{observed value of the coefficient \eqn{\mathcal I}{I}}
 \item{p.value}{approximate p-value of the test}
 \item{data.name}{description of data}
}
\references{
 Bakirov, N.K., Rizzo, M.L., and Szekely, G.J. (2006), A Multivariate
 Nonparametric Test of Independence, \emph{Journal of Multivariate Analysis} 
 93/1, 58-80, http://dx.doi.org/10.1016/j.jmva.2005.10.005 
 }
\author{ Maria L. Rizzo \email{mrizzo @ bgnet.bgsu.edu} and
Gabor J. Szekely \email{gabors @ bgnet.bgsu.edu}
}
\examples{

 ## independent univariate data
 x <- sin(runif(30, 0, 2*pi) * 2)
 y <- sin(runif(30, 0, 2*pi) * 4)
 indep.etest(x, y, R = 99)

 ## dependent univariate data
 u <- runif(30, 0, 2*pi)
 x <- sin(2 * u)
 y <- sin(3 * u)
 indep.etest(x, y, R = 99)
 
 u <- runif(50, 0, 2*pi)
 x <- sin(2 * u)
 y <- sin(4 * u)
 indep.etest(x, y, R = 99)
 
 ## independent multivariate data
 x <- matrix(rnorm(60), nrow=20, ncol=3)
 y <- matrix(rnorm(40), nrow=20, ncol=2)
 indep.e(x, y)
 indep.etest(x, y, R = 99)
 
 ## independent bivariate data
 x <- matrix(rnorm(50), nrow=25, ncol=2)
 y <- matrix(rnorm(50), nrow=25, ncol=2)
 indep.e(x, y)
 indep.etest(x, y, R = 99)
 
 ## dependent bivariate data
 library(MASS)
 Sigma <- matrix(c(1, .5, .5, 1), 2, 2)
 x <- mvrnorm(30, c(0, 0), Sigma)
 indep.etest(x[,1], x[,2], R = 99)
 
 ## dependent multivariate data
 Sigma <- matrix(c(1, .1, 0, 0 , 1, 0, 0 ,.1, 1), 3, 3)
 x <- mvrnorm(30, c(0, 0, 0), diag(3))
 y <- mvrnorm(30, c(0, 0, 0), Sigma) * x
 indep.etest(x, y, R = 99)
}

\keyword{ htest }
\keyword{ multivariate }
\concept{ energy statistics }

