% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/critMatlm.R
\name{critMatlm}
\alias{critMatlm}
\title{FARIMA Order Selection Matrix}
\usage{
critMatlm(X, p.max = 5, q.max = 5, criterion = c("bic", "aic"))
}
\arguments{
\item{X}{a numeric vector that contains the observed time series ordered
from past to present; the series is assumed to follow an FARIMA process.}

\item{p.max}{an integer value \eqn{>= 0} that defines the maximum
autoregressive order to calculate the criterion for; is set to \code{5}
by default; decimal numbers will be rounded off to integers.}

\item{q.max}{an integer value \eqn{>= 0} that defines the maximum
moving-average order to to calculate the criterion for; is set to \code{5}
by default; decimal numbers will be rounded off to integers.}

\item{criterion}{a character value that defines the information criterion
that will be calculated; the Bayesian Information Criterion (\code{"bic"})
and Akaike Information Criterion (\code{"aic"}) are the supported choices;
is set to \code{"bic"} by default.}
}
\value{
The function returns a \code{p.max + 1} by \code{q.max + 1} matrix, where the
rows represent the AR orders from \eqn{p = 0} to \eqn{p = p_{max}}{p = p_max}
and the columns represent the MA orders from \eqn{q = 0} to
\eqn{q = q_{max}}{q = q_max}. The values within the matrix are the values of
the previously selected information criterion for the different combinations
of \eqn{p} and \eqn{q}.
}
\description{
An information criterion is calculated for different orders of a fractionally
integrated autoregressive-moving-average (FARIMA) model.
}
\details{
The series passed to \code{X} is assumed to follow an
FARIMA(\eqn{p,d,q}) model. A \code{p.max + 1} by \code{q.max + 1} matrix is
calculated for this series. More precisely, the criterion chosen via the
argument \code{criterion} is calculated for all combinations of orders
\eqn{p = 0, 1, ..., p_{max}}{p = 0, 1, ..., p_max} and
\eqn{q = 0, 1, ..., q_{max}}{q = 0, 1, ..., q_max}.

Within the function, two information criteria are supported: the Bayesian
Information Criterion (BIC) and Akaike's Information Criterion (AIC). The AIC
is given by
\deqn{AIC_{p,q} := \ln(\hat{\sigma}_{p,q}^{2}) + \frac{2(p+q)}{n},}{AIC_[p,q]
:= ln(hat[sigma]^[2]_[p,q]) + \{2(p + q)\}/n,}
where \eqn{\hat{\sigma}_{p,q}^{2}}{hat[sigma]^[2]_[p,q]} is the estimated
innovation variance, \eqn{p} and \eqn{q} are the ARMA orders and \eqn{n} is
the number of observations.

The BIC, on the other hand, is defined by
\deqn{BIC_{p,q} := k \ln(n) - 2\ln(\hat{L})}{BIC_[p,q] := k * ln(n) -
2ln(hat[L])}
with \eqn{k} being the number of estimated parameters and
\eqn{\hat{L}}{hat[L]} being the estimated Log-Likelihood. Since the parameter
\eqn{k} only differs with respect to the orders \eqn{p} and \eqn{q} for all
estimated models, the term \eqn{k \ln(n)}{k * ln(n)} is reduced to
\eqn{(p + q) \ln(n)}{(p + q) * ln(n)} within the function. Exemplary,
if the mean of the series is estimated as well, it is usually considered
within the parameter \eqn{k} when calculating the BIC.
However, since the mean is estimated for all models, not considering this
estimated parameter within the calculation of the BIC will reduce all BIC
values by the same amount of \eqn{\ln(n)}{ln(n)}. Therefore, the selection
via this simplified criterion is still valid, if the number of the estimated
parameters only differs with respect to \eqn{p} and \eqn{q} between the
models that the BIC is obtained for.

The optimal orders are considered to be the ones which minimize either the
BIC or the AIC. The use of the BIC is however recommended, because the BIC
is consistent, whereas the AIC is not.

NOTE:

Within this function, the \code{\link[fracdiff]{fracdiff}} function of the
\code{fracdiff} package is used throughout for
the estimation of FARIMA models.
}
\examples{
# Simulate a FARIMA(2, 0.3 ,1) process
set.seed(1)
X.sim <- fracdiff::fracdiff.sim(n = 1000, ar = c(1.2, -0.71), ma = -0.46,
                               d = 0.3)$series
# Application of the function with BIC-criterion
BIC_mat <- critMatlm(X.sim)
BIC_mat
# determining the optimal order
smoots::optOrd(BIC_mat)
}
\author{
\itemize{
\item Dominik Schulz (Scientific Employee) (Department of Economics, Paderborn
University), \cr
Author
}
}
