% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EventStudy.R
\name{EventStudy}
\alias{EventStudy}
\title{Estimates Equation (2) in \href{https://www.nber.org/papers/w29170}{Freyaldenhoven et al. (2021)}}
\usage{
EventStudy(
  estimator,
  data,
  outcomevar,
  policyvar,
  idvar,
  timevar,
  controls = NULL,
  proxy = NULL,
  proxyIV = NULL,
  FE = TRUE,
  TFE = TRUE,
  post,
  overidpost = 1,
  pre,
  overidpre = post + pre,
  normalize = -1 * (pre + 1),
  cluster = TRUE,
  anticipation_effects_normalization = TRUE
)
}
\arguments{
\item{estimator}{Accepts one of "OLS" or "FHS". If "OLS" is specified, implements Ordinary Least Squares. If "FHS" is specified, implements Instrumental Variables (IV) estimator proposed in \href{https://www.aeaweb.org/articles?id=10.1257/aer.20180609}{Freyaldenhoven Hansen Shapiro (FHS, 2019)}.}

\item{data}{Data frame containing the variables of interest.}

\item{outcomevar}{Character indicating column of outcome variable y.}

\item{policyvar}{Character indicating column of policy variable z.}

\item{idvar}{Character indicating column of units.}

\item{timevar}{Character indicating column of time periods.}

\item{controls}{Optional character vector indicating a set of control variables q.}

\item{proxy}{Character indicating column of variable that is thought to be affected by the confound but not by the policy.
Should be specified if and only if estimator is specified as "FHS".}

\item{proxyIV}{Character of column to be used as an instrument. Should be specified if and only if estimator is specified as "FHS".
If NULL, defaults to the strongest lead of the policy variable based on the first stage.}

\item{FE}{Logical indicating whether unit fixed-effects should be included. Defaults to TRUE.}

\item{TFE}{Logical indicating whether time fixed-effects should be included. Defaults to TRUE.}

\item{post}{Whole number indicating the number of periods in the past before which the past values of the policy
are not supposed to affect the value of the outcome. Corresponds to M in equation (2) of
\href{https://www.nber.org/papers/w29170}{Freyaldenhoven et al. (2021)}.}

\item{overidpost}{Optional whole number indicating the number of event times after "post" to be included in estimation.
Defaults to 1.
Corresponds to L_M in equation (2) of \href{https://www.nber.org/papers/w29170}{Freyaldenhoven et al. (2021)}.}

\item{pre}{Whole number indicating the number of periods in the future after which the future values of the policy are
not supposed to affect the value of the outcome today. Corresponds to G in equation (2) of
\href{https://www.nber.org/papers/w29170}{Freyaldenhoven et al. (2021)}.}

\item{overidpre}{Optional whole number indicating the number of event times earlier than -"pre" to be included in estimation.
Defaults to "post" + "pre".
Corresponds to L_G in equation (2) of \href{https://www.nber.org/papers/w29170}{Freyaldenhoven et al. (2021)}.}

\item{normalize}{Specifies the event-time coefficient to be normalized. Defaults to - pre - 1.}

\item{cluster}{Logical indicating whether to use clustered errors by units. If FALSE, will use unclustered heteroskedasticity-robust standard errors.
Defaults to TRUE. Must be TRUE if FE is TRUE.}

\item{anticipation_effects_normalization}{If set to TRUE, runs the default process and switches coefficient to be normalized to 0
when there are anticipation effects. If set to FALSE, does not make the switch. Defaults to TRUE.}
}
\value{
A list that contains, under "output", the estimation output as an lm_robust object, and under "arguments", the arguments passed to the function.
}
\description{
\code{EventStudy} uses regression methods to estimate the effect of a policy on a given outcome.
}
\examples{

# A minimal example
eventstudy_model <-
  EventStudy(
    estimator = "OLS",
    data = df_sample_dynamic,
    outcomevar = "y_base",
    policyvar = "z",
    idvar = "id",
    timevar = "t",
    pre = 0, post = 3,
    normalize = -1
  )

### Access estimated model
eventstudy_model$output

summary(eventstudy_model$output)

### data.frame of estimates
estimatr::tidy(eventstudy_model$output)

### Access arguments
eventstudy_model$arguments

# A dynamic OLS model with anticipation effects and controls
eventstudy_model_dyn <-
  EventStudy(
    estimator = "OLS",
    data = df_sample_dynamic,
    outcomevar = "y_base",
    policyvar = "z",
    idvar = "id",
    timevar = "t",
    controls = "x_r",
    FE = TRUE, TFE = TRUE,
    post = 3, overidpost = 5,
    pre  = 2, overidpre  = 4,
    normalize = - 3,
    cluster = TRUE,
    anticipation_effects_normalization = TRUE
  )

summary(eventstudy_model_dyn$output)

# A static model
eventstudy_model_static <-
  EventStudy(
    estimator = "OLS",
    data = df_sample_static,
    outcomevar = "y_static",
    policyvar = "z",
    idvar = "id",
    timevar = "t",
    FE = TRUE, TFE = TRUE,
    post = 0, overidpost = 0,
    pre  = 0, overidpre  = 0,
    cluster = TRUE
  )

summary(eventstudy_model_static$output)

# A dynamic model estimated using IV
eventstudy_model_iv <-
  EventStudy(
    estimator = "FHS",
    data = df_sample_dynamic,
    outcomevar = "y_base",
    policyvar = "z",
    idvar = "id",
    timevar = "t",
    controls = "x_r",
    proxy = "eta_m",
    FE = TRUE, TFE = TRUE,
    post = 2, overidpost = 1,
    pre  = 0, overidpre  = 3,
    normalize = -1,
    cluster = TRUE
  )

summary(eventstudy_model_iv$output)

}
