% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fastCrrp.R
\name{fastCrrp}
\alias{fastCrrp}
\title{Penalized Fine-Gray Model Estimation via two-way linear scan}
\usage{
fastCrrp(formula, data, eps = 1e-06, max.iter = 1000,
  getBreslowJumps = TRUE, standardize = TRUE, penalty = c("LASSO",
  "RIDGE", "MCP", "SCAD", "ENET"), lambda = NULL, alpha = 0,
  lambda.min.ratio = 0.001, nlambda = 25, penalty.factor = rep(1,
  ncol(X)), gamma = switch(penalty, scad = 3.7, 2.7))
}
\arguments{
\item{formula}{a formula object, with the response on the left of a ~ operator, and the terms on the right. The response must be a Crisk object as returned by the \code{Crisk} function.}

\item{data}{a data.frame in which to interpret the variables named in the formula.}

\item{eps}{Numeric: algorithm stops when the relative change in any coefficient is less than \code{eps} (default is \code{1E-6})}

\item{max.iter}{Numeric: maximum iterations to achieve convergence (default is 1000)}

\item{getBreslowJumps}{Logical: Output jumps in Breslow estimator for the cumulative hazard.}

\item{standardize}{Logical: Standardize design matrix.}

\item{penalty}{Character: Penalty to be applied to the model. Options are "lasso", "scad", "ridge", "mcp", and "enet".}

\item{lambda}{A user-specified sequence of \code{lambda} values for tuning parameters.}

\item{alpha}{L1/L2 weight for elastic net regression.}

\item{lambda.min.ratio}{Smallest value for \code{lambda}, as a fraction of \code{lambda.max} (if \code{lambda} is NULL).}

\item{nlambda}{Number of \code{lambda} values (default is 25).}

\item{penalty.factor}{A vector of weights applied to the penalty for each coefficient. Vector must be of length equal to the number of columns in \code{X}.}

\item{gamma}{Tuning parameter for the MCP/SCAD penalty. Default is 2.7 for MCP and 3.7 for SCAD and should be left unchanged.}
}
\value{
Returns a list of class \code{fcrrp}.
\item{coef}{fitted coefficients matrix with \code{nlambda} columns and \code{nvars} columns}
\item{logLik}{vector of log-pseudo likelihood at the estimated regression coefficients}
\item{logLik.null}{log-pseudo likelihood when the regression coefficients are 0}
\item{lambda.path}{sequence of tuning parameter values}
\item{iter}{number of iterations needed until convergence at each tuning parameter value}
\item{converged}{convergence status at each tuning parameter value}
\item{breslowJump}{Jumps in the Breslow baseline cumulative hazard (used by \code{predict.fcrr})}
\item{uftime}{vector of unique failure (event) times}
\item{penalty}{same as above}
\item{gamma}{same as above}
\item{above}{same as above}
}
\description{
Performs penalized regression for the proportional subdistribution hazards model.
Penalties currently include LASSO, MCP, SCAD, and ridge regression. User-specificed weights can be assigned
to the penalty for each coefficient (e.g. implementing adaptive LASSO and broken adaptive ridge regerssion).
}
\details{
The \code{fastCrrp} functions performed penalized Fine-Gray regression.
Parameter estimation is performed via cyclic coordinate descent and using a two-way linear scan approach to effiiciently
calculate the gradient and Hessian values. Current implementation includes LASSO, SCAD, MCP, and ridge regression.
}
\examples{

library(fastcmprsk)
set.seed(10)
ftime <- rexp(200)
fstatus <- sample(0:2, 200, replace = TRUE)
cov <- matrix(runif(1000), nrow = 200)
dimnames(cov)[[2]] <- c('x1','x2','x3','x4','x5')
fit <- fastCrrp(Crisk(ftime, fstatus) ~ cov, lambda = 1, penalty = "RIDGE")
fit$coef

}
\references{
Fu, Z., Parikh, C.R., Zhou, B. (2017) Penalized variable selection in competing risks
regression. \emph{Lifetime Data Analysis} 23:353-376.

Breheny, P. and Huang, J. (2011) Coordinate descent algorithms for nonconvex penalized regression, with applications to biological feature selection. \emph{Ann. Appl. Statist.}, 5: 232-253.

Fine J. and Gray R. (1999) A proportional hazards model for the subdistribution of a competing risk.  \emph{JASA} 94:496-509.
}
