% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/concats.r
\name{concats,GRaster-method}
\alias{concats,GRaster-method}
\alias{concats}
\title{Combine values/categories of multiple GRasters into a single GRaster}
\usage{
\S4method{concats}{GRaster}(x, ..., na.rm = TRUE)
}
\arguments{
\item{x}{A \code{GRaster} with one or more layers, each of which must be have cells that represent integers or categories (factors).}

\item{...}{Either missing or integer/categorical (factor) \code{GRaster}s.}

\item{na.rm}{Logical: If \code{TRUE} (default), then any combinations that include an \code{NA} cell will result in an \code{NA} cell in the output.}
}
\value{
A \code{GRaster}. If the inputs are all categorical (factor) rasters, then a levels table will also be returned with the new combined levels.
}
\description{
This function takes from 2 to 10 integer or categorical (factor) \code{GRaster}s and creates a single \code{GRaster} that has one value per combination of values in the inputs. For example, say that there were two input rasters, with values 1 and 2 in the one raster, and 3 and 4 in the other. If the following combinations of values occurred between the two rasters, then the output raster would be re-coded with the new values:\tabular{lll}{
   \code{input_raster1} \tab \code{input_raster2} \tab \code{output_raster} \cr
   1 \tab 3 \tab 0 \cr
   1 \tab 4 \tab 1 \cr
   2 \tab 3 \tab 2 \cr
   2 \tab 4 \tab 3 \cr
}


If the argument \code{na.rm} is set to \code{TRUE} (which it is, by default), then whenever at least one cell has an \code{NA} value, then the output will also have an \code{NA} (i.e., a new category number is not created). However, if \code{na.rm} is \code{FALSE}, then combinations that include an \code{NA} are assigned a new category number, unless all values are \code{NA} (in which case the output will be \code{NA}).

The difference between this function and \code{\link[=combineLevels]{combineLevels()}} is that this one creates a "combined" \code{GRaster} with a combined levels table, whereas \code{combineLevels()} just merges the levels tables.

If the inputs are all categorical rasters, then a \code{\link[=levels]{levels()}} table will also be returned with the new levels.
}
\examples{
if (grassStarted()) {

# Setup
library(terra)

# Example data: Land cover raster
madCover <- fastData("madCover")

# Convert categorical SpatRaster to categorical GRaster:
cover <- fast(madCover)

### Properties of categorical rasters

cover # note categories
is.factor(cover) # Is the raster categorical?
nlevels(cover) # number of levels
levels(cover) # just the value and active column
cats(cover) # all columns
minmax(cover) # min/max values
minmax(cover, levels = TRUE) # min/max categories
catNames(cover) # column names of the levels table
missingCats(cover) # categories in table with no values in raster
freq(cover) # frequency of each category (number of cells)
zonalGeog(cover) # geometric statistics

### Active column

# Which column sets the category labels?
activeCat(cover)
activeCat(cover, names = TRUE)

activeCats(c(cover, cover))

# Choose a different column for category labels:
levels(cover)
activeCat(cover) <- 2
levels(cover)

### Managing levels tables

# Remove unused levels:
nlevels(cover)
cover <- droplevels(cover)
nlevels(cover)

# Re-assign levels:
value <- c(20, 30, 40, 50, 120, 130, 140, 170)
label <- c("Cropland", "Cropland", "Forest", "Forest",
 "Grassland", "Shrubland", "Herbaceous", "Flooded")

newCats <- data.frame(value = value, label = label)

cover <- categories(cover, layer = 1, value = newCats)
cats(cover)

# This is the same as:
levels(cover) <- newCats
cats(cover)

# Are there any values not assigned a category?
missingCats(cover)

# Let's assign a category for value 210 (water):
water <- data.frame(value = 210, label = "Water")
addCats(cover) <- water
levels(cover)

# Add more information to the levels table using merge():
landType <- data.frame(
     Value = c(20, 30, 40, 50, 120),
     Type = c("Irrigated", "Rainfed", "Broadleaf evergreen",
     "Broadleaf deciduous", "Mosaic with forest")
)
cats(cover)
cover <- addCats(cover, landType, merge = TRUE)
cats(cover)

### Logical operations on categorical rasters

cover < "Forest" # 1 for cells with a value < 40, 0 otherwise
cover <= "Forest" # 1 for cells with a value < 120, 0 otherwise
cover == "Forest" # 1 for cells with value of 40-120, 0 otherwise
cover != "Forest" # 1 for cells with value that is not 40-120, 0 otherwise
cover > "Forest" # 1 for cells with a value > 120, 0 otherwise
cover >= "Forest" # 1 for cells with a value >= 120, 0 otherwise

cover \%in\% c("Cropland", "Forest") # 1 for cropland/forest cells, 0 otherwise

### Combine categories from different rasters

# For the example, will create a second categorical raster fromm elevation.

# Divide elevation raster into "low/medium/high" levels:
madElev <- fastData("madElev")
elev <- fast(madElev)
elev <- project(elev, cover, method = "near") # convert to same CRS
fun <- "= if(madElev < 100, 0, if(madElev < 400, 1, 2))"
elevCat <- app(elev, fun)

levs <- data.frame(
     value = c(0, 1, 2),
     elevation = c("low", "medium", "high")
)
levels(elevCat) <- list(levs)

# Combine levels:
combined <- concats(cover, elevCat)
combined
levels(combined)

# Combine levels, treating value/NA combinations as new categories:
combinedNA <- concats(cover, elevCat, na.rm = FALSE)
combinedNA
levels(combinedNA)

}
}
\seealso{
\code{\link[=combineLevels]{combineLevels()}}, \code{\link[terra:concats]{terra::concats()}}, \code{vignette("GRasters", package = "fasterRaster")}, \strong{GRASS} manual page for module \code{r.cross} (see \code{grassHelp("r.cross")})
}
