% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/abundance.R
\name{abundance}
\alias{abundance}
\alias{basal_area}
\title{Abundance and basal area, optionally by groups.}
\usage{
abundance(data)

basal_area(data)
}
\arguments{
\item{data}{A dataframe. \code{\link[=basal_area]{basal_area()}} requires a column named \code{dbh} (case
insensitive).}
}
\description{
\itemize{
\item \code{\link[=abundance]{abundance()}} counts the number of rows in a dataset, optionally by groups
created with \code{\link[dplyr:group_by]{dplyr::group_by()}} (similar to \code{\link[dplyr:context]{dplyr::n()}}). It warns if it
detects duplicated values of treeid.
\item \code{\link[=basal_area]{basal_area()}} sums the basal area of
all stems in a dataset, optionally by groups created with \code{\link[=group_by]{group_by()}}. It
warns if it detects duplicated values of stemid. It does not convert units
(but see examples).
}

Both \code{\link[=abundance]{abundance()}} and \code{\link[=basal_area]{basal_area()}} warn if they detect
multiple censusid and multiple plots.
}
\details{
You may want to calculate the abundance or basal area for a specific subset
of data (e.g. "alive" stems or stems which \code{dbh} is within some range).
Subsetting data is not the job of these functions. Instead see
\code{\link[base:subset]{base::subset()}}, \code{\link[dplyr:filter]{dplyr::filter()}}, or \code{[}.
}
\examples{
library(fgeo.tool)

# abundance() -------------------------------------------------------------

abundance(data.frame(1))

# One stem per tree
tree <- tribble(
  ~TreeID, ~StemID, ~DBH,
  "1", "1.1", 11,
  "2", "2.1", 21
)

abundance(tree)

# One tree with multiple stems
stem <- tribble(
  ~TreeID, ~StemID, ~DBH,
  "1", "1.1", 11,
  "1", "1.2", 12
)

abundance(stem)

# Skip R CMD check for speed
\donttest{
# Similar but more realistic
assert_is_installed("fgeo.x")
stem <- fgeo.x::download_data("luquillo_stem5_random")

abundance(stem)

abundance(pick_main_stem(stem))
}

vft <- tribble(
  ~PlotName, ~CensusID, ~TreeID, ~StemID, ~DBH,
  "p", 1, "1", "1.1", 10,
  "q", 2, "1", "1.1", 10
)

# * Warns if it detects multiple values of censusid or plotname
# * Also warns if it detects duplicated values of treeid
abundance(vft)

# If trees have buttressess, the data may have multiple stems per treeid or
# multiple measures per stemid.
vft2 <- tribble(
  ~CensusID, ~TreeID, ~StemID, ~DBH, ~HOM,
  1, "1", "1.1", 88, 130,
  1, "1", "1.1", 10, 160,
  1, "2", "2.1", 20, 130,
  1, "2", "2.2", 30, 130,
)

# You should count only the main stem of each tree
(main_stem <- pick_main_stem(vft2))

abundance(main_stem)

vft3 <- tribble(
  ~CensusID, ~TreeID, ~StemID, ~DBH, ~HOM,
  1, "1", "1.1", 20, 130,
  1, "1", "1.2", 10, 160, # Main stem
  2, "1", "1.1", 12, 130,
  2, "1", "1.2", 22, 130 # Main stem
)

# You can compute by groups
by_census <- group_by(vft3, CensusID)
(main_stems_by_census <- pick_main_stem(by_census))

abundance(main_stems_by_census)

# basal_area() ------------------------------------------------------------

# Data must have a column named dbh (case insensitive)
basal_area(data.frame(dbh = 1))

# * Warns if it detects multiple values of censusid or plotname
# * Also warns if it detects duplicated values of stemid
basal_area(vft)

# First you may pick the main stemid of each stem
(main_stemids <- pick_main_stemid(vft2))

basal_area(main_stemids)

# You can compute by groups
basal_area(by_census)

# Skip R CMD check for speed
\donttest{
measurements_is_installed <- requireNamespace("measurements", quietly = TRUE)
if (measurements_is_installed) {
  library(measurements)

  # Convert units
  ba <- basal_area(by_census)
  ba$basal_area_he <- conv_unit(
    ba$basal_area,
    from = "mm2",
    to = "hectare"
  )

  ba
}
}

}
\seealso{
\code{\link[dplyr:context]{dplyr::n()}}, \code{\link[dplyr:group_by]{dplyr::group_by()}}.

Other functions for abundance and basal area: 
\code{\link{abundance_byyr}()}
}
\concept{functions for abundance and basal area}
