% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fnets.R
\name{fnets}
\alias{fnets}
\title{Factor-adjusted network estimation}
\usage{
fnets(
  x,
  center = TRUE,
  fm.restricted = FALSE,
  q = c("ic", "er"),
  ic.op = NULL,
  kern.bw = NULL,
  common.args = list(factor.var.order = NULL, max.var.order = NULL, trunc.lags = 20,
    n.perm = 10),
  var.order = 1,
  var.method = c("lasso", "ds"),
  var.args = list(n.iter = NULL, n.cores = min(parallel::detectCores() - 1, 3)),
  do.threshold = FALSE,
  do.lrpc = TRUE,
  lrpc.adaptive = FALSE,
  tuning.args = list(tuning = c("cv", "bic"), n.folds = 1, penalty = NULL, path.length
    = 10, do.plot = FALSE)
)
}
\arguments{
\item{x}{input time series matrix, with each row representing a variable}

\item{center}{whether to de-mean the input \code{x} row-wise}

\item{fm.restricted}{whether to estimate a restricted factor model using static PCA}

\item{q}{Either the number of factors or a string specifying the factor number selection method; possible values are:
\itemize{
   \item{\code{"ic"}}{ information criteria-based methods of Alessi, Barigozzi & Capasso (2010) when \code{fm.restricted = TRUE} or Hallin and Liška (2007) when \code{fm.restricted = FALSE} modifying Bai and Ng (2002)}
   \item{\code{"er"}}{ eigenvalue ratio of Ahn and Horenstein (2013)}
}
see \link[fnets]{factor.number}.}

\item{ic.op}{choice of the information criterion penalty, see \link[fnets]{factor.number} for further details}

\item{kern.bw}{a positive integer specifying the kernel bandwidth for dynamic PCA; by default, it is set to \code{floor(4 *(dim(x)[2]/log(dim(x)[2]))^(1/3)))}.  When \code{fm.restricted = TRUE}, it is used to compute the number of lags for which autocovariance matrices are estimated}

\item{common.args}{a list specifying the tuning parameters required for estimating the impulse response functions and common shocks. It contains:
\itemize{
   \item{\code{factor.var.order}}{ order of the blockwise VAR representation of the common component. If \code{factor.var.order = NULL}, it is selected blockwise by Schwarz criterion}
   \item{\code{max.var.order}}{ maximum blockwise VAR order for the Schwarz criterion}
   \item{\code{trunc.lags}}{ truncation lag for impulse response function estimation}
   \item{\code{n.perm}}{ number of cross-sectional permutations involved in impulse response function estimation}
}}

\item{var.order}{order of the idiosyncratic VAR process; if a vector of integers is supplied, the order is chosen via \code{tuning}}

\item{var.method}{a string specifying the method to be adopted for idiosyncratic VAR process estimation; possible values are:
\itemize{
       \item{\code{"lasso"}}{ Lasso-type \code{l1}-regularised \code{M}-estimation}
       \item{\code{"ds"}}{ Dantzig Selector-type constrained \code{l1}-minimisation}
}}

\item{var.args}{a list specifying the tuning parameters required for estimating the idiosyncratic VAR process. It contains:
\itemize{
   \item{\code{n.iter}}{ maximum number of descent steps, by default depends on \code{var.order}; applicable when \code{var.method = "lasso"}}
   \item{\code{n.cores}}{ number of cores to use for parallel computing, see \link[parallel]{makePSOCKcluster}; applicable when \code{var.method = "ds"}}
}}

\item{do.threshold}{whether to perform adaptive thresholding of all parameter estimators with \link[fnets]{threshold}}

\item{do.lrpc}{whether to estimate the long-run partial correlation}

\item{lrpc.adaptive}{whether to use the adaptive estimation procedure}

\item{tuning.args}{a list specifying arguments for \code{tuning}
for selecting the tuning parameters involved in VAR parameter and (long-run) partial correlation matrix estimation. It contains:
\itemize{
   \item{\code{tuning}} a string specifying the selection procedure for \code{var.order} and \code{lambda}; possible values are:
   \itemize{
      \item{\code{"cv"}}{ cross validation}
      \item{\code{"bic"}}{ information criterion}
   }
   \item{\code{n.folds}}{ if \code{tuning = "cv"}, positive integer number of folds}
   \item{\code{penalty}}{ if \code{tuning = "bic"}, penalty multiplier between 0 and 1; if \code{penalty = NULL}, it is set to \code{1/(1+exp(dim(x)[1])/dim(x)[2]))}} by default
   \item{\code{path.length}}{ positive integer number of regularisation parameter values to consider; a sequence is generated automatically based in this value}
   \item{\code{do.plot}}{ whether to plot the output of the cross validation step}
}}
}
\value{
an S3 object of class \code{fnets}, which contains the following fields:
\item{q}{ number of factors}
\item{spec}{ if \code{fm.restricted = FALSE} a list containing estimates of the spectral density matrices for \code{x}, common and idiosyncratic components}
\item{acv}{ a list containing estimates of the autocovariance matrices for \code{x}, common and idiosyncratic components}
\item{loadings}{ if \code{fm.restricted = TRUE}, factor loadings; if \code{fm.restricted = FALSE} and \code{q >= 1},
a list containing estimators of the impulse response functions (as an array of dimension \code{(p, q, trunc.lags + 2)})}
\item{factors}{ if \code{fm.restricted = TRUE}, factor series; else, common shocks (an array of dimension \code{(q, n)})}
\item{idio.var}{ a list containing the following fields:
\itemize{
\item{\code{beta}}{ estimate of VAR parameter matrix; each column contains parameter estimates for the regression model for a given variable}
\item{\code{Gamma}}{ estimate of the innovation covariance matrix}
\item{\code{lambda}}{ regularisation parameter}
\item{\code{var.order}}{ VAR order}
}}
\item{lrpc}{ see the output of \link[fnets]{par.lrpc}}
\item{mean.x}{ if \code{center = TRUE}, returns a vector containing row-wise sample means of \code{x}; if \code{center = FALSE}, returns a vector of zeros}
\item{var.method}{ input parameter}
\item{do.lrpc}{ input parameter}
\item{kern.bw}{ input parameter}
}
\description{
Operating under a factor-adjusted vector autoregressive (VAR) model,
the function estimates the spectral density and autocovariance matrices of the factor-driven common component and the idiosyncratic VAR process,
the impulse response functions and common shocks for the common component,
and VAR parameters, innovation covariance matrix and long-run partial correlations for the idiosyncratic component.
}
\details{
See Barigozzi, Cho and Owens (2022) and Owens, Cho and Barigozzi (2022) for further details.
List arguments do not need to be specified with all list components; any missing entries will be filled in with the default argument.
}
\examples{
\donttest{
set.seed(123)
n <- 500
p <- 50
common <- sim.unrestricted(n, p)
idio <- sim.var(n, p)
x <- common$data + idio$data
out <- fnets(x,
  q = NULL, var.order = 1, var.method = "lasso", do.threshold = TRUE,
  do.lrpc = TRUE, tuning.args = list(tuning = "cv", n.folds = 1, path.length = 10, do.plot = TRUE),
  var.args = list(n.cores = 2)
)
pre <- predict(out, x, h = 1, common.method = "unrestricted")
plot(out, type = "granger", display = "network")
plot(out, type = "lrpc", display = "heatmap")
}
}
\references{
Ahn, S. C. & Horenstein, A. R. (2013) Eigenvalue ratio test for the number of factors. Econometrica, 81(3), 1203--1227.

Alessi, L., Barigozzi, M.,  & Capasso, M. (2010) Improved penalization for determining the number of factors in approximate factor models. Statistics & Probability Letters, 80(23-24):1806–1813.

Bai, J. & Ng, S. (2002) Determining the number of factors in approximate factor models. Econometrica. 70: 191-221.

Barigozzi, M., Cho, H. & Owens, D. (2022) FNETS: Factor-adjusted network estimation and forecasting for high-dimensional time series. arXiv preprint arXiv:2201.06110.

Hallin, M. & Liška, R. (2007) Determining the number of factors in the general dynamic factor model. Journal of the American Statistical Association, 102(478), 603--617.

Owens, D., Cho, H. & Barigozzi, M. (2022) fnets: An R Package for Network Estimation and Forecasting via Factor-Adjusted VAR Modelling. arXiv preprint arXiv:2301.11675.
}
\seealso{
\link[fnets]{predict.fnets}, \link[fnets]{plot.fnets}
}
