% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/galamm.R
\name{galamm}
\alias{galamm}
\title{Fit a generalized additive latent and mixed model}
\usage{
galamm(
  formula,
  weights = NULL,
  data,
  family = gaussian,
  family_mapping = rep(1, nrow(data)),
  load.var = NULL,
  lambda = NULL,
  factor = NULL,
  factor_interactions = NULL,
  na.action = getOption("na.action"),
  start = NULL,
  control = galamm_control()
)
}
\arguments{
\item{formula}{A formula specifying the model. Smooth terms are defined in
the style of the \code{mgcv} and \code{gamm4} packages, see
\insertCite{woodGeneralizedAdditiveModels2017}{galamm} for an
introduction. Random effects are specified using \code{lme4} syntax, which
is described in detail in
\insertCite{batesFittingLinearMixedEffects2015}{galamm}. Factor loadings
will also be part of the model formula, and is based on the syntax of the
\code{PLmixed} package
\insertCite{rockwoodEstimatingComplexMeasurement2019}{galamm}.}

\item{weights}{An optional formula object specifying an expression for the
residual variance. Defaults to \code{NULL}, corresponding to homoscedastic
errors. The formula is defined in \code{lme4} style; see vignettes and
examples for details.}

\item{data}{A data.frame containing all the variables specified by the model
formula, with the exception of factor loadings.}

\item{family}{A a list or character vector containing one or more model
families. For each element in \code{family} there should be a corresponding
element in \code{family_mapping} specifying which elements of the response
are conditionally distributed according to the given family. Currently
family can be one of \code{gaussian}, \code{binomial}, and \code{poisson},
and only canonical link functions are supported. The family arguments can
either be provided as character values, e.g., \code{c("gaussian",
  "poisson")} or \code{list("gaussian", "poisson")}, as function names, e.g.,
\code{c(gaussian, poisson)} or \code{list(gaussian, poisson)}, or as
function calls, e.g., \code{list(gaussian(), poisson())}. In the latter
case, they must be provided in a list, and bot as a vector. Mixing the
different ways of describing the family also works, e.g.,
\code{list("gaussian", poisson())}, but in this case they must be provided
in a list. When provided as character values, the argument is case
sensitive.}

\item{family_mapping}{Optional vector mapping from the elements of
\code{family} to rows of \code{data}. Defaults to \code{rep(1,
  nrow(data))}, which means that all observations are distributed according
to the first element of \code{family}. The length of \code{family_mapping}
must be identical to the number of observations, \code{nrow(data)}.}

\item{load.var}{Optional character specifying the name of the variable in
\code{data} identifying what the factors load onto. Default to \code{NULL},
which means that there are no loading variables. Argument is case
sensitive.}

\item{lambda}{Optional factor loading matrix. Numerical values indicate that
the given value is fixed, while \code{NA} means that the entry is a
parameter to be estimated. Numerical values can only take the values 0 or
\enumerate{
\item The number of columns of \code{lambda} must be identical to the number
of elements in \code{factor}. Defaults to \code{NULL}, which means that
there is no factor loading matrix. If \code{lambda} is provided as a
vector, it will be converted to a \code{matrix} with a single column.
}}

\item{factor}{Optional character vector whose \eqn{j}th entry corresponds to
the \eqn{j}th column of the corresponding matrix in \code{lambda}. The
number of elements in \code{factor} must be equal to the number of columns
in \code{lambda}. Defaults to \code{NULL}, which means that there are no
factor loadings. Argument is case sensitive.}

\item{factor_interactions}{Optional list of length equal to the number of
columns in \code{lambda}. Each list element should be a \code{formula}
object containing the write-hand side of a regression model, of the form
\code{~ x + z}. Defaults to \code{NULL}, which means that no factor
interactions are used.}

\item{na.action}{Character of length one specifying a function which
indicates what should happen when the data contains \code{NA}s. The
defaults is set to the \code{na.action} setting of \code{options}, which
can be seen with \code{options("na.action")}. The other alternatives are
\code{"na.fail"} or \code{"na.exclude"}, which means that the function
fails if there as \code{NA}s in \code{data}.}

\item{start}{Optional named list of starting values for parameters. Possible
names of list elements are \code{"theta"}, \code{"beta"}, \code{"lambda"},
and \code{"weights"}, all of should be numerical vectors with starting
values. Default to \code{NULL}, which means that some relatively sensible
defaults are used. Names of parameters must be given in all lower case.}

\item{control}{Optional control object for the optimization procedure of
class \code{galamm_control} resulting from calling
\code{\link{galamm_control}}. Defaults to \code{NULL}, which means that the
defaults of \code{\link{galamm_control}} are used.}
}
\value{
A model object of class \code{galamm}, containing the following
elements:
\itemize{
\item \code{call} the matched call used when fitting the model.
\item \code{random_effects} a list containing the following two elements:
\itemize{
\item \code{b} random effects in original parametrization.
\item \code{u} random effects standardized to have identity covariance matrix.
}
\item \code{model} a list with various elements related to the model setup and
fit:
\itemize{
\item \code{deviance} deviance of final model.
\item \code{deviance_residuals} deviance residuals of the final model.
\item \code{df} degrees of freedom of model.
\item \code{family} a list of one or more family objects, as specified in the
\code{family} arguments to \code{galamm}.
\item \code{factor_interactions} List of formulas specifying interactions
between latent and observed variables, as provided to the argument
\code{factor_interactions} to \code{galamm}. If not provided, it is
\code{NULL}.
\item \code{fit} a numeric vector with fitted values.
\item \code{fit_population} a numeric vector with fitted values excluding
random effects.
\item \code{hessian} Hessian matrix of final model, i.e., the second
derivative of the log-likelihood with respect to all model parameters.
\item \code{lmod} Linear model object returned by \code{lme4::lFormula}, which
is used internally for setting up the models.
\item \code{loglik} Log-likelihood of final model.
\item \code{n} Number of observations.
\item \code{pearson_residual} Pearson residuals of final model.
\item \code{reduced_hessian} Logical specifying whether the full Hessian
matrix was computed, or a Hessian matrix with derivatives only with respect
to beta and lambda.
\item \code{response} A numeric vector containing the response values used when
fitting the model.
\item \code{weights_object} Object with weights used in model fitting. Is
\code{NULL} when no weights were used.
}
\item \code{parameters} A list object with model parameters and related
information:
\itemize{
\item \code{beta_inds} Integer vector specifying the indices of fixed
regression coefficients among the estimated model parameters.
\item \code{dispersion_parameter} One or more dispersion parameters of the
final model.
\item \code{lambda_dummy} Dummy matrix of factor loadings, which shows the
structure of the loading matrix that was supplied in the \code{lambda}
arguments.
\item \code{lambda_inds} Integer vector specifying the indices of factor
loadings among the estimated model parameters.
\item \code{lambda_interaction_inds} Integer vector specifying the indices
of regression coefficients for interactions between latent and observed
variables.
\item \code{parameter_estimates} Numeric vector of final parameter estimates.
\item \code{parameter_names} Names of all parameters estimates.
\item \code{theta_inds} Integer vector specifying the indices of variance
components among the estimated model parameters. Technically these are the
entries of the Cholesky decomposition of the covariance matrix.
\item \code{weights_inds} Integer vector specifying the indices of estimated
weights (used in heteroscedastic Gaussian models) among the estimated model
parameters.
}
\item \code{gam} List containing information about smooth terms in the model.
If no smooth terms are contained in the model, then it is a list of length
zero.
}
}
\description{
This function fits a generalized additive latent and mixed model
(GALAMMs), as described in
\insertCite{sorensenLongitudinalModelingAgeDependent2023;textual}{galamm}.
The building blocks of these models are generalized additive mixed models
(GAMMs) \insertCite{woodGeneralizedAdditiveModels2017}{galamm}, of which
generalized linear mixed models
\insertCite{breslowApproximateInferenceGeneralized1993,harvilleMaximumLikelihoodApproaches1977,hendersonBestLinearUnbiased1975,lairdRandomEffectsModelsLongitudinal1982}{galamm}
are special cases. GALAMMs extend upon GAMMs by allowing factor structures,
as commonly used to model hypothesized latent traits underlying observed
measurements. In this sense, GALAMMs are an extension of generalized linear
latent and mixed models (GLLAMMs)
\insertCite{skrondalGeneralizedLatentVariable2004,rabe-heskethGeneralizedMultilevelStructural2004}{galamm}
which allows semiparametric estimation. The implemented algorithm used to
compute model estimates is described in
\insertCite{sorensenLongitudinalModelingAgeDependent2023;textual}{galamm},
and is an extension of the algorithm used for fitting generalized linear
mixed models by the \code{lme4} package
\insertCite{batesFittingLinearMixedEffects2015}{galamm}. The syntax used to
define factor structures is based on that used by the \code{PLmixed}
package, which is detailed in
\insertCite{rockwoodEstimatingComplexMeasurement2019;textual}{galamm}.
}
\examples{
# Mixed response model ------------------------------------------------------

# The mresp dataset contains a mix of binomial and Gaussian responses.

# We need to estimate a factor loading which scales the two response types.
loading_matrix <- matrix(c(1, NA), ncol = 1)

# Define mapping to families.
families <- c(gaussian, binomial)
family_mapping <- ifelse(mresp$itemgroup == "a", 1, 2)


# Fit the model
mod <- galamm(
  formula = y ~ x + (0 + level | id),
  data = mresp,
  family = families,
  family_mapping = family_mapping,
  factor = "level",
  load.var = "itemgroup",
  lambda = loading_matrix
)

# Summary information
summary(mod)


# Heteroscedastic model -----------------------------------------------------
# Residuals allowed to differ according to the item variable
# We also set the initial value of the random intercept standard deviation
# to 1
mod <- galamm(
  formula = y ~ x + (1 | id), weights = ~ (1 | item),
  data = hsced, start = list(theta = 1)
)
summary(mod)

# Generalized additive mixed model with factor structures -------------------

# The cognition dataset contains simulated measurements of three latent
# time-dependent processes, corresponding to individuals' abilities in
# cognitive domains. We focus here on the first domain, and take a single
# random timepoint per person:
dat <- subset(cognition, domain == 1)
dat <- split(dat, f = dat$id)
dat <- lapply(dat, function(x) x[x$timepoint \%in\% sample(x$timepoint, 1), ])
dat <- do.call(rbind, dat)
dat$item <- factor(dat$item)

# At each timepoint there are three items measuring ability in the cognitive
# domain. We fix the factor loading for the first measurement to one, and
# estimate the remaining two. This is specified in the loading matrix.
loading_matrix <- matrix(c(1, NA, NA), ncol = 1)

# We can now estimate the model.
mod <- galamm(
  formula = y ~ 0 + item + sl(x, factor = "loading") +
    (0 + loading | id),
  data = dat,
  load.var = "item",
  lambda = loading_matrix,
  factor = "loading"
)

# We can plot the estimated smooth term
plot_smooth(mod, shade = TRUE)


# Interaction between observed and latent covariates ------------------------
# Define the loading matrix
lambda <- matrix(c(1, NA, NA), ncol = 1)

# Define the regression functions, one for each row in the loading matrix
factor_interactions <- list(~1, ~1, ~x)

# Fit the model
mod <- galamm(
  formula = y ~ type + x:response + (0 + loading | id),
  data = latent_covariates,
  load.var = "type",
  lambda = lambda,
  factor = "loading",
  factor_interactions = factor_interactions
)

# The summary output now include an interaction between the latent variable
# and x, for predicting the third element in "type"
summary(mod)

}
\references{
\insertAllCited{}
}
\seealso{
Other modeling functions: 
\code{\link{s}()},
\code{\link{t2}()}
}
\concept{modeling functions}
